/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractEditor.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.goe;


import java.awt.Component;
import java.awt.Container;
import java.awt.Dialog;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.beans.PropertyEditorSupport;

import javax.swing.JComponent;
import javax.swing.JPopupMenu;

import adams.gui.core.GUIHelper;
import adams.gui.core.MouseUtils;

/**
 * A superclass for editors with custom editors.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractPropertyEditorSupport
  extends PropertyEditorSupport {

  /** the custom editor. */
  protected JComponent m_CustomEditor;

  /** the window adapter for de-registering the DB change listener. */
  protected WindowAdapter m_WindowAdapter;

  /**
   * Initializes the editor.
   */
  protected AbstractPropertyEditorSupport() {
    super();
    initialize();
  }

  /**
   * For initializing members.
   * <p/>
   * Default implementation does nothing.
   */
  protected void initialize() {
  }

  /**
   * Returns true since this editor is paintable.
   *
   * @return 		always true.
   */
  public boolean isPaintable() {
    return true;
  }

  /**
   * Returns true because we do support a custom editor.
   *
   * @return 		always true
   */
  public boolean supportsCustomEditor() {
    return true;
  }

  /**
   * Set (or change) the object that is to be edited.
   *
   * @param value The new target object to be edited.  Note that this
   *     object should not be modified by the PropertyEditor, rather
   *     the PropertyEditor should create a new object to hold any
   *     modified value.
   */
  public void setValue(Object value) {
    super.setValue(value);
    if (m_CustomEditor != null)
      initForDisplay();
  }

  /**
   * Creates the window adapater to attach.
   *
   * @return		the WindowAdapter to use
   */
  protected WindowAdapter createWindowAdapter() {
    return new WindowAdapter() {
      public void windowClosing(WindowEvent e) {
	cleanUp();
	super.windowClosing(e);
      }
    };
  }

  /**
   * Adds a window listener to the dialog.
   *
   * @see		#initForDisplay()
   */
  protected void addWindowAdapter() {
    Dialog 	dlg;

    if (m_CustomEditor instanceof Container) {
      dlg = GUIHelper.getParentDialog((Container) m_CustomEditor);
      if (dlg != null) {
	if (m_WindowAdapter == null)
	  m_WindowAdapter = createWindowAdapter();
	dlg.removeWindowListener(m_WindowAdapter);
	dlg.addWindowListener(m_WindowAdapter);
      }
    }
  }

  /**
   * Cleans up when the dialog is closed.
   * <p/>
   * Default implementation does nothing.
   */
  protected void cleanUp() {
  }

  /**
   * Closes the dialog.
   */
  protected void closeDialog() {
    Dialog 	dlg;

    cleanUp();

    if (m_CustomEditor instanceof Container) {
      dlg = GUIHelper.getParentDialog((Container) m_CustomEditor);
      if (dlg != null)
	dlg.setVisible(false);
    }
  }

  /**
   * Creates the custom editor.
   *
   * @return		the editor
   */
  protected abstract JComponent createCustomEditor();

  /**
   * Initializes the display of the value.
   */
  protected abstract void initForDisplay();

  /**
   * Gets the custom editor component.
   *
   * @return 		the editor
   */
  public Component getCustomEditor() {
    if (m_CustomEditor == null) {
      m_CustomEditor = createCustomEditor();
      m_CustomEditor.addMouseListener(new MouseAdapter() {
	public void mouseClicked(MouseEvent e) {
	  JPopupMenu popup = createPopup();
	  if (MouseUtils.isRightClick(e) && (popup != null))
	    popup.show(m_CustomEditor, e.getX(), e.getY());
	  else
	    super.mouseClicked(e);
	}
      });
      addWindowAdapter();
    }

    // some editors require to have access to their parent application, hence
    // we need to initialize whenever the custom editor is accessed.
    initForDisplay();

    return m_CustomEditor;
  }

  /**
   * Creates a popup menu.
   *
   * @return		the popup menu, null if not supported
   */
  protected JPopupMenu createPopup() {
    return VariableSupport.createPopup(VariableSupport.findParent(m_CustomEditor), this);
  }
}
