/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * ImageViewerPanel.java
 * Copyright (C) 2010-2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.visualization.image;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.Vector;

import javax.swing.JCheckBoxMenuItem;
import javax.swing.JColorChooser;
import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.core.Properties;
import adams.core.io.PlaceholderFile;
import adams.env.Environment;
import adams.env.ImageViewerPanelDefinition;
import adams.gui.chooser.ImageFileChooser;
import adams.gui.core.BaseMenu;
import adams.gui.core.BasePanel;
import adams.gui.core.BaseTabbedPane;
import adams.gui.core.GUIHelper;
import adams.gui.core.MenuBarProvider;
import adams.gui.core.MouseUtils;
import adams.gui.core.RecentFilesHandler;
import adams.gui.core.TitleGenerator;
import adams.gui.core.Undo.UndoPoint;
import adams.gui.event.RecentFileEvent;
import adams.gui.event.RecentFileListener;
import adams.gui.sendto.SendToActionSupporter;
import adams.gui.sendto.SendToActionUtils;
import adams.gui.visualization.core.PopupMenuCustomizer;
import adams.gui.visualization.image.plugins.AbstractImageViewerPlugin;

/**
 * A simple image viewer.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class ImageViewerPanel
  extends BasePanel
  implements MenuBarProvider, SendToActionSupporter {

  /** for serialization. */
  private static final long serialVersionUID = -7291572371004753723L;

  /** the name of the props file. */
  public final static String FILENAME = "ImageViewer.props";

  /** the file to store the recent files in. */
  public final static String SESSION_FILE = "ImageViewerSession.props";

  /** the properties. */
  protected static Properties m_Properties;

  /** the panel itself. */
  protected ImageViewerPanel m_Self;

  /** the file chooser for the pictures. */
  protected ImageFileChooser m_FileChooser;

  /** an optional customizer for the right-click popup. */
  protected PopupMenuCustomizer m_PopupMenuCustomizer;

  /** the menu bar, if used. */
  protected JMenuBar m_MenuBar;

  /** the "load recent" submenu. */
  protected JMenu m_MenuItemFileLoadRecent;

  /** the menu item "save as". */
  protected JMenuItem m_MenuItemFileSaveAs;

  /** the menu item "close". */
  protected JMenuItem m_MenuItemFileClose;

  /** the menu item "enable undo". */
  protected JCheckBoxMenuItem m_MenuItemEditEnableUndo;

  /** the menu item "undo". */
  protected JMenuItem m_MenuItemEditUndo;

  /** the menu item "redo". */
  protected JMenuItem m_MenuItemEditRedo;

  /** the menu item "copy". */
  protected JMenuItem m_MenuItemEditCopy;

  /** the menu "zoom". */
  protected JMenu m_MenuViewZoom;

  /** the menu item "zoom in". */
  protected JMenuItem m_MenuItemViewZoomIn;

  /** the menu item "zoom out". */
  protected JMenuItem m_MenuItemViewZoomOut;

  /** the menu item "background color". */
  protected JMenuItem m_MenuItemViewBackgroundColor;

  /** the menu item "show properties". */
  protected JMenuItem m_MenuItemViewShowProperties;

  /** the menu "plugins". */
  protected BaseMenu m_MenuPlugins;

  /** the plugins. */
  protected Vector<AbstractImageViewerPlugin> m_Plugins;

  /** the plugin menu items. */
  protected Vector<JMenuItem> m_MenuItemPlugins;

  /** the tabbed pane with the images. */
  protected BaseTabbedPane m_TabbedPane;

  /** the recent files handler. */
  protected RecentFilesHandler m_RecentFilesHandler;

  /** for generating the title. */
  protected TitleGenerator m_TitleGenerator;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Self                = this;
    m_FileChooser         = new ImageFileChooser();
    m_FileChooser.setCurrentDirectory(new File(getProperties().getString("InitialDir", "%h")));
    m_FileChooser.setAutoAppendExtension(true);
    m_PopupMenuCustomizer = null;
    m_RecentFilesHandler  = null;
    m_TitleGenerator      = new TitleGenerator("Image viewer", true);
    m_MenuItemPlugins     = new Vector<JMenuItem>();
    m_Plugins             = new Vector<AbstractImageViewerPlugin>();
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    super.initGUI();

    setLayout(new BorderLayout());

    m_TabbedPane = new BaseTabbedPane() {
      private static final long serialVersionUID = -2247884686333300541L;
      protected boolean canCloseTabWithMiddleMouseButton(int index) {
	return checkForModified((ImagePanel) m_TabbedPane.getComponentAt(index));
      };
    };
    m_TabbedPane.setCloseTabsWithMiddelMouseButton(true);
    m_TabbedPane.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	if (getCurrentPanel() != null)
	  getCurrentPanel().showStatus("");
	update();
      }
    });
    add(m_TabbedPane, BorderLayout.CENTER);

    addMouseListener(new MouseAdapter() {
      public void mouseClicked(MouseEvent e) {
	if (!MouseUtils.isPrintScreenClick(e)) {
	  if (MouseUtils.isRightClick(e)) {
	    JPopupMenu menu = getPopupMenu(e);
	    menu.show(m_Self, e.getX(), e.getY());
	  }
	}
      }
    });
  }

  /**
   * Updates the title and menu.
   */
  protected void update() {
    updateTitle();
    updateTabTitle();
    updateMenu();
  }

  /**
   * Updates the title of the current tab, taken modified state into account.
   */
  protected void updateTabTitle() {
    int		index;
    String	title;
    boolean	modified;

    index = m_TabbedPane.getSelectedIndex();
    if (index >= 0) {
      title   = m_TabbedPane.getTitleAt(index);
      modified = title.startsWith("*");
      if (modified)
	title = title.substring(1);
      if (getCurrentPanel().isModified() != modified) {
	if (getCurrentPanel().isModified())
	  title = "*" + title;
	m_TabbedPane.setTitleAt(index, title);
      }
    }
  }

  /**
   * Updats the title of the dialog/frame if applicable.
   */
  protected void updateTitle() {
    boolean	modified;

    modified = false;
    if (getCurrentPanel() != null)
      modified = getCurrentPanel().isModified();

    setParentTitle(m_TitleGenerator.generate(getCurrentFile(), modified));
  }

  /**
   * updates the enabled state of the menu items.
   */
  protected void updateMenu() {
    boolean	imageAvailable;
    ImagePanel	panel;
    int		i;
    boolean	enabled;

    if (m_MenuBar == null)
      return;

    imageAvailable = (getCurrentImage() != null);
    panel          = getCurrentPanel();

    // File
    m_MenuItemFileLoadRecent.setEnabled(m_RecentFilesHandler.size() > 0);
    m_MenuItemFileSaveAs.setEnabled(imageAvailable);
    m_MenuItemFileClose.setEnabled(imageAvailable);

    // Edit
    m_MenuItemEditEnableUndo.setEnabled(panel != null);
    m_MenuItemEditEnableUndo.setSelected((panel != null) && panel.getUndo().isEnabled());
    m_MenuItemEditUndo.setEnabled((panel != null) && panel.getUndo().canUndo());
    if ((panel != null) && panel.getUndo().canUndo()) {
      m_MenuItemEditUndo.setText("Undo - " + panel.getUndo().peekUndoComment(true));
      m_MenuItemEditUndo.setToolTipText(panel.getUndo().peekUndoComment());
    }
    else {
      m_MenuItemEditUndo.setText("Undo");
      m_MenuItemEditUndo.setToolTipText(null);
    }
    m_MenuItemEditRedo.setEnabled((panel != null) && panel.getUndo().canRedo());
    if ((panel != null) && panel.getUndo().canRedo()) {
      m_MenuItemEditRedo.setText("Redo - " + panel.getUndo().peekRedoComment(true));
      m_MenuItemEditRedo.setToolTipText(panel.getUndo().peekRedoComment());
    }
    else {
      m_MenuItemEditRedo.setText("Redo");
      m_MenuItemEditRedo.setToolTipText(null);
    }
    m_MenuItemEditCopy.setEnabled(imageAvailable);

    // View
    m_MenuViewZoom.setEnabled(imageAvailable);
    m_MenuItemViewZoomIn.setEnabled(imageAvailable);
    m_MenuItemViewZoomOut.setEnabled(imageAvailable);
    m_MenuItemViewBackgroundColor.setEnabled(imageAvailable);
    m_MenuItemViewShowProperties.setSelected(imageAvailable && getCurrentPanel().getShowProperties());
    m_MenuItemViewShowProperties.setEnabled(imageAvailable);

    // Plugins
    for (i = 0; i < m_Plugins.size(); i++) {
      enabled = m_Plugins.get(i).canExecute(getCurrentPanel());
      m_MenuItemPlugins.get(i).setEnabled(enabled);
    }
  }

  /**
   * Returns the image panel in the currently selected tab.
   *
   * @return		the image panel, null if none available
   */
  public ImagePanel getCurrentPanel() {
    ImagePanel	result;
    int		index;

    result = null;

    index = m_TabbedPane.getSelectedIndex();
    if (index >= 0)
      result = (ImagePanel) m_TabbedPane.getComponentAt(index);

    return result;
  }

  /**
   * Returns the underlying image.
   *
   * @return		the current image, can be null
   */
  public BufferedImage getCurrentImage() {
    BufferedImage	result;

    result = null;

    if (getCurrentPanel() != null)
      result = getCurrentPanel().getCurrentImage();

    return result;
  }

  /**
   * Returns the current filename.
   *
   * @return		the current filename, can be null
   */
  public File getCurrentFile() {
    File	result;

    result = null;

    if (getCurrentPanel() != null)
      result = getCurrentPanel().getCurrentFile();

    return result;
  }

  /**
   * Creates a menu bar (singleton per panel object). Can be used in frames.
   *
   * @return		the menu bar
   */
  public JMenuBar getMenuBar() {
    JMenuBar		result;
    JMenu		menu;
    JMenu		submenu;
    JMenuItem		menuitem;
    int			i;
    int[]		zooms;
    String[]		shortcuts;
    String[]		plugins;

    if (m_MenuBar == null) {
      result = new JMenuBar();

      // File
      menu = new JMenu("File");
      result.add(menu);
      menu.setMnemonic('F');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });

      // File/Open...
      menuitem = new JMenuItem("Open...");
      menu.add(menuitem);
      menuitem.setMnemonic('O');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed O"));
      menuitem.setIcon(GUIHelper.getIcon("open.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  open();
	}
      });

      // File/Recent files
      submenu = new JMenu("Open recent");
      menu.add(submenu);
      m_RecentFilesHandler = new RecentFilesHandler(
	  SESSION_FILE, getProperties().getInteger("MaxRecentImages", 5), submenu);
      m_RecentFilesHandler.addRecentFileListener(new RecentFileListener() {
	public void recentFileAdded(RecentFileEvent e) {
	  // ignored
	}
	public void recentFileSelected(RecentFileEvent e) {
	  load(e.getFile());
	}
      });
      m_MenuItemFileLoadRecent = submenu;

      // File/Open...
      menuitem = new JMenuItem("Save as...");
      menu.add(menuitem);
      menuitem.setMnemonic('v');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl shift pressed S"));
      menuitem.setIcon(GUIHelper.getIcon("save.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  saveAs();
	}
      });
      m_MenuItemFileSaveAs = menuitem;

      // File/Close tab
      menuitem = new JMenuItem("Close tab");
      menu.add(menuitem);
      menuitem.setMnemonic('t');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed W"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  close();
	}
      });
      m_MenuItemFileClose = menuitem;

      // File/Send to
      menu.addSeparator();
      if (SendToActionUtils.addSendToSubmenu(this, menu))
	menu.addSeparator();

      // File/Close
      menuitem = new JMenuItem("Close");
      menu.add(menuitem);
      menuitem.setMnemonic('C');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed Q"));
      menuitem.setIcon(GUIHelper.getIcon("exit.png"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  exit();
	}
      });

      // Edit
      menu = new JMenu("Edit");
      result.add(menu);
      menu.setMnemonic('E');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });

      // Edit/Enable undo
      menuitem = new JCheckBoxMenuItem("Undo enabled");
      menu.add(menuitem);
      menuitem.setMnemonic('n');
      menuitem.setIcon(GUIHelper.getEmptyIcon());
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  getCurrentPanel().getUndo().setEnabled(!getCurrentPanel().getUndo().isEnabled());
	}
      });
      m_MenuItemEditEnableUndo = (JCheckBoxMenuItem) menuitem;

      // Edit/Undo
      menuitem = new JMenuItem("Undo");
      menu.add(menuitem);
      menuitem.setMnemonic('U');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed Z"));
      menuitem.setIcon(GUIHelper.getIcon("undo.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  undo();
	}
      });
      m_MenuItemEditUndo = menuitem;

      // Edit/Redo
      menuitem = new JMenuItem("Redo");
      menu.add(menuitem);
      menuitem.setMnemonic('R');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed Y"));
      menuitem.setIcon(GUIHelper.getIcon("redo.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  redo();
	}
      });
      m_MenuItemEditRedo = menuitem;

      // Edit/Copy
      menuitem = new JMenuItem("Copy");
      menu.addSeparator();
      menu.add(menuitem);
      menuitem.setMnemonic('C');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed C"));
      menuitem.setIcon(GUIHelper.getIcon("copy.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  copy();
	}
      });
      m_MenuItemEditCopy = menuitem;

      // View
      menu = new JMenu("View");
      result.add(menu);
      menu.setMnemonic('V');
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });

      // View/Zoom
      submenu = new JMenu("Zoom");
      menu.add(submenu);
      submenu.setMnemonic('Z');
      submenu.setIcon(GUIHelper.getIcon("glasses.gif"));
      submenu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });
      m_MenuViewZoom = submenu;

      //View/Zoom/Zoom in
      menuitem = new JMenuItem("Zoom in");
      submenu.add(menuitem);
      menuitem.setMnemonic('i');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl shift pressed I"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  getCurrentPanel().setScale(getCurrentPanel().getScale() * 1.5);
	}
      });
      m_MenuItemViewZoomIn = menuitem;

      //View/Zoom/Zoom out
      menuitem = new JMenuItem("Zoom out");
      submenu.add(menuitem);
      menuitem.setMnemonic('o');
      menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl shift pressed O"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  getCurrentPanel().setScale(getCurrentPanel().getScale() / 1.5);
	}
      });
      m_MenuItemViewZoomOut = menuitem;

      // zoom levels
      // TODO: add "fit" zoom
      zooms = new int[]{
	  /* -100, */
	  25,
	  50,
	  66,
	  75,
	  100,
	  150,
	  200,
	  400,
	  800};
      shortcuts = new String[]{
	  /* "F", */
	  "",
	  "",
	  "",
	  "",
	  "1",
	  "",
	  "2",
	  "4",
	  ""
      };
      submenu.addSeparator();
      for (i = 0; i < zooms.length; i++) {
	final int fZoom = zooms[i];
	menuitem = new JMenuItem(zooms[i] + "%");
	submenu.add(menuitem);
	if (shortcuts[i].length() > 0)
	  menuitem.setAccelerator(GUIHelper.getKeyStroke(shortcuts[i]));
	menuitem.addActionListener(new ActionListener() {
	  public void actionPerformed(ActionEvent e) {
	    zoom(fZoom);
	  }
	});
      }

      // View/Background color
      menuitem = new JMenuItem("Background color...");
      menu.add(menuitem);
      menuitem.setIcon(GUIHelper.getIcon("colorpicker.png"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  chooseBackgroundColor();
	}
      });
      m_MenuItemViewBackgroundColor = menuitem;

      // View/Properties
      menuitem = new JCheckBoxMenuItem("Show properties");
      menu.add(menuitem);
      menuitem.setIcon(GUIHelper.getIcon("properties.gif"));
      menuitem.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  getCurrentPanel().setShowProperties(!getCurrentPanel().getShowProperties());
	}
      });
      m_MenuItemViewShowProperties = menuitem;

      // Plugins
      plugins = AbstractImageViewerPlugin.getPlugins();
      menu = new BaseMenu("Plugins");
      result.add(menu);
      menu.setMnemonic('P');
      menu.setVisible(plugins.length > 0);
      menu.addChangeListener(new ChangeListener() {
	public void stateChanged(ChangeEvent e) {
	  updateMenu();
	}
      });
      m_MenuPlugins = (BaseMenu) menu;

      // add plugins
      m_MenuItemPlugins.clear();
      m_Plugins.clear();
      for (i = 0; i < plugins.length; i++) {
	try {
	  final AbstractImageViewerPlugin plugin = (AbstractImageViewerPlugin) Class.forName(plugins[i]).newInstance();
	  menuitem = new JMenuItem(plugin.getCaption());
	  menu.add(menuitem);
	  menuitem.addActionListener(new ActionListener() {
	    public void actionPerformed(ActionEvent e) {
	      getCurrentPanel().addUndoPoint("Saving undo data...", "Plugin: " + plugin.getCaption());
	      String error = plugin.execute(getCurrentPanel());
	      if (error != null)
		GUIHelper.showErrorMessage(
		    getCurrentPanel(),
		    "Error occurred executing plugin '" + plugin.getCaption() + "':\n" + error);
	      update();
	    }
	  });
	  m_Plugins.add(plugin);
	  m_MenuItemPlugins.add(menuitem);
	}
	catch (Exception e) {
	  System.err.println("Failed to install plugin '" + plugins[i] + "':");
	  e.printStackTrace();
	}
      }
      m_MenuPlugins.sort();

      // update menu
      m_MenuBar = result;
      updateMenu();
    }
    else {
      result = m_MenuBar;
    }

    return result;
  }

  /**
   * Opens an image.
   */
  protected void open() {
    int		retVal;

    retVal = m_FileChooser.showOpenDialog(this);
    if (retVal != ImageFileChooser.APPROVE_OPTION)
      return;

    load(m_FileChooser.getSelectedFile());
  }

  /**
   * Loads the specified file in a new panel.
   *
   * @param file	the file to load
   */
  public void load(File file) {
    ImagePanel	panel;

    panel = new ImagePanel();
    if (!panel.load(file)) {
      GUIHelper.showErrorMessage(
	  this, "Failed to open image '" + file + "'!");
    }
    else {
      panel.setScale(getProperties().getDouble("ZoomLevel") / 100);
      m_TabbedPane.addTab(file.getName(), panel);
      m_TabbedPane.setSelectedComponent(panel);
      if (m_RecentFilesHandler != null)
	m_RecentFilesHandler.addRecentFile(file);
    }

    update();
  }

  /**
   * Saves the current image under a new name.
   */
  protected void saveAs() {
    int		index;
    int		retVal;
    File	file;
    ImagePanel	panel;

    index  = m_TabbedPane.getSelectedIndex();
    if (index < 0)
      return;
    panel = (ImagePanel) m_TabbedPane.getComponentAt(index);

    m_FileChooser.setSelectedFile(getCurrentFile());
    retVal = m_FileChooser.showSaveDialog(this);
    if (retVal != ImageFileChooser.APPROVE_OPTION)
      return;

    file = m_FileChooser.getSelectedFile();
    if (!panel.save(file)) {
      GUIHelper.showErrorMessage(
	  this, "Failed to write image to '" + file + "'!");
    }
    else {
      if (m_RecentFilesHandler != null)
	m_RecentFilesHandler.addRecentFile(file);
    }

    update();
  }

  /**
   * Returns whether we can proceed with the operation or not, depending on
   * whether the user saved the flow or discarded the changes.
   *
   * @param panel	the panel to check
   * @return		true if safe to proceed
   */
  protected boolean checkForModified(ImagePanel panel) {
    boolean 	result;
    int		retVal;
    String	msg;

    if (panel == null)
      return true;

    result = !panel.isModified();

    if (!result) {
      if (getCurrentFile() == null)
	msg = "Image not saved - save?";
      else
	msg = "Image '" + getCurrentFile() + "' not saved - save?";

      retVal = JOptionPane.showConfirmDialog(
	  this,
	  msg,
	  "Image not saved",
	  JOptionPane.YES_NO_CANCEL_OPTION);

      switch (retVal) {
	case JOptionPane.YES_OPTION:
	  saveAs();
	  result = !panel.isModified();
	  break;
	case JOptionPane.NO_OPTION:
	  result = true;
	  break;
	case JOptionPane.CANCEL_OPTION:
	  result = false;
	  break;
      }
    }

    return result;
  }

  /**
   * Closes the current image.
   */
  protected void close() {
    int		index;
    boolean	canClose;

    index    = m_TabbedPane.getSelectedIndex();
    canClose = false;
    if (index >= 0)
      canClose = checkForModified(getCurrentPanel());

    if (canClose) {
      m_TabbedPane.remove(index);
      update();
    }
  }

  /**
   * Exits the viewer.
   */
  protected void exit() {
    int		i;

    i = 0;
    while (i < m_TabbedPane.getTabCount()) {
      if (!checkForModified((ImagePanel) m_TabbedPane.getComponentAt(i)))
	return;
      else
	m_TabbedPane.remove(i);
    }

    if (getParentFrame() != null) {
      getParentFrame().setVisible(false);
      getParentFrame().dispose();
    }
    else if (getParentDialog() != null) {
      getParentDialog().setVisible(false);
      getParentDialog().dispose();
    }
  }

  /**
   * Copies the current image to the clipboard.
   */
  protected void copy() {
    GUIHelper.copyToClipboard(getCurrentImage());
  }

  /**
   * Zooms in/out.
   *
   * @param zoom	the zoom (in percent)
   */
  protected void zoom(int zoom) {
    getCurrentPanel().setScale((double) zoom / 100);
  }

  /**
   * Lets the user select the background color for the image.
   */
  protected void chooseBackgroundColor() {
    Color	bgcolor;

    bgcolor = JColorChooser.showDialog(this, "Background color", getCurrentPanel().getBackgroundColor());
    if (bgcolor == null)
      return;
    getCurrentPanel().setBackgroundColor(bgcolor);
  }

  /**
   * Sets the class to customize the right-click popup menu.
   *
   * @param value	the customizer
   */
  public void setPopupMenuCustomizer(PopupMenuCustomizer value) {
    m_PopupMenuCustomizer = value;
  }

  /**
   * Returns the current customizer, can be null.
   *
   * @return		the customizer
   */
  public PopupMenuCustomizer getPopupMenuCustomizer() {
    return m_PopupMenuCustomizer;
  }

  /**
   * Returns the popup menu, potentially customized.
   *
   * @param e		the mouse event
   * @return		the popup menu
   * @see		#m_PopupMenuCustomizer
   */
  public JPopupMenu getPopupMenu(MouseEvent e) {
    JPopupMenu	result;

    result = new JPopupMenu();

    // customize it?
    if (m_PopupMenuCustomizer != null)
      m_PopupMenuCustomizer.customizePopupMenu(e, result);

    return result;
  }

  /**
   * Returns the classes that the supporter generates.
   *
   * @return		the classes
   */
  public Class[] getSendToClasses() {
    return new Class[]{PlaceholderFile.class, JComponent.class};
  }

  /**
   * Checks whether something to send is available.
   *
   * @param cls		the classes to retrieve an item for
   * @return		true if an object is available for sending
   */
  public boolean hasSendToItem(Class[] cls) {
    return (getCurrentImage() != null);
  }

  /**
   * Returns the object to send.
   *
   * @param cls		the classes to retrieve the item for
   * @return		the item to send
   */
  public Object getSendToItem(Class[] cls) {
    Object	result;
    File	file;

    result = null;
    file   = getCurrentFile();

    if (SendToActionUtils.isAvailable(PlaceholderFile.class, cls)) {
      if ((file == null) && (getCurrentImage() != null)) {
	result = SendToActionUtils.nextTmpFile("imageviewer", "png");
	getCurrentPanel().save((File) result);
	getCurrentPanel().setCurrentImage(getCurrentPanel().getCurrentImage());
      }
      else if (file != null) {
	result = new PlaceholderFile(file);
      }
    }
    else if (SendToActionUtils.isAvailable(JComponent.class, cls)) {
      if (getCurrentPanel() != null)
	result = getCurrentPanel().getPaintPanel();
    }

    return result;
  }

  /**
   * peforms an undo if possible.
   */
  public void undo() {
    ImagePanel	panel;
    UndoPoint 	point;

    panel = getCurrentPanel();
    if (panel == null)
      return;
    if (!panel.getUndo().canUndo())
      return;

    panel.showStatus("Performing Undo...");

    // add redo point
    panel.getUndo().addRedo(panel.getState(), panel.getUndo().peekUndoComment());

    point = panel.getUndo().undo();
    panel.setState((Vector) point.getData());
    update();
    panel.showStatus("");
  }

  /**
   * peforms a redo if possible.
   */
  public void redo() {
    ImagePanel	panel;
    UndoPoint 	point;

    panel = getCurrentPanel();
    if (panel == null)
      return;
    if (!panel.getUndo().canRedo())
      return;

    panel.showStatus("Performing Redo...");

    // add undo point
    panel.getUndo().addUndo(panel.getState(), panel.getUndo().peekRedoComment(), true);

    point = panel.getUndo().redo();
    panel.setState((Vector) point.getData());

    update();
    panel.showStatus("");
  }

  /**
   * Returns the properties that define the editor.
   *
   * @return		the properties
   */
  public static synchronized Properties getProperties() {
    if (m_Properties == null)
      m_Properties = Environment.getInstance().read(ImageViewerPanelDefinition.KEY);

    return m_Properties;
  }
}
