/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * AbstractFilenameGenerator.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import adams.core.option.AbstractOptionHandler;

/**
 * Ancestor for filename generators.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractFilenameGenerator
  extends AbstractOptionHandler {

  /** for serialization. */
  private static final long serialVersionUID = -9019484464686478277L;

  /** the parent directory of the file. */
  protected PlaceholderDirectory m_Directory;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "dir", "directory",
	    new PlaceholderDirectory("."));
  }

  /**
   * Sets the parent directory for the generated filename.
   *
   * @param value	the directory
   */
  public void setDirectory(PlaceholderDirectory value) {
    m_Directory = value;
    reset();
  }

  /**
   * Returns the parent directory for the generated filename.
   *
   * @return		the directory
   */
  public PlaceholderDirectory getDirectory() {
    return m_Directory;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String directoryTipText() {
    return "The parent directory of the generated filename.";
  }

  /**
   * Performs checks on the object to generate the filename for.
   * <p/>
   * Default implementation only checks whether object is null.
   *
   * @param obj		the object to check
   */
  protected void check(Object obj) {
    if (obj == null)
      throw new IllegalArgumentException("Object to generate filename for is null!");
  }

  /**
   * Performs the actual generation of the filename.
   *
   * @param obj		the object to generate the filename for
   * @return		the generated filename
   */
  protected abstract String doGenerate(Object obj);

  /**
   * Generates a filename for the object.
   *
   * @param obj		the object to generate the filename for
   * @return		the generated filename
   */
  public String generate(Object obj) {
    check(obj);
    return doGenerate(obj);
  }
}
