/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Conversion.java
 * Copyright (C) 2010-2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.util.Enumeration;
import java.util.Hashtable;

import adams.core.Properties;
import adams.env.ConversionDefinition;
import adams.env.Environment;

/**
 * Helper class that converts classnames from options on the fly into the
 * most up-to-date format.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class Conversion {

  /** the name of the props file. */
  public final static String FILENAME = "Conversion.props";

  /** the key that lists the classname mapping keys. */
  public final static String KEY_RENAME = "Rename";

  /** the key that lists the partial renaming keys. */
  public final static String KEY_PARTIAL_RENAME = "PartialRename";

  /** the properties. */
  protected static Properties m_Properties;

  /** the mapping (exact classname replacement). */
  protected static Hashtable<String,String> m_Mapping;

  /** the mapping (partial renaming). */
  protected static Hashtable<String,String> m_MappingPartial;

  /**
   * Returns the properties that define the editor.
   *
   * @return		the properties
   */
  public static synchronized Properties getProperties() {
    if (m_Properties == null)
      m_Properties = Environment.getInstance().read(ConversionDefinition.KEY);

    return m_Properties;
  }

  /**
   * Initializes the mapping.
   */
  protected static synchronized void initMappings() {
    Properties		props;
    String[]		keys;
    int			i;
    String[]		parts;

    if (m_Mapping == null) {
      props     = getProperties();

      // exact mappings
      keys      = props.getString(KEY_RENAME, "").split(",");
      m_Mapping = new Hashtable<String,String>();
      if ((keys.length >= 1) && (keys[0].length() != 0)) {
	for (i = 0; i < keys.length; i++) {
	  if (!props.hasKey(keys[i])) {
	    System.err.println("Rename mapping '" + keys[i] + "' not found - skipped!");
	    continue;
	  }

	  parts = props.getString(keys[i], "").split("-");
	  if (parts.length != 2) {
	    System.err.println(
		"Wrong format for rename mapping ('" + keys[i] + "'): "
		+ props.getString(keys[i]));
	    continue;
	  }

	  m_Mapping.put(parts[0], parts[1]);
	}
      }

      // partial renaming
      keys             = props.getString(KEY_PARTIAL_RENAME, "").split(",");
      m_MappingPartial = new Hashtable<String,String>();
      if ((keys.length >= 1) && (keys[0].length() != 0)) {
	for (i = 0; i < keys.length; i++) {
	  if (!props.hasKey(keys[i])) {
	    System.err.println("Partial renaming '" + keys[i] + "' not found - skipped!");
	    continue;
	  }

	  parts = props.getString(keys[i], "").split("\t");
	  if (parts.length != 2) {
	    System.err.println(
		"Wrong format for partial renaming ('" + keys[i] + "'): "
		+ props.getString(keys[i]));
	    continue;
	  }

	  m_MappingPartial.put(parts[0], parts[1]);
	}
      }
    }
  }

  /**
   * Renames a classname if necessary.
   *
   * @param classname	the classname to process
   * @return		the potentially updated classname
   */
  public static synchronized String rename(String classname) {
    String		result;
    Enumeration<String>	keys;
    String		key;

    result = classname;

    initMappings();

    // replace classname
    while (m_Mapping.containsKey(result))
      result = m_Mapping.get(result);

    // partial renaming
    keys = m_MappingPartial.keys();
    while (keys.hasMoreElements()) {
      key    = keys.nextElement();
      result = result.replace(key, m_MappingPartial.get(key));
    }

    return result;
  }
}
