/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractContainer.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.container;

import java.io.Serializable;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Hashtable;
import java.util.Vector;

import adams.core.CloneHandler;
import adams.core.Utils;

/**
 * Ancestor of all containers. A container allows the access to the stored
 * values via their names.
 * <p/>
 * NB: containers need to declare a default constructor. The default constructor
 * is used to generate help information.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4681 $
 */
public abstract class AbstractContainer
  implements Serializable, CloneHandler<AbstractContainer> {

  /** for serialization. */
  private static final long serialVersionUID = -6949950627956848217L;

  /** for storing the values. */
  protected Hashtable<String,Object> m_Values;

  /** additional names for values. */
  protected HashSet<String> m_AdditionalNames;
  
  /**
   * Initializes the container.
   */
  protected AbstractContainer() {
    checkDefaultConstructor();
    
    m_Values          = new Hashtable<String,Object>();
    m_AdditionalNames = new HashSet<String>();
  }

  /**
   * Checks whether a default constructor is available - necessary for
   * generating HTML help.
   */
  protected void checkDefaultConstructor() {
    try {
      getClass().getConstructor(new Class[0]);
    }
    catch (Exception e) {
      throw new IllegalStateException(getClass().getName() + " does not have default constructor!");
    }
  }

  /**
   * Returns a clone of itself using serialization.
   *
   * @return		the clone
   */
  public AbstractContainer getClone() {
    return (AbstractContainer) Utils.deepCopy(this);
  }

  /**
   * Returns all value names that can be used (theoretically).
   *
   * @return		enumeration over all possible value names
   */
  public abstract Enumeration<String> names();

  /**
   * Returns all available stored value names.
   *
   * @return		enumeration over all stored value names
   */
  public Enumeration<String> stored() {
    Vector<String>	result;

    result = new Vector<String>(m_Values.keySet());
    Collections.sort(result);

    return result.elements();
  }

  /**
   * Checks whether a given value is non-null.
   *
   * @param name	the name of the value to check
   * @return		true if the value is non-null
   */
  public boolean hasValue(String name) {
    return (getValue(name) != null);
  }

  /**
   * Returns the value associated with the given name.
   *
   * @param name	the name of the value
   * @return		the associated value or null if not available
   */
  public Object getValue(String name) {
    return m_Values.get(name);
  }

  /**
   * Checks whether the name of the object is valid.
   *
   * @param name	the name to check
   * @return		true if valid
   * @see		#names()
   */
  protected boolean isValidName(String name) {
    boolean		result;
    Enumeration<String>	names;

    result = m_AdditionalNames.contains(name);
    
    if (!result) {
      names = names();
      while (names.hasMoreElements()) {
	if (names.nextElement().equals(name)) {
	  result = true;
	  break;
	}
      }
    }

    return result;
  }

  /**
   * Stores the value under the name.
   *
   * @param name	the name of the value
   * @param value	the value to store
   * @return		true if successfully stored, i.e., value is not null
   */
  protected boolean store(String name, Object value) {
    if (value == null)
      return false;

    m_Values.put(name, value);

    return true;
  }

  /**
   * Sets the named value.
   *
   * @param name	the name of the value
   * @param value	the value to store in the container
   * @return		true if the name was recognized and the value was
   * 			stored successfully, false otherwise
   */
  public boolean setValue(String name, Object value) {
    if (isValidName(name))
      return store(name, value);
    else
      return false;
  }

  /**
   * Checks whether the setup of the container is valid.
   *
   * @return		true if all the necessary values are available
   */
  public abstract boolean isValid();

  /**
   * Adds the specified name as valid name for a value.
   * 
   * @param name	the name to add
   * @return		true if the additional names list changed
   */
  public boolean addAdditionalName(String name) {
    return m_AdditionalNames.add(name);
  }
  
  /**
   * Removes the additional name again.
   * 
   * @param name	the name to remove
   * @return		true if the name was present and got removed
   */
  public boolean removeAdditionalName(String name) {
    return m_AdditionalNames.remove(name);
  }
  
  /**
   * Returns a short description of the stored data.
   *
   * @return		short description
   */
  public String toString() {
    String		result;
    Enumeration<String>	names;
    String		name;

    result = "";
    names  = stored();
    while (names.hasMoreElements()) {
      name = names.nextElement();
      if (result.length() > 0)
	result += ", ";
      result += name + "=" + getValue(name);
    }

    return result;
  }
}
