/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractDisplay.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.core;

import java.awt.BorderLayout;
import java.util.Hashtable;

import javax.swing.ImageIcon;
import javax.swing.SwingUtilities;

import adams.core.CleanUpHandler;
import adams.gui.core.BaseDialog;
import adams.gui.core.BaseFrame;
import adams.gui.core.BasePanel;
import adams.gui.core.GUIHelper;
import adams.gui.core.MenuBarProvider;

/**
 * Ancestor for actors that display stuff.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4660 $
 */
public abstract class AbstractDisplay
  extends AbstractActor {

  /** for serialization. */
  private static final long serialVersionUID = 8175993838879683118L;

  /** the key for storing the input token in the backup. */
  public final static String BACKUP_INPUT = "input";

  /** the current token. */
  protected transient Token m_InputToken;

  /** the width of the dialog. */
  protected int m_Width;

  /** the height of the dialog. */
  protected int m_Height;

  /** the X position of the dialog. */
  protected int m_X;

  /** the Y position of the dialog. */
  protected int m_Y;

  /** the panel to display. */
  protected BasePanel m_Panel;

  /** the dialog that's being displayed. */
  protected BaseFrame m_Frame;

  /** whether the GUI is currently being updated. */
  protected Boolean m_Updating;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "width", "width",
	    getDefaultWidth(), 1, null);

    m_OptionManager.add(
	    "height", "height",
	    getDefaultHeight(), 1, null);

    m_OptionManager.add(
	    "x", "x",
	    getDefaultX(), -3, null);

    m_OptionManager.add(
	    "y", "y",
	    getDefaultY(), -3, null);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("x");
    result   = "X:";
    if (variable != null) {
      result = variable;
    }
    else {
      if (m_X == -1)
	result += "left";
      else if (m_X == -2)
	result += "center";
      else if (m_X == -3)
	result += "right";
      else
	result += m_X;
    }

    variable = getOptionManager().getVariableForProperty("y");
    result   += ", Y:";
    if (variable != null) {
      result += variable;
    }
    else {
      if (m_Y == -1)
	result += "top";
      else if (m_Y == -2)
	result += "center";
      else if (m_Y == -3)
	result += "bottom";
      else
	result += m_Y;
    }

    variable = getOptionManager().getVariableForProperty("width");
    result   += ", W:";
    if (variable != null)
      result += variable;
    else
      result += m_Width;

    variable = getOptionManager().getVariableForProperty("height");
    result   += ", H:";
    if (variable != null)
      result += variable;
    else
      result += m_Height;

    return result;
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    if (m_InputToken != null)
      result.put(BACKUP_INPUT, m_InputToken);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_INPUT)) {
      m_InputToken = (Token) state.get(BACKUP_INPUT);
      state.remove(BACKUP_INPUT);
    }

    super.restoreState(state);
  }

  /**
   * The method that accepts the input token and then processes it.
   *
   * @param token	the token to accept and process
   * @see		#m_InputToken
   */
  public void input(Token token) {
    if (!m_Skip)
      m_InputToken = token;
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    super.wrapUp();

    m_InputToken = null;
  }

  /**
   * Returns the default X position for the dialog.
   *
   * @return		the default X position
   */
  protected int getDefaultX() {
    return -1;
  }

  /**
   * Returns the default Y position for the dialog.
   *
   * @return		the default Y position
   */
  protected int getDefaultY() {
    return -1;
  }

  /**
   * Returns the default width for the dialog.
   *
   * @return		the default width
   */
  protected int getDefaultWidth() {
    return 800;
  }

  /**
   * Returns the default height for the dialog.
   *
   * @return		the default height
   */
  protected int getDefaultHeight() {
    return 600;
  }

  /**
   * Sets the width of the dialog.
   *
   * @param value 	the width
   */
  public void setWidth(int value) {
    m_Width = value;
    reset();
  }

  /**
   * Returns the currently set width of the dialog.
   *
   * @return 		the width
   */
  public int getWidth() {
    return m_Width;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String widthTipText() {
    return "The width of the dialog.";
  }

  /**
   * Sets the height of the dialog.
   *
   * @param value 	the height
   */
  public void setHeight(int value) {
    m_Height = value;
    reset();
  }

  /**
   * Returns the currently set height of the dialog.
   *
   * @return 		the height
   */
  public int getHeight() {
    return m_Height;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String heightTipText() {
    return "The height of the dialog.";
  }

  /**
   * Sets the X position of the dialog.
   *
   * @param value 	the X position
   */
  public void setX(int value) {
    m_X = value;
    reset();
  }

  /**
   * Returns the currently set X position of the dialog.
   *
   * @return 		the X position
   */
  public int getX() {
    return m_X;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String xTipText() {
    return "The X position of the dialog (>=0: absolute, -1: left, -2: center, -3: right).";
  }

  /**
   * Sets the Y position of the dialog.
   *
   * @param value 	the Y position
   */
  public void setY(int value) {
    m_Y = value;
    reset();
  }

  /**
   * Returns the currently set Y position of the dialog.
   *
   * @return 		the Y position
   */
  public int getY() {
    return m_Y;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String yTipText() {
    return "The Y position of the dialog (>=0: absolute, -1: top, -2: center, -3: bottom).";
  }

  /**
   * Resets the object. Removes graphical components as well.
   */
  protected void reset() {
    super.reset();

    cleanUpGUI();
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Panel    = null;
    m_Frame    = null;
    m_Updating = false;
  }

  /**
   * Clears the content of the panel.
   */
  public abstract void clearPanel();

  /**
   * Creates the panel to display in the dialog.
   *
   * @return		the panel
   */
  protected abstract BasePanel newPanel();

  /**
   * Returns the panel.
   * 
   * @return		the panel, null if not available
   */
  public BasePanel getPanel() {
    return m_Panel;
  }
  
  /**
   * Creates a title for the dialog. Default implementation only returns
   * the full name of the actor.
   *
   * @return		the title of the dialog
   */
  protected String createTitle() {
    return getFullName();
  }

  /**
   * Hook method before the frame gets created.
   * <p/>
   * Default implementation does nothing.
   *
   * @param panel	the panel to display in the frame
   */
  protected void preCreateFrame(BasePanel panel) {
  }

  /**
   * Creates the actual frame.
   *
   * @param panel	the panel to display in the frame
   * @return		the created frame
   */
  protected BaseFrame doCreateFrame(BasePanel panel) {
    BaseFrame	result;
    ImageIcon	icon;
    int		width;
    int		height;

    result = new BaseFrame(createTitle());

    // limit width/height to screen size (taking X/Y into account)
    width  = Math.min(GUIHelper.getScreenBounds(result).width - m_X, getWidth());
    height = Math.min(GUIHelper.getScreenBounds(result).height - m_Y, getHeight());

    result.getContentPane().setLayout(new BorderLayout());
    result.getContentPane().add(panel, BorderLayout.CENTER);
    result.setDefaultCloseOperation(BaseDialog.HIDE_ON_CLOSE);
    result.setSize(width, height);
    icon = GUIHelper.getIcon(getClass());
    if (icon != null)
      result.setIconImage(icon.getImage());
    else
      result.setIconImage(GUIHelper.getIcon("flow.gif").getImage());
    if (panel instanceof MenuBarProvider)
      result.setJMenuBar(((MenuBarProvider) panel).getMenuBar());
    else if (this instanceof MenuBarProvider)
      result.setJMenuBar(((MenuBarProvider) this).getMenuBar());
    result.setLocation(ActorUtils.determineLocation(result, m_X, m_Y));

    return result;
  }

  protected void afterFrameShow(BaseFrame frame) {
  }

  /**
   * Hook method after the frame got created.
   * <p/>
   * Default implementation does nothing.
   *
   * @param frame	the frame that got just created
   * @param panel	the panel displayed in the frame
   */
  protected void postCreateFrame(BaseFrame frame, BasePanel panel) {
  }

  /**
   * Creates and initializes a frame with the just created panel.
   *
   * @param panel	the panel to use in the frame
   * @return		the created frame
   */
  protected BaseFrame createFrame(BasePanel panel) {
    BaseFrame	result;

    preCreateFrame(panel);
    result = doCreateFrame(panel);
    postCreateFrame(result, panel);

    return result;
  }

  /**
   * Returns the frame.
   * 
   * @return		the frame, null if not available
   */
  public BaseFrame getFrame() {
    return m_Frame;
  }

  /**
   * Returns a runnable that displays frame, etc.
   * Must call notifyAll() on the m_Self object and set m_Updating to false.
   *
   * @return		the runnable
   * @see		#m_Updating
   */
  protected abstract Runnable newDisplayRunnable();

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    Runnable	runnable;

    if (!isHeadless()) {
      if (m_Panel == null) {
	m_Panel = newPanel();
	m_Frame = createFrame(m_Panel);
      }

      m_Updating = true;
      runnable   = newDisplayRunnable();

      synchronized(m_Self) {
	SwingUtilities.invokeLater(runnable);
	try {
	  m_Self.wait();
	}
	catch (Exception e) {
	  // ignored
	}
      }
    }

    return null;
  }

  /**
   * Removes all graphical components.
   */
  protected void cleanUpGUI() {
    if (m_Frame != null) {
      if (m_Panel instanceof CleanUpHandler)
	((CleanUpHandler) m_Panel).cleanUp();

      m_Frame.setVisible(false);
      m_Frame.dispose();

      m_Frame = null;
      m_Panel = null;
    }
  }

  /**
   * Stops the execution. No message set.
   */
  public void stopExecution() {
    try {
      synchronized(m_Self) {
	m_Self.notifyAll();
      }
    }
    catch (Exception e) {
      // ignored
    }

    super.stopExecution();
  }

  /**
   * Cleans up after the execution has finished. Also removes graphical
   * components.
   */
  public void cleanUp() {
    Runnable	runnable;

    super.cleanUp();

    runnable = new Runnable() {
      public void run() {
	cleanUpGUI();
      }
    };
    SwingUtilities.invokeLater(runnable);
  }
}
