/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * RandomNumberGenerator.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.source;

import adams.data.random.AbstractRandomNumberGenerator;
import adams.data.random.JavaRandomDouble;
import adams.flow.core.Token;

/**
 <!-- globalinfo-start -->
 * Random number generator. The type of random numbers depends on the chosen generator.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Double<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: RandomNumberGenerator
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-generator &lt;adams.data.random.AbstractRandomNumberGenerator [options]&gt; (property: generator)
 * &nbsp;&nbsp;&nbsp;The random number generator to use.
 * &nbsp;&nbsp;&nbsp;default: adams.data.random.Normal
 * </pre>
 *
 * <pre>-max-num &lt;int&gt; (property: maxNum)
 * &nbsp;&nbsp;&nbsp;The maximum number of random numbers to generate (-1 is unlimited).
 * &nbsp;&nbsp;&nbsp;default: 1000
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class RandomNumberGenerator
  extends AbstractSource {

  /** for serialization. */
  private static final long serialVersionUID = 6216146938771296415L;

  /** the random number generator to use. */
  protected AbstractRandomNumberGenerator m_Generator;

  /** the maximum number of random numbers to generate. */
  protected int m_MaxNum;

  /** the counter for the random numbers generated. */
  protected int m_Count;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Random number generator. The type of random numbers depends on the chosen generator.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "generator", "generator",
	    new JavaRandomDouble());

    m_OptionManager.add(
	    "max-num", "maxNum",
	    1000);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("generator");
    if (variable != null)
      result = variable;
    else
      result = m_Generator.getClass().getName().replace("adams.data.random.", "");

    result += "/";
    variable = getOptionManager().getVariableForProperty("maxNum");
    if (variable != null)
      result += variable;
    else
      result += m_MaxNum;

    return result;
  }

  /**
   * Sets the random number generator to use.
   *
   * @param value	the generator
   */
  public void setGenerator(AbstractRandomNumberGenerator value) {
    m_Generator = value;
    reset();
  }

  /**
   * Returns the random number generator to use.
   *
   * @return		the generator
   */
  public AbstractRandomNumberGenerator getGenerator() {
    return m_Generator;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String generatorTipText() {
    return "The random number generator to use.";
  }

  /**
   * Sets the maximum number of random numbers to generate.
   *
   * @param value	the maximum number (-1 is unlimited)
   */
  public void setMaxNum(int value) {
    m_MaxNum = value;
    reset();
  }

  /**
   * Returns the maximum number of random numbers to generate.
   *
   * @return		the maximum number (-1 is unlimited)
   */
  public int getMaxNum() {
    return m_MaxNum;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String maxNumTipText() {
    return "The maximum number of random numbers to generate (-1 is unlimited).";
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.Double.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    return new Class[]{Double.class};
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    if (m_Generator != null)
      m_Generator.reset();
    m_Count = 0;
  }

  /**
   * Does nothing.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    return null;
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    m_Count++;
    result = new Token(new Double(m_Generator.next().doubleValue()));

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return m_Executed && (((m_MaxNum > -1) && (m_Count < m_MaxNum)) || (m_MaxNum == -1));
  }
}
