/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * RecentFilesHandler.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.core;

import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.JMenu;
import javax.swing.JMenuItem;

import adams.core.ConsoleObject;
import adams.core.Properties;
import adams.core.io.FileUtils;
import adams.env.Environment;
import adams.gui.event.RecentFileEvent;
import adams.gui.event.RecentFileListener;


/**
 * A class that handles a list of recent files. Reads/writes them from/to
 * a props file in the application's home directory.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @see Environment#getHome()
 */
public class RecentFilesHandler
  extends ConsoleObject {

  /** for serialization. */
  private static final long serialVersionUID = 7532226757387619342L;

  /** the property for storing the number of recent files. */
  public final static String RECENTFILES_COUNT = "RecentFilesCount";

  /** the property prefix for a recent file. */
  public final static String RECENTFILES_PREFIX = "RecentFile.";

  /** the props file to use. */
  protected String m_PropertiesFile;

  /** the prefix for the properties. */
  protected String m_PropertyPrefix;

  /** the maximum number of files to keep. */
  protected int m_MaxCount;

  /** the JMenu to add the files as sub-items to. */
  protected JMenu m_Menu;

  /** the files. */
  protected Vector<File> m_RecentFiles;

  /** whether to ignore changes temporarily. */
  protected boolean m_IgnoreChanges;

  /** the event listeners. */
  protected HashSet<RecentFileListener> m_Listeners;

  /**
   * Initializes the handler with a maximum of 5 items.
   *
   * @param propsFile	the props file to store the files in
   * @param menu	the menu to add the recent files as subitems to
   */
  public RecentFilesHandler(String propsFile, JMenu menu) {
    this(propsFile, 5, menu);
  }

  /**
   * Initializes the handler.
   *
   * @param propsFile	the props file to store the files in
   * @param maxCount	the maximum number of files to keep in menu
   * @param menu	the menu to add the recent files as subitems to
   */
  public RecentFilesHandler(String propsFile, int maxCount, JMenu menu) {
    this(propsFile, null, maxCount, menu);
  }

  /**
   * Initializes the handler.
   *
   * @param propsFile	the props file to store the files in
   * @param propPrefix	the properties prefix, use null to ignore
   * @param maxCount	the maximum number of files to keep in menu
   * @param menu	the menu to add the recent files as subitems to
   */
  public RecentFilesHandler(String propsFile, String propPrefix, int maxCount, JMenu menu) {
    super();

    m_PropertiesFile = Environment.getInstance().getHome() + File.separator + new File(propsFile).getName();
    m_PropertyPrefix = propPrefix;
    m_MaxCount       = maxCount;
    m_Menu           = menu;
    m_RecentFiles    = new Vector<File>();
    m_IgnoreChanges  = false;
    m_Listeners      = new HashSet<RecentFileListener>();

    readProps();
    updateMenu();
  }

  /**
   * Returns the props file used to store the recent files in.
   *
   * @return		the filename
   */
  public String getPropertiesFile() {
    return m_PropertiesFile;
  }

  /**
   * Returns the prefix for the property names.
   *
   * @return		the prefix
   */
  public String getPropertyPrefix() {
    return m_PropertyPrefix;
  }

  /**
   * Returns the maximum number of files to keep.
   *
   * @return		the maximum number
   */
  public int getMaxCount() {
    return m_MaxCount;
  }

  /**
   * Returns the menu to add the recent files as subitems to.
   *
   * @return		the menu
   */
  public JMenu getMenu() {
    return m_Menu;
  }

  /**
   * Adds the prefix to the property name if provided.
   *
   * @param property	the property to expand
   * @return		the expanded property name
   */
  protected String expand(String property) {
    if (m_PropertyPrefix == null)
      return property;
    else
      return m_PropertyPrefix + property;
  }

  /**
   * Loads the properties file from disk, if possible.
   *
   * @return		the properties file
   */
  protected Properties loadProps() {
    Properties	result;
    File	file;

    try {
      result = new Properties();
      file = new File(m_PropertiesFile);
      if (file.exists())
	result.load(m_PropertiesFile);
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = new Properties();
    }

    return result;
  }

  /**
   * Reads the recent files from the props file.
   */
  protected void readProps() {
    int		count;
    Properties	props;
    int		i;
    String	filename;
    File	file;

    m_IgnoreChanges = true;

    props = loadProps();
    count = props.getInteger(expand(RECENTFILES_COUNT), 0);
    m_RecentFiles.clear();
    for (i = count - 1; i >= 0; i--) {
      filename = props.getString(expand(RECENTFILES_PREFIX + i), "");
      if (filename.length() > 0) {
	file = new File(filename);
	if (file.exists())
	  addRecentFile(file);
      }
    }

    m_IgnoreChanges = false;
  }

  /**
   * Writes the current recent files back to the props file.
   */
  protected synchronized void writeProps() {
    Properties	props;
    int		i;

    props = loadProps();
    props.setInteger(expand(RECENTFILES_COUNT), m_RecentFiles.size());
    for (i = 0; i < m_RecentFiles.size(); i++)
      props.setString(expand(RECENTFILES_PREFIX + i), m_RecentFiles.get(i).getAbsolutePath());

    try {
      props.save(m_PropertiesFile);
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
    }
  }

  /**
   * Determines the minimum number of parent directories that need to be
   * included in the filename to make the filenames in the menu distinguishable.
   *
   * @return		the minimum number of parent directories, -1 means
   * 			full path
   */
  protected synchronized int determineMinimumNumberOfParentDirs() {
    int			result;
    HashSet<String>	files;
    int			num;
    int			i;

    result = -1;

    num = 0;
    do {
      files = new HashSet<String>();
      for (i = 0; i < m_RecentFiles.size(); i++)
	files.add(FileUtils.createPartialFilename(m_RecentFiles.get(i), num));
      if (files.size() == m_RecentFiles.size())
	result = num;
      else
	num++;
    }
    while (files.size() < m_RecentFiles.size());

    return result;
  }

  /**
   * Updates the menu with the currently stored recent files.
   */
  protected synchronized void updateMenu() {
    int			i;
    JMenuItem		item;
    int			num;

    num = determineMinimumNumberOfParentDirs();

    m_Menu.removeAll();
    m_Menu.setEnabled(m_RecentFiles.size() > 0);
    for (i = 0; i < m_RecentFiles.size(); i++) {
      final File file = m_RecentFiles.get(i);
      item = new JMenuItem((i+1) + " - " + FileUtils.createPartialFilename(file, num));
      if (i < 9)
	item.setMnemonic(Integer.toString(i+1).charAt(0));
      if (i == 10)
	item.setMnemonic('0');
      item.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  notifyRecentFileListenersOfSelect(file);
	}
      });

      m_Menu.add(item);
    }
  }

  /**
   * Adds the file to the internal list.
   *
   * @param file	the file to add to the list
   */
  public synchronized void addRecentFile(File file) {
    file = new File(file.getAbsolutePath());

    // is it the first file again? -> ignore it
    if (m_RecentFiles.size() > 0) {
      if (file.equals(m_RecentFiles.get(0)))
	return;
    }

    m_RecentFiles.remove(file);
    m_RecentFiles.add(0, file);
    while (m_RecentFiles.size() > m_MaxCount)
      m_RecentFiles.remove(m_RecentFiles.size() - 1);

    if (m_IgnoreChanges)
      return;

    writeProps();
    updateMenu();

    notifyRecentFileListenersOfAdd(file);
  }

  /**
   * Removes the file to the internal list, e.g., if it no longer exists on
   * disk.
   *
   * @param file	the file to remove from the list
   */
  public synchronized void removeRecentFile(File file) {
    file = new File(file.getAbsolutePath());
    m_RecentFiles.remove(file);

    if (m_IgnoreChanges)
      return;

    writeProps();
    updateMenu();
  }

  /**
   * Returns the currently stored recent files.
   *
   * @return		the files
   */
  public Vector<File> getRecentFiles() {
    return new Vector(m_RecentFiles);
  }

  /**
   * Returns the number of recent files currently stored.
   *
   * @return		the number of files
   */
  public int size() {
    return m_RecentFiles.size();
  }

  /**
   * Adds the listener to the internal list.
   *
   * @param l		the listener to add
   */
  public void addRecentFileListener(RecentFileListener l) {
    m_Listeners.add(l);
  }

  /**
   * Removes the listener from the internal list.
   *
   * @param l		the listener to remove
   */
  public void removeRecentFileListener(RecentFileListener l) {
    m_Listeners.remove(l);
  }

  /**
   * Notifies the listeners of a file that got added.
   *
   * @param file	the affected file
   */
  protected void notifyRecentFileListenersOfAdd(File file) {
    Iterator<RecentFileListener>	iter;
    RecentFileEvent			e;

    e    = new RecentFileEvent(this, file);
    iter = m_Listeners.iterator();
    while (iter.hasNext())
      iter.next().recentFileAdded(e);
  }

  /**
   * Notifies the listeners of a file that got selected.
   *
   * @param file	the affected file
   */
  protected void notifyRecentFileListenersOfSelect(File file) {
    Iterator<RecentFileListener>	iter;
    RecentFileEvent			e;

    e    = new RecentFileEvent(this, file);
    iter = m_Listeners.iterator();
    while (iter.hasNext())
      iter.next().recentFileSelected(e);
  }
}
