/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SpreadSheetTableModel.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.core;

import adams.core.Utils;
import adams.core.io.SpreadSheet;
import adams.core.io.SpreadSheet.Cell;
import adams.core.io.SpreadSheet.Row;


/**
 * The table model for displaying a SpreadSheet object.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class SpreadSheetTableModel
  extends AbstractBaseTableModel {

  /** for serialization. */
  private static final long serialVersionUID = 8062515320279133441L;

  /** the underlying spreadsheet. */
  protected SpreadSheet m_Sheet;

  /** the number of decimals to display. */
  protected int m_NumDecimals;

  /**
   * Initializes the model with an empty spread sheet.
   */
  public SpreadSheetTableModel() {
    this(new SpreadSheet());
  }

  /**
   * Initializes the model with the given spread sheet.
   *
   * @param sheet	the spread sheet to display
   */
  public SpreadSheetTableModel(SpreadSheet sheet) {
    this(sheet, -1);
  }

  /**
   * Initializes the model with the given spread sheet and number of decimals
   * to display.
   *
   * @param sheet	the spread sheet to display
   * @param numDec	the number of decimals to display
   */
  public SpreadSheetTableModel(SpreadSheet sheet, int numDec) {
    super();

    m_Sheet       = sheet;
    m_NumDecimals = numDec;
  }

  /**
   * Returns the number of rows in the sheet.
   *
   * @return		the number of rows
   */
  public int getRowCount() {
    return m_Sheet.getRowCount();
  }

  /**
   * Returns the number of columns in the sheet.
   *
   * @return		the number of columns
   */
  public int getColumnCount() {
    return m_Sheet.getColumnCount() + 1;
  }

  /**
   * Returns the name of the column at <code>columnIndex</code>.  This is used
   * to initialize the table's column header name.  Note: this name does
   * not need to be unique; two columns in a table can have the same name.
   *
   * @param	columnIndex	the index of the column
   * @return  the name of the column
   */
  public String getColumnName(int columnIndex) {
    String	result;
    Row	row;

    if (columnIndex == 0) {
      result = "Row";
    }
    else {
      row    = m_Sheet.getHeaderRow();
      result = row.getCell(row.getCellKey(columnIndex - 1)).getContent();
    }

    return result;
  }

  /**
   * Returns the most specific superclass for all the cell values in the
   * column.
   *
   * @param columnIndex     the index of the column
   * @return                the class of the specified column
   */
  public Class getColumnClass(int columnIndex) {
    Class	result;

    if (columnIndex == 0)
      result = Integer.class;
    else
      result = String.class;

    return result;
  }

  /**
   * Returns the value for the cell at columnIndex and rowIndex.
   *
   * @param rowIndex      the row
   * @param columnIndex   the column
   * @return              the value of the sepcified cell
   */
  public Object getValueAt(int rowIndex, int columnIndex) {
    Object	result;
    Row		row;
    Cell	cell;
    String	content;
    double	d;
    double	factor;

    if (columnIndex == 0) {
      result = new Integer(rowIndex + 1);
    }
    else {
      row     = m_Sheet.getRow(rowIndex);
      cell    = row.getCell(m_Sheet.getHeaderRow().getCellKey(columnIndex - 1));
      if (cell == null) {
	result = null;
      }
      else {
	content = cell.getContent();
	if (cell.isMissing()) {
	  result = null;
	}
	else if (cell.isLong()) {
	  result = cell.toLong();
	}
	else if (cell.isDouble()) {
	  d = cell.toDouble();
	  if (Double.isNaN(d)) {
	    result = "NaN";
	  }
	  else {
	    if (m_NumDecimals > -1) {
	      factor = Math.pow(10, m_NumDecimals);
	      d      = Math.round(d * factor) / factor;
	    }
	    result = Utils.doubleToStringFixed(d, m_NumDecimals);
	  }
	}
	else {
	  result = content;
	}
      }
    }

    return result;
  }

  /**
   * Returns the spread sheet cell at the specified location.
   *
   * @param rowIndex		the current display row index
   * @param columnIndex	the column index
   * @return			the cell or null if invalid coordinates
   */
  public Cell getCellAt(int rowIndex, int columnIndex) {
    Cell	result;
    Row	row;

    result = null;

    if (columnIndex > 0) {
      row    = m_Sheet.getRow(rowIndex);
      result = row.getCell(m_Sheet.getHeaderRow().getCellKey(columnIndex - 1));
    }

    return result;
  }

  /**
   * Sets the number of decimals to display. Use -1 to display all.
   *
   * @param value	the number of decimals
   */
  public void setNumDecimals(int value) {
    if (value >= -1) {
      if (m_NumDecimals != value) {
	m_NumDecimals = value;
	fireTableDataChanged();
      }
    }
    else {
      System.err.println(
	  "Number of displayed decimals must be >=0 or -1 to display all - "
	  + "provided: " + value);
    }
  }

  /**
   * Returns the currently set number of decimals. -1 if displaying all.
   *
   * @return		the number of decimals
   */
  public int getNumDecimals() {
    return m_NumDecimals;
  }

  /**
   * Returns the underlying sheet.
   *
   * @return		the spread sheet
   */
  public SpreadSheet getSheet() {
    return m_Sheet;
  }
}
