/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * StyledTextEditorPanel.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 * Copyright (C) Patrick Chan and Addison Wesley, Java Developers Almanac 2000 (undo/redo)
 */
package adams.gui.core;

import java.awt.BorderLayout;
import java.awt.Font;
import java.awt.event.ActionEvent;
import java.io.File;
import java.util.HashSet;
import java.util.List;

import javax.swing.AbstractAction;
import javax.swing.JOptionPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.event.DocumentEvent;
import javax.swing.event.DocumentListener;
import javax.swing.event.UndoableEditEvent;
import javax.swing.event.UndoableEditListener;
import javax.swing.text.Document;
import javax.swing.undo.UndoManager;

import adams.core.License;
import adams.core.Utils;
import adams.core.annotation.MixedCopyright;
import adams.core.io.FileUtils;
import adams.gui.chooser.BaseFileChooser;

/**
 * A panel that allows the editing of text, including undo/redo support.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4653 $
 */
@MixedCopyright(
    copyright = "Patrick Chan and Addision Wesley, Java Developers Almanac 2000",
    license = License.BSD3,
    url = "http://java.sun.com/developer/codesamples/examplets/javax.swing.undo/236.html",
    note = "Undo/redo"
)
public class StyledTextEditorPanel
  extends BasePanel {

  /** for serialization. */
  private static final long serialVersionUID = 3594108882868668611L;

  /** for displaying the text. */
  protected BaseTextPane m_TextPane;

  /** whether the content was modified. */
  protected boolean m_Modified;

  /** whether to ignore changes. */
  protected boolean m_IgnoreChanges;

  /** for saving the content. */
  protected BaseFileChooser m_FileChooser;

  /** for managing undo/redo. */
  protected UndoManager m_Undo;

  /** the last search string used. */
  protected String m_LastFind;

  /** the listeners for modification events. */
  protected HashSet<ChangeListener> m_ChangeListeners;

  /** the current file. */
  protected File m_CurrentFile;

  /**
   * For initializing members.
   */
  protected void initialize() {
    super.initialize();

    m_CurrentFile     = null;
    m_ChangeListeners = new HashSet<ChangeListener>();
    m_Undo            = new UndoManager();
    m_FileChooser     = new BaseFileChooser();
    m_FileChooser.addChoosableFileFilter(ExtensionFileFilter.getTextFileFilter());
    m_FileChooser.setDefaultExtension(ExtensionFileFilter.getTextFileFilter().getExtensions()[0]);
    m_FileChooser.setAutoAppendExtension(true);
  }

  /**
   * For initializing the GUI.
   */
  protected void initGUI() {
    super.initGUI();

    setLayout(new BorderLayout());

    // text
    m_TextPane = newBaseTextPane();
    m_TextPane.setFont(new Font("monospaced", Font.PLAIN, 12));
    add(new BaseScrollPane(m_TextPane), BorderLayout.CENTER);

    // listen for text changes
    m_TextPane.getDocument().addDocumentListener(new DocumentListener() {
      public void removeUpdate(DocumentEvent e) {
	m_Modified = true;
      }
      public void insertUpdate(DocumentEvent e) {
	m_Modified = true;
      }
      public void changedUpdate(DocumentEvent e) {
	m_Modified = true;
      }
    });

    // Listen for undo and redo events
    m_TextPane.getDocument().addUndoableEditListener(new UndoableEditListener() {
      public void undoableEditHappened(UndoableEditEvent evt) {
	m_Modified = true;
	m_Undo.addEdit(evt.getEdit());
	notifyChangeListeners();
      }
    });

    // Create an undo action and add it to the text component
    m_TextPane.getActionMap().put("Undo", new AbstractAction("Undo") {
      private static final long serialVersionUID = -3023997491519283074L;
      public void actionPerformed(ActionEvent evt) {
	undo();
      }
    });

    // Bind the undo action to ctl-Z
    m_TextPane.getInputMap().put(GUIHelper.getKeyStroke("control Z"), "Undo");

    // Create a redo action and add it to the text component
    m_TextPane.getActionMap().put("Redo", new AbstractAction("Redo") {
      private static final long serialVersionUID = -3579642465298206034L;
      public void actionPerformed(ActionEvent evt) {
	redo();
      }
    });

    // Bind the redo action to ctl-Y
    m_TextPane.getInputMap().put(GUIHelper.getKeyStroke("control Y"), "Redo");

    setSize(600, 800);
  }

  /**
   * Returns a new text pane.
   *
   * @return		the text pane
   */
  protected BaseTextPane newBaseTextPane() {
    return new BaseTextPane();
  }

  /**
   * Sets the modified state. If false, all edits are discarded and the
   * last search string reset as well.
   *
   * @param value 	if true then the content is flagged as modified
   */
  public void setModified(boolean value) {
    m_Modified = value;
    if (!m_Modified)
      m_Undo.discardAllEdits();
    notifyChangeListeners();
  }

  /**
   * Returns whether the content has been modified.
   *
   * @return		true if the content was modified
   */
  public boolean isModified() {
    return m_Modified;
  }

  /**
   * Sets the content to display. Resets the modified state.
   *
   * @param value	the text
   */
  public void setContent(String value) {
    m_TextPane.setText(value);
  }

  /**
   * Returns the content to display.
   *
   * @return		the text
   */
  public String getContent() {
    return m_TextPane.getText();
  }

  /**
   * Sets whether the text area is editable or not.
   *
   * @param value	if true then the text will be editable
   */
  public void setEditable(boolean value) {
    m_TextPane.setEditable(value);
  }

  /**
   * Returns whether the text area is editable or not.
   *
   * @return		true if the text is editable
   */
  public boolean isEditable() {
    return m_TextPane.isEditable();
  }

  /**
   * Sets the font of the text area.
   *
   * @param value	the font to use
   */
  public void setTextFont(Font value) {
    m_TextPane.setFont(value);
  }

  /**
   * Returns the font currently in use by the text area.
   *
   * @return		the font in use
   */
  public Font getTextFont() {
    return m_TextPane.getFont();
  }

  /**
   * Returns the last search string.
   *
   * @return		the last search string, can be null if no search
   * 			performed yet
   */
  public String getLastFind() {
    return m_LastFind;
  }

  /**
   * Returns the underlying JTextArea element.
   *
   * @return		the component
   */
  public BaseTextPane getTextPane() {
    return m_TextPane;
  }

  /**
   * Returns the underlying document of the text area.
   *
   * @return		the document
   */
  public Document getDocument() {
    return m_TextPane.getDocument();
  }

  /**
   * Sets the position of the cursor.
   *
   * @param value	the position
   */
  public void setCaretPosition(int value) {
    m_TextPane.setCaretPosition(value);
  }

  /**
   * Returns the current position of the cursor.
   *
   * @return		the cursor position
   */
  public int getCaretPosition() {
    return m_TextPane.getCaretPosition();
  }

  /**
   * Sets the wordwrap state.
   *
   * @param value	whether to wrap or not
   */
  public void setWordWrap(boolean value) {
    m_TextPane.setWordWrap(value);
  }

  /**
   * Returns the wordwrap status.
   *
   * @return		true if wordwrap is on
   */
  public boolean getWordWrap() {
    return m_TextPane.getWordWrap();
  }

  /**
   * Returns whether we can proceed with the operation or not, depending on
   * whether the user saved the content or discarded the changes.
   *
   * @return		true if safe to proceed
   */
  public boolean checkForModified() {
    boolean 	result;
    int		retVal;
    String	msg;

    result = !isModified();

    if (!result) {
      msg = "Content not saved - save?";

      retVal = JOptionPane.showConfirmDialog(
        	  this,
        	  msg,
        	  "Content not saved",
        	  JOptionPane.YES_NO_CANCEL_OPTION);

      switch (retVal) {
	case JOptionPane.YES_OPTION:
	  saveAs();
	  result = !isModified();
	  break;
	case JOptionPane.NO_OPTION:
	  result = true;
	  break;
	case JOptionPane.CANCEL_OPTION:
	  result = false;
	  break;
      }
    }

    return result;
  }

  /**
   * Pops up dialog to open a file.
   */
  public void open() {
    int		retVal;

    retVal = m_FileChooser.showOpenDialog(this);
    if (retVal != BaseFileChooser.APPROVE_OPTION)
      return;

    open(m_FileChooser.getSelectedFile());
  }

  /**
   * Opens the specified file and loads/displays the content.
   *
   * @param file	the file to load
   */
  public void open(File file) {
    List<String>	content;

    content = FileUtils.loadFromFile(file);
    setContent(Utils.flatten(content, "\n"));
    setModified(false);

    m_CurrentFile = file;

    notifyChangeListeners();
  }

  /**
   * Pops up dialog to save the content in a file.
   */
  public void save() {
    if (m_CurrentFile == null)
      saveAs();
    else
      save(m_CurrentFile);
  }

  /**
   * Pops up dialog to save the content in a file.
   */
  public void saveAs() {
    int		retVal;

    retVal = m_FileChooser.showSaveDialog(this);
    if (retVal != BaseFileChooser.APPROVE_OPTION)
      return;

    save(m_FileChooser.getSelectedFile());
  }

  /**
   * Saves the content under the specified file.
   *
   * @param file	the file to save the content int
   */
  protected void save(File file) {
    if (!FileUtils.writeToFile(file.getAbsolutePath(), m_TextPane.getText(), false)) {
      GUIHelper.showErrorMessage(
	  this, "Error saving content to file '" + file + "'!");
    }
    else {
      m_CurrentFile = file;
      m_Modified    = false;
    }

    notifyChangeListeners();
  }

  /**
   * Removes all content. Does not reset the undos.
   */
  public void clear() {
    try {
      m_TextPane.getDocument().remove(0, m_TextPane.getDocument().getLength());
    }
    catch (Exception e) {
      // ignored
    }
    m_Modified = false;
    notifyChangeListeners();
  }

  /**
   * Checks whether an undo action is available.
   *
   * @return		true if an undo action is available
   */
  public boolean canUndo() {
    try {
      return m_Undo.canUndo();
    }
    catch (Exception ex) {
      return false;
    }
  }

  /**
   * Performs an undo, if possible.
   */
  public void undo() {
    try {
      // perform undo
      if (m_Undo.canUndo())
	m_Undo.undo();

      // last change undone?
      if (!m_Undo.canUndo())
	m_Modified = false;

      notifyChangeListeners();
    }
    catch (Exception ex) {
      // ignored
    }
  }

  /**
   * Checks whether a redo action is available.
   *
   * @return		true if a redo action is available
   */
  public boolean canRedo() {
    try {
      return m_Undo.canRedo();
    }
    catch (Exception ex) {
      return false;
    }
  }

  /**
   * Performs a redo, if possible.
   */
  public void redo() {
    try {
      if (m_Undo.canRedo()) {
	m_Undo.redo();
	m_Modified = true;
	notifyChangeListeners();
      }
    }
    catch (Exception ex) {
      // ignored
    }
  }

  /**
   * Checks whether text can be cut at the moment.
   *
   * @return		true if text is available for cutting
   */
  public boolean canCut() {
    if (isEditable() && (m_TextPane.getSelectedText() != null))
      return true;
    else
      return false;
  }

  /**
   * Cuts the currently selected text and places it on the clipboard.
   */
  public void cut() {
    m_TextPane.cut();
    notifyChangeListeners();
  }

  /**
   * Checks whether text can be copied at the moment.
   *
   * @return		true if text is available for copying
   */
  public boolean canCopy() {
    return (m_TextPane.getSelectedText() != null);
  }

  /**
   * Copies the currently selected text to the clipboard.
   */
  public void copy() {
    if (m_TextPane.getSelectedText() == null)
      GUIHelper.copyToClipboard(m_TextPane.getText());
    else
      m_TextPane.copy();
  }

  /**
   * Checks whether text can be pasted at the moment.
   *
   * @return		true if text is available for pasting
   */
  public boolean canPaste() {
    if (isEditable() && GUIHelper.canPasteFromClipboard())
      return true;
    else
      return false;
  }

  /**
   * Pastes the text from the clipboard into the document.
   */
  public void paste() {
    m_TextPane.paste();
    notifyChangeListeners();
  }

  /**
   * Selects all the text.
   */
  public void selectAll() {
    m_TextPane.selectAll();
  }

  /**
   * Initiates a search.
   */
  public void find() {
    String	search;
    int		index;

    search = JOptionPane.showInputDialog(this, "Enter search string", m_LastFind);
    if (search == null)
      return;

    index = m_TextPane.getText().indexOf(search, m_TextPane.getCaretPosition());
    if (index > -1) {
      m_LastFind = search;
      m_TextPane.setCaretPosition(index + search.length());
      m_TextPane.setSelectionStart(index);
      m_TextPane.setSelectionEnd(index + search.length());
    }
    else {
      GUIHelper.showErrorMessage(this, "Search string '" + search + "' not found!");
    }

    notifyChangeListeners();
  }

  /**
   * Finds the next occurrence.
   */
  public void findNext() {
    int		index;

    index = m_TextPane.getText().indexOf(m_LastFind, m_TextPane.getCaretPosition());
    if (index > -1) {
      m_TextPane.setCaretPosition(index + m_LastFind.length());
      m_TextPane.setSelectionStart(index);
      m_TextPane.setSelectionEnd(index + m_LastFind.length());
    }
    else {
      GUIHelper.showErrorMessage(this, "Search string '" + m_LastFind + "' not found!");
    }

    notifyChangeListeners();
  }

  /**
   * Pops up a print dialog.
   */
  public void printText() {
    m_TextPane.printText();
  }

  /**
   * Pops up a dialog for selecting the font.
   */
  public void selectFont() {
    m_TextPane.selectFont();
  }

  /**
   * Adds the given change listener to its internal list.
   *
   * @param l		the listener to add
   */
  public void addChangeListener(ChangeListener l) {
    m_ChangeListeners.add(l);
  }

  /**
   * Removes the given change listener from its internal list.
   *
   * @param l		the listener to remove
   */
  public void removeChangeListener(ChangeListener l) {
    m_ChangeListeners.add(l);
  }

  /**
   * Sends an event to all change listeners.
   */
  protected synchronized void notifyChangeListeners() {
    ChangeEvent 	e;

    e = new ChangeEvent(this);
    for (ChangeListener l: m_ChangeListeners)
      l.stateChanged(e);
  }
}
