/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * PropertiesPreferencesDialog.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.dialog;

import java.awt.Component;
import java.awt.Dialog;
import java.awt.Font;
import java.awt.Frame;
import java.util.Collections;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.JCheckBox;
import javax.swing.JSpinner;
import javax.swing.JTextField;

import adams.core.Properties;
import adams.core.io.PlaceholderDirectory;
import adams.core.io.PlaceholderFile;
import adams.gui.chooser.ColorChooserPanel;
import adams.gui.chooser.DirectoryChooserPanel;
import adams.gui.chooser.FileChooserPanel;
import adams.gui.chooser.FontChooserPanel;
import adams.gui.goe.FontEditor;

/**
 * Displays all properties in a props file as preferences (alphabetically
 * sorted). Simply uses text fields for the values.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class PropertiesPreferencesDialog
  extends AbstractPreferencesDialog {

  /** for serialization. */
  private static final long serialVersionUID = -822178750857036833L;

  /**
   * The various data types a preference can have.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4584 $
   */
  public enum PreferenceType {
    /** boolean. */
    BOOLEAN,
    /** integer. */
    INTEGER,
    /** double. */
    DOUBLE,
    /** string. */
    STRING,
    /** date. */
    DATE,
    /** color. */
    COLOR,
    /** font. */
    FONT,
    /** file. */
    FILE,
    /** directory. */
    DIRECTORY
  }

  /** the property/preference type relation. */
  protected Hashtable<String,PreferenceType> m_PreferenceTypes;

  /**
   * Creates a modal preferences dialog without a title with the specified
   * Dialog as its owner.
   *
   * @param owner	the owning dialog
   */
  public PropertiesPreferencesDialog(Dialog owner) {
    super(owner);
  }

  /**
   * Creates a modal preferences dialog without a title with the specified
   * Frame as its owner.
   *
   * @param owner	the owning frame
   */
  public PropertiesPreferencesDialog(Frame owner) {
    super(owner);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_PreferenceTypes = new Hashtable<String,PreferenceType>();
  }

  /**
   * Removes all property/preference type relations.
   */
  public void clearPreferenceTypes() {
    m_PreferenceTypes.clear();
  }

  /**
   * Associates the preference type with the specified property.
   *
   * @param property	the property to associate a type with
   * @param type	the preference type
   */
  public void addPreferenceType(String property, PreferenceType type) {
    m_PreferenceTypes.put(property, type);
  }

  /**
   * Checks whether a preference type has been specified for a particular
   * property.
   *
   * @param property	the property to associate a type with
   * @return		true if a type has been specified
   */
  public boolean hasPreferenceType(String property) {
    return m_PreferenceTypes.containsKey(property);
  }

  /**
   * Checks whether a preference type has been specified for a particular
   * property.
   *
   * @param property	the property to associate a type with
   * @return		true if a type has been specified
   */
  public PreferenceType getPreferenceType(String property) {
    if (hasPreferenceType(property))
      return m_PreferenceTypes.get(property);
    else
      return PreferenceType.STRING;
  }

  /**
   * Sets the properties to base the preferences on.
   *
   * @param value	the properties to use
   */
  public void setPreferences(Properties value) {
    Vector<String>		keys;
    JTextField			textfield;
    JCheckBox			checkbox;
    JSpinner			spinner;
    PreferenceType		type;
    FontChooserPanel		fontPanel;
    DirectoryChooserPanel	dirPanel;
    FileChooserPanel		filePanel;
    ColorChooserPanel		colorPanel;

    clearPreferences();
    keys = new Vector<String>(value.keySetAll());
    Collections.sort(keys);
    for (String key: keys) {
      type = getPreferenceType(key);
      switch (type) {
	case DATE:
	  // TODO
	case DOUBLE:
	case STRING:
	  textfield = new JTextField(20);
	  textfield.setText(value.getProperty(key));
	  addPreference(key, key, textfield);
	  break;
	case BOOLEAN:
	  checkbox = new JCheckBox();
	  checkbox.setSelected(value.getBoolean(key));
	  addPreference(key, key, checkbox);
	  break;
	case INTEGER:
	  spinner = new JSpinner();
	  spinner.setValue(value.getInteger(key));
	  addPreference(key, key, spinner);
	  break;
	case FONT:
	  fontPanel = new FontChooserPanel();
	  fontPanel.setCurrent((Font) FontEditor.valueOf(null, value.getProperty(key)));
	  addPreference(key, key, fontPanel);
	  break;
	case DIRECTORY:
	  dirPanel = new DirectoryChooserPanel();
	  dirPanel.setCurrent(new PlaceholderDirectory(value.getString(key)));
	  addPreference(key, key, dirPanel);
	  break;
	case FILE:
	  filePanel = new FileChooserPanel();
	  filePanel.setCurrent(new PlaceholderFile(value.getString(key)));
	  addPreference(key, key, filePanel);
	  break;
	case COLOR:
	  colorPanel = new ColorChooserPanel();
	  colorPanel.setCurrent(value.getColor(key));
	  addPreference(key, key, colorPanel);
	  break;
	default:
	  throw new IllegalStateException("Unhandled preference type (property '" + keys + "'): " + type);
      }
    }

    pack();
  }

  /**
   * Returns the currently display preferences as a properties object.
   *
   * @return		the preferences
   */
  public Properties getPreferences() {
    Properties			result;
    int				i;
    Component			comp;
    JTextField			textfield;
    JCheckBox			checkbox;
    JSpinner			spinner;
    FontChooserPanel		fontPanel;
    DirectoryChooserPanel	dirPanel;
    FileChooserPanel		filePanel;
    ColorChooserPanel		colorPanel;

    result = new Properties();

    for (i = 0; i < getPreferenceCount(); i++) {
      comp = getPreference(i);
      if (comp instanceof JTextField) {
	textfield = (JTextField) comp;
	result.setProperty(m_Identifiers.get(i), textfield.getText());
      }
      else if (comp instanceof JCheckBox) {
	checkbox = (JCheckBox) comp;
	result.setBoolean(m_Identifiers.get(i), checkbox.isSelected());
      }
      else if (comp instanceof JSpinner) {
	spinner = (JSpinner) comp;
	result.setInteger(m_Identifiers.get(i), ((Number) spinner.getValue()).intValue());
      }
      else if (comp instanceof FontChooserPanel) {
	fontPanel = (FontChooserPanel) comp;
	result.setProperty(m_Identifiers.get(i), FontEditor.toString(null, fontPanel.getCurrent()));
      }
      else if (comp instanceof DirectoryChooserPanel) {
	dirPanel = (DirectoryChooserPanel) comp;
	result.setProperty(m_Identifiers.get(i), dirPanel.getCurrent().toString());
      }
      else if (comp instanceof FileChooserPanel) {
	filePanel = (FileChooserPanel) comp;
	result.setProperty(m_Identifiers.get(i), filePanel.getCurrent().toString());
      }
      else if (comp instanceof ColorChooserPanel) {
	colorPanel = (ColorChooserPanel) comp;
	result.setColor(m_Identifiers.get(i), colorPanel.getCurrent());
      }
      else {
	throw new IllegalStateException("Unhandled component type: " + comp.getClass().getName());
      }
    }

    return result;
  }
}