/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * ExportDialog.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.flow;

import java.awt.BorderLayout;
import java.awt.Dialog;
import java.awt.Frame;
import java.io.File;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.core.option.AbstractOptionProducer;
import adams.core.option.FileBasedProducer;
import adams.core.option.NestedProducer;
import adams.gui.chooser.FileChooserPanel;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.core.ParameterPanel;
import adams.gui.dialog.AbstractApprovalDialog;
import adams.gui.goe.GenericObjectEditorPanel;

/**
 * A dialog for exporting a flow into various formats.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class ExportDialog
  extends AbstractApprovalDialog {

  /** for serialization. */
  private static final long serialVersionUID = 201725070566669323L;

  /** the GOE panel for the export format. */
  protected GenericObjectEditorPanel m_PanelGOE;

  /** the panel with the file. */
  protected FileChooserPanel m_PanelFile;

  /**
   * Creates a modeless dialog without a title with the specified Dialog as
   * its owner.
   *
   * @param owner	the owning dialog
   */
  public ExportDialog(Dialog owner) {
    super(owner);
  }

  /**
   * Creates a dialog with the specified owner Dialog and modality.
   *
   * @param owner	the owning dialog
   * @param modality	the type of modality
   */
  public ExportDialog(Dialog owner, ModalityType modality) {
    super(owner, modality);
  }

  /**
   * Creates a modeless dialog without a title with the specified Frame as
   * its owner.
   *
   * @param owner	the owning frame
   */
  public ExportDialog(Frame owner) {
    super(owner);
  }

  /**
   * Creates a dialog with the specified owner Frame, modality and an empty
   * title.
   *
   * @param owner	the owning frame
   * @param modal	whether the dialog is modal or not
   */
  public ExportDialog(Frame owner, boolean modal) {
    super(owner, modal);
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    ParameterPanel	params;

    super.initGUI();

    setTitle("Export flow");

    params = new ParameterPanel();
    getContentPane().add(params, BorderLayout.CENTER);

    m_PanelGOE = new GenericObjectEditorPanel(AbstractOptionProducer.class, new NestedProducer(), true);
    m_PanelGOE.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	updateFileChooser();
      }
    });
    params.addParameter("For_mat", m_PanelGOE);

    m_PanelFile = new FileChooserPanel();
    m_PanelFile.setPromptOverwriteFile(true);
    m_PanelFile.setUseSaveDialog(true);
    params.addParameter("_File", m_PanelFile);

    pack();
  }

  /**
   * Updates the file chooser's extension file filters.
   */
  protected void updateFileChooser() {
    FileBasedProducer	producer;
    ExtensionFileFilter	filter;

    m_PanelFile.setDefaultExtension(null);
    m_PanelFile.removeChoosableFileFilters();
    m_PanelFile.setAcceptAllFileFilterUsed(true);
    m_PanelFile.setAutoAppendExtension(false);

    if (m_PanelGOE.getCurrent() instanceof FileBasedProducer) {
      producer = (FileBasedProducer) m_PanelGOE.getCurrent();
      filter   = new ExtensionFileFilter(producer.getFileFormat(), producer.getFileExtensions());
      m_PanelFile.setDefaultExtension(producer.getDefaultFileExtension());
      m_PanelFile.setAutoAppendExtension(true);
      m_PanelFile.addChoosableFileFilter(filter);
      m_PanelFile.setFileFilter(filter);
    }
  }

  /**
   * Sets the producer to generate the output format.
   *
   * @param value	the producer to use
   */
  public void setProducer(AbstractOptionProducer value) {
    if (value == null)
      value = new NestedProducer();
    m_PanelGOE.setCurrent(value);
  }

  /**
   * Returns the producer currently in use.
   *
   * @return		the current producer
   */
  public AbstractOptionProducer getProducer() {
    return (AbstractOptionProducer) m_PanelGOE.getCurrent();
  }

  /**
   * Sets the file to save the options to.
   *
   * @param value	the file
   */
  public void setFile(File value) {
    m_PanelFile.setCurrent(value);
  }

  /**
   * Returns the current file to save the options to.
   *
   * @return		the file
   */
  public File getFile() {
    return m_PanelFile.getCurrent();
  }
}
