/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    ColorEditor.java
 *    Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.gui.goe;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.FlowLayout;
import java.awt.FontMetrics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

import javax.swing.JButton;
import javax.swing.JColorChooser;
import javax.swing.JComponent;
import javax.swing.JPanel;

import adams.core.option.AbstractOption;
import adams.gui.core.BasePanel;
import adams.gui.core.ColorHelper;

/**
 * A PropertyEditor for Color objects that lets the user select a color from
 * the color dialog.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class ColorEditor
  extends AbstractPropertyEditorSupport
  implements CustomStringRepresentationHandler {

  /** The color chooser used for selecting colors. */
  protected JColorChooser m_ColorChooser;

  /**
   * Returns the color as string.
   *
   * @param option	the current option
   * @param object	the color object to convert
   * @return		the generated string
   */
  public static String toString(AbstractOption option, Object object) {
    return ColorHelper.toHex((Color) object);
  }

  /**
   * Returns a color generated from the string.
   *
   * @param option	the current option
   * @param str		the string to convert to a color
   * @return		the generated color
   */
  public static Object valueOf(AbstractOption option, String str) {
    Color	result;

    result = ColorHelper.valueOf(str);
    if (result == null)
      result = Color.BLACK;

    return result;
  }

  /**
   * Returns a custom string representation of the object.
   *
   * @param obj		the object to turn into a string
   * @return		the string representation
   */
  public String toCustomStringRepresentation(Object obj) {
    return toString(null, obj);
  }

  /**
   * Returns an object created from the custom string representation.
   *
   * @param str		the string to turn into an object
   * @return		the object
   */
  public Object fromCustomStringRepresentation(String str) {
    return valueOf(null, str);
  }

  /**
   * Returns a representation of the current property value as java source.
   *
   * @return 		always "null"
   */
  public String getJavaInitializationString() {
    return "null";
  }

  /**
   * Gets the custom editor component.
   *
   * @return 		a value of type 'Component'
   */
  protected JComponent createCustomEditor() {
    BasePanel	result;
    JPanel	panelButtons;
    JButton	buttonOK;
    JButton	buttonCancel;
    Color 	currentColor;

    currentColor   = (Color) getValue();
    m_ColorChooser = new JColorChooser();
    m_ColorChooser.setColor(currentColor);

    result = new BasePanel(new BorderLayout());
    result.add(m_ColorChooser, BorderLayout.CENTER);

    panelButtons = new JPanel(new FlowLayout(FlowLayout.RIGHT));
    result.add(panelButtons, BorderLayout.SOUTH);

    buttonOK = new JButton("OK");
    buttonOK.setMnemonic('O');
    buttonOK.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	setValue(m_ColorChooser.getColor());
	closeDialog();
      }
    });
    panelButtons.add(buttonOK);

    buttonCancel = new JButton("Cancel");
    buttonCancel.setMnemonic('C');
    buttonCancel.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	closeDialog();
      }
    });
    panelButtons.add(buttonCancel);

    return result;
  }

  /**
   * Initializes the display of the value.
   */
  protected void initForDisplay() {
    Color 	currentColor;

    currentColor = (Color) getValue();
    if (currentColor != null)
      m_ColorChooser.setColor(currentColor);
  }

  /**
   * Paints a representation of the current Object.
   *
   * @param gfx 	the graphics context to use
   * @param box 	the area we are allowed to paint into
   */
  public void paintValue(java.awt.Graphics gfx, java.awt.Rectangle box) {
    FontMetrics fm = gfx.getFontMetrics();
    int vpad = (box.height - fm.getHeight()) / 2 ;
    Color color = (Color) getValue();
    String val = "No color";
    if (color != null)
      val = toString(null, color);
    gfx.drawString(val, 2, fm.getHeight() + vpad);
  }
}

