/**
 * DebugHelper.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core;

/**
 * Simple helper class for debugging output. Used in classes that don't
 * have natively debugging output.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2194 $
 */
public class DebugHelper {

  /** the environment variable suffix to general debug look for. */
  public final static String DEBUG_SUFFIX = ".DEBUG";

  /** the environment variable suffix of the debug level to look for. */
  public final static String DEBUGLEVEL_SUFFIX = ".DEBUGLEVEL";

  /**
   * Checks whether an environment variable is set for the class that determines
   * debugging. E.g., for the class "hello.world.App" the environment variable
   * "hello.world.App.DEBUG" is checked or the "hello.world.App.DEBUGLEVEL={int}"
   * with "{int}" greater than 0.
   *
   * @param cls		the class to check for debugging
   * @return		true if debugging is on
   */
  public static boolean isDebugOn(Class cls) {
    return    Boolean.getBoolean(cls.getName() + DEBUG_SUFFIX)
           || (getDebugLevel(cls) > 0);
  }

  /**
   * Returns the debug level for the specified class. E.g., for the class
   * "hello.world.App" the environment variable "hello.world.App.DEBUGLEVEL={int}"
   * is inspected and "{int}" (>= 0) returned.
   *
   * @param cls		the class to return the debug level for
   * @return		the debug level (>= 0; 0 = off)
   */
  public static int getDebugLevel(Class cls) {
    Integer	result;

    result = 0;

    if (Integer.getInteger(cls.getName() + DEBUGLEVEL_SUFFIX) != null)
      result = Integer.getInteger(cls.getName() + DEBUGLEVEL_SUFFIX);
    else if (Boolean.getBoolean(cls.getName() + DEBUG_SUFFIX))
      result = 1;

    if ((result == null) || (result < 0))
      result = 0;

    return result;
  }

  /**
   * Outputs some debugging information.
   *
   * @param cls		the class this output is for
   * @param msg		the message to output
   */
  public static void debug(Class cls, String msg) {
    String[]	lines;
    String	prefix;

    prefix = "[DEBUG] " + cls.getName() + ": ";
    lines  = msg.split("\n");
    for (String line: lines)
      System.out.println(prefix + line);
  }
}
