/**
 * AbstractSpreadSheetWriter.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import java.io.BufferedWriter;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileWriter;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;

import org.apache.commons.io.output.WriterOutputStream;

import adams.core.ClassLister;
import adams.core.option.OptionHandlingObject;

/**
 * Ancestor for classes that can write spreadsheet objects.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3896 $
 */
public abstract class AbstractSpreadSheetWriter
  extends OptionHandlingObject {

  /** for serialization. */
  private static final long serialVersionUID = -3547064795252689769L;

  /** The placeholder for missing values. */
  protected String m_MissingValue;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "missing", "missingValue",
	    getDefaultMissingValue());
  }

  /**
   * Returns a string describing the format (used in the file chooser).
   *
   * @return 			a description suitable for displaying in the
   * 				file chooser
   */
  public abstract String getFormatDescription();

  /**
   * Returns the extension(s) of the format.
   *
   * @return 			the extension (without the dot!)
   */
  public abstract String[] getFormatExtensions();

  /**
   * Returns the default missing value.
   *
   * @return		the default for missing values
   */
  protected String getDefaultMissingValue() {
    return "";
  }

  /**
   * Sets the placeholder for missing values.
   *
   * @param value	the placeholder
   */
  public void setMissingValue(String value) {
    m_MissingValue = value;
    reset();
  }

  /**
   * Returns the current placeholder for missing values.
   *
   * @return		the placeholder
   */
  public String getMissingValue() {
    return m_MissingValue;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   *         		displaying in the explorer/experimenter gui
   */
  public String missingValueTipText() {
    return "The placeholder for missing values.";
  }

  /**
   * Returns whether to write to an OutputStream rather than a Writer when
   * using a file name.
   *
   * @return		true if to write to an OutputStream
   */
  protected abstract boolean getUseOutputStream();

  /**
   * Writes the given content to the specified file.
   *
   * @param content	the content to write
   * @param file	the file to write to
   * @return		true if successfully written
   */
  public boolean write(SpreadSheet content, File file) {
    return write(content, file.getAbsolutePath());
  }

  /**
   * Writes the spreadsheet in CSV format to the given file.
   *
   * @param content	the spreadsheet to write
   * @param filename	the file to write the spreadsheet to
   * @return		true if successfully written
   */
  public boolean write(SpreadSheet content, String filename) {
    boolean		result;
    BufferedWriter	writer;
    OutputStream	output;

    result = true;

    try {
      if (getUseOutputStream()) {
	output = new FileOutputStream(filename);
	result = doWrite(content, output);
	output.close();
      }
      else {
	writer = new BufferedWriter(new FileWriter(filename));
	result = doWrite(content, writer);
	writer.close();
      }
    }
    catch (Exception e) {
      result = false;
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Writes the spreadsheet in CSV format to the given output stream. The caller
   * must ensure that the stream gets closed.
   *
   * @param content	the spreadsheet to write
   * @param stream	the output stream to write the spreadsheet to
   * @return		true if successfully written
   */
  public boolean write(SpreadSheet content, OutputStream stream) {
    if (getUseOutputStream())
      return doWrite(content, stream);
    else
      return doWrite(content, new OutputStreamWriter(stream));
  }

  /**
   * Writes the spreadsheet in CSV format to the given writer. The caller
   * must ensure that the writer gets closed.
   *
   * @param content	the spreadsheet to write
   * @param writer	the writer to write the spreadsheet to
   * @return		true if successfully written
   */
  public boolean write(SpreadSheet content, Writer writer) {
    if (getUseOutputStream())
      return doWrite(content, new WriterOutputStream(writer));
    else
      return doWrite(content, writer);
  }

  /**
   * Performs the actual writing. The caller must ensure that the writer gets
   * closed.
   * <p/>
   * Default implementation returns always false.
   *
   * @param content	the spreadsheet to write
   * @param writer	the writer to write the spreadsheet to
   * @return		true if successfully written
   */
  protected boolean doWrite(SpreadSheet content, Writer writer) {
    return false;
  }

  /**
   * Performs the actual writing. The caller must ensure that the output stream
   * gets closed.
   * <p/>
   * Default implementation returns always false.
   *
   * @param content	the spreadsheet to write
   * @param out		the output stream to write the spreadsheet to
   * @return		true if successfully written
   */
  protected boolean doWrite(SpreadSheet content, OutputStream out) {
    return false;
  }

  /**
   * Returns a list with classnames of writers.
   *
   * @return		the writer classnames
   */
  public static String[] getWriters() {
    return ClassLister.getSingleton().getClassnames(AbstractSpreadSheetWriter.class);
  }
}
