/**
 * ArrayProducer.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.Vector;

/**
 * Generates the string array format that is used on the command-line.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3030 $
 */
public class ArrayProducer
  extends AbstractRecursiveOptionProducer<String[],Vector<String>>
  implements RecursiveOptionProducer {

  /** for serialization. */
  private static final long serialVersionUID = 2014571979604068762L;

  /** the output vector. */
  protected Vector<String> m_OutputVector;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Generates a string array that can be used on the command-line.";
  }

  /**
   * Initializes the visitor.
   */
  protected void initialize() {
    super.initialize();

    m_OutputVector = new Vector<String>();
  }

  /**
   * Initializes the output data structure.
   *
   * @return		the created data structure
   */
  protected String[] initOutput() {
    return new String[0];
  }

  /**
   * Visits a boolean option.
   *
   * @param option	the boolean option
   * @return		the last internal data structure that was generated
   */
  public Vector<String> processOption(BooleanOption option) {
    Vector<String>	result;
    Object	currValue;

    result = new Vector<String>();

    try {
      currValue = getCurrentValue(option);
      if (!((Boolean) currValue).equals((Boolean) option.getDefaultValue()))
	result.add(getOptionIdentifier(option));
    }
    catch (Exception e) {
      System.err.println("Error obtaining current value for '" + option.getProperty() + "':");
      e.printStackTrace();
    }

    if (m_Nesting.empty())
      m_OutputVector.addAll(result);
    else
      ((Vector) m_Nesting.peek()).addAll(result);

    return result;
  }

  /**
   * Visits an argument option.
   *
   * @param option	the argument option
   * @return		the last internal data structure that was generated
   */
  public Vector<String> processOption(AbstractArgumentOption option) {
    Vector<String>	result;
    Object		currValue;
    Object		currValues;
    int			i;

    result = new Vector<String>();

    if (option.isVariableAttached() && !m_OutputVariableValues) {
      result.add(getOptionIdentifier(option));
      result.add(option.getVariable());
    }
    else {
      currValue = getCurrentValue(option);

      if (!isDefaultValue(option, currValue)) {
	currValues = null;

	if (currValue != null) {
	  if (!option.isMultiple()) {
	    currValues = Array.newInstance(option.getBaseClass(), 1);
	    Array.set(currValues, 0, currValue);
	  }
	  else {
	    currValues = currValue;
	  }

	  for (i = 0; i < Array.getLength(currValues); i++) {
	    result.add(getOptionIdentifier(option));
	    result.add(option.toString(Array.get(currValues, i)));
	  }
	}
      }
    }

    if (m_Nesting.empty())
      m_OutputVector.addAll(result);
    else
      ((Vector) m_Nesting.peek()).addAll(result);

    return result;
  }

  /**
   * Visits a class option.
   *
   * @param option	the class option
   * @return		the last internal data structure that was generated
   */
  public Vector<String> processOption(ClassOption option) {
    Vector<String>		result;
    Object			currValue;
    Object			currValues;
    int				i;
    Object			value;
    Vector<String>		nested;
    AbstractCommandLineHandler	handler;

    result = new Vector<String>();

    if (option.isVariableAttached() && !m_OutputVariableValues) {
      result.add(getOptionIdentifier(option));
      result.add(option.getVariable());
    }
    else {
      currValue = getCurrentValue(option);

      if (!isDefaultValue(option, currValue)) {
	currValues = null;

	if (currValue != null) {
	  if (!option.isMultiple()) {
	    currValues = Array.newInstance(option.getBaseClass(), 1);
	    Array.set(currValues, 0, currValue);
	  }
	  else {
	    currValues = currValue;
	  }

	  for (i = 0; i < Array.getLength(currValues); i++) {
	    value = Array.get(currValues, i);
	    result.add(getOptionIdentifier(option));
	    nested = new Vector<String>();
	    nested.add(value.getClass().getName());
	    if (value instanceof OptionHandler) {
	      m_Nesting.push(nested);
	      doProduce(((OptionHandler) value).getOptionManager());
	      m_Nesting.pop();
	    }
	    else {
	      handler = AbstractCommandLineHandler.getHandler(value);
	      nested.addAll(Arrays.asList(handler.getOptions(value)));
	    }
	    result.add(OptionUtils.joinOptions(nested.toArray(new String[nested.size()])));
	  }
	}
      }
    }

    if (result.size() > 0) {
      if (m_Nesting.empty())
	m_OutputVector.addAll(result);
      else
	((Vector) m_Nesting.peek()).addAll(result);
    }

    return result;
  }

  /**
   * Hook-method before starting visiting options.
   */
  protected void preProduce() {
    super.preProduce();

    m_Output = null;

    m_OutputVector.clear();
    m_OutputVector.add(m_Input.getClass().getName());
  }

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output
   */
  public String[] getOutput() {
    if (m_Output == null)
      m_Output = m_OutputVector.toArray(new String[m_OutputVector.size()]);

    return m_Output;
  }

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output, null in case of an error
   */
  public String toString() {
    return OptionUtils.joinOptions(getOutput());
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    super.cleanUp();

    m_OutputVector = null;
  }

  /**
   * Returns the string array with all the options of the handler, but without
   * the classname of the handler.
   *
   * @param handler	the handler to return the options for
   * @return		the options as flat string array
   */
  public static String[] getOptions(OptionHandler handler) {
    String[]		result;
    String[]		tmp;
    ArrayProducer	producer;

    producer = new ArrayProducer();
    producer.produce(handler);
    tmp    = producer.getOutput();
    result = new String[tmp.length - 1];
    System.arraycopy(tmp, 1, result, 0, tmp.length - 1);

    return result;
  }
}
