/**
 * ClassOption.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.lang.reflect.Array;
import java.lang.reflect.Method;

import adams.core.ClassLocator;
import adams.core.EnumWithCustomDisplay;
import adams.core.Utils;
import adams.gui.goe.Editors;
import adams.gui.goe.EnumEditor;

/**
 * Option class for enums. Enums get automatically registered with the
 * GenericObjectEditor.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2420 $
 */
public class EnumOption
  extends AbstractArgumentOption {

  /** for serialization. */
  private static final long serialVersionUID = -7052702973855610177L;

  /** an instance of an enum with a custom display, used for parsing. */
  protected EnumWithCustomDisplay m_CustomDisplayInstance;

  /**
   * Initializes the option. Will always output the default value.
   *
   * @param owner		the owner of this option
   * @param commandline		the commandline string to identify the option
   * @param property 		the name of bean property
   * @param defValue		the default value, if null then the owner's
   * 				current state is used
   */
  protected EnumOption(OptionManager owner, String commandline, String property,
      Object defValue) {

    super(owner, commandline, property, defValue);
  }

  /**
   * Initializes the option.
   *
   * @param owner		the owner of this option
   * @param commandline		the commandline string to identify the option
   * @param property 		the name of bean property
   * @param defValue		the default value, if null then the owner's
   * 				current state is used
   * @param outputDefValue	whether to output the default value or not
   */
  protected EnumOption(OptionManager owner, String commandline, String property,
      Object defValue, boolean outputDefValue) {

    super(owner, commandline, property, defValue, outputDefValue);

    // register enums automatically with the GOE
    Editors.registerCustomEditor(getBaseClass(), EnumEditor.class);
    m_CustomDisplayInstance = null;
  }

  /**
   * Compares the two values.
   *
   * @param value	the value to compare against the default value
   * @param defValue	the default value to compare against
   * @return		true if both are equal
   */
  protected boolean compareValues(Object value, Object defValue) {
    return toString(value).equals(toString(defValue));
  }

  /**
   * Tries to instantiate an instance of the enumeration type.
   *
   * @return		the instance or null if failed to instantiate
   * @see		#m_CustomDisplayInstance
   */
  protected synchronized EnumWithCustomDisplay getCustomDisplayInstance() {
    Method	method;
    Object	values;

    if (m_CustomDisplayInstance == null) {
      try {
	method                  = getBaseClass().getMethod("values", new Class[0]);
	values                  = method.invoke(null, new Object[0]);
	m_CustomDisplayInstance = (EnumWithCustomDisplay) Array.get(values, 0);
      }
      catch (Exception e) {
	m_CustomDisplayInstance = null;
	e.printStackTrace();
      }
    }

    return m_CustomDisplayInstance;
  }

  /**
   * Turns the string into the appropriate object.
   *
   * @param s		the string to parse
   * @return		the generated object
   * @throws Exception	if parsing of string fails
   */
  public Object valueOf(String s) throws Exception {
    Object			result;
    Class			cl;
    EnumWithCustomDisplay	value;

    result = null;

    if (ClassLocator.hasInterface(EnumWithCustomDisplay.class, getBaseClass())) {
      value = getCustomDisplayInstance();
      if (value != null)
	result = value.parse(s);
    }
    else if (ClassLocator.isSubclass(Enum.class, getBaseClass())) {
      cl     = getBaseClass().asSubclass(Enum.class);
      result = Enum.valueOf(cl, s);
    }

    return result;
  }

  /**
   * Returns a string representation of the specified object.
   *
   * @param obj		the object to turn into a string
   * @return		the string representation
   */
  public String toString(Object obj) {
    String	result;

    result = "";

    if (ClassLocator.hasInterface(EnumWithCustomDisplay.class, getBaseClass()))
      result = ((EnumWithCustomDisplay) obj).toRaw();
    else if (ClassLocator.isSubclass(Enum.class, getBaseClass()))
      result = obj.toString();

    return result;
  }

  /**
   * Adds additional information about the argument, e.g., the class.
   *
   * @param buffer	the buffer to add the information to
   */
  protected void addArgumentInfo(StringBuilder buffer) {
    String	text;
    Method	method;
    Object[]	vals;

    text = "";
    try {
      method = getBaseClass().getMethod("values", new Class[0]);
      vals   = (Object[]) method.invoke(null, new Object[0]);
      text   = Utils.arrayToString(vals).replaceAll(",", "|");
    }
    catch (Exception e) {
      e.printStackTrace();
      text = "Error retrieving enum values";
    }
    buffer.append(" <" + text + ">");
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    super.cleanUp();

    m_CustomDisplayInstance = null;
  }
}
