/*
 * NonlinearExpression.java
 * Copyright (C) 2008-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.data.fit;

import java.util.HashMap;

import adams.core.Utils;
import adams.core.base.BaseString;
import adams.parser.GrammarSupplier;
import adams.parser.MathematicalExpression;

/**
 <!-- globalinfo-start -->
 * Combines the provided expressions in the following form:<br/>
 * <br/>
 *   f(x) = expr0 + expr1 + expr2 + ... + exprN<br/>
 * <br/>
 * Expressions for the partial derivations must be provided as well.Notes:<br/>
 * - 'x' must be upper case in the expressions.<br/>
 * - the coefficients can be used in the expressions as 'AN' with 'N' starting from 0.<br/>
 * <br/>
 * The expressions use the following grammar:<br/>
 * <br/>
 * expr_list ::= expr_list expr_part | expr_part ;<br/>
 * expr_part ::= expr ;<br/>
 * expr      ::=   NUMBER<br/>
 *               | -expr<br/>
 *               | ( expr )<br/>
 *               | constexpr<br/>
 *               | opexpr<br/>
 *               | varexpr<br/>
 *               | funcexpr<br/>
 *               ;<br/>
 * <br/>
 * constexpr ::=   PI<br/>
 *               | E<br/>
 *               ;<br/>
 * <br/>
 * opexpr    ::=   expr + expr<br/>
 *               | expr - expr<br/>
 *               | expr * expr<br/>
 *               | expr &#47; expr<br/>
 *               | expr ^ expr<br/>
 *               ;<br/>
 * <br/>
 * varexpr  ::=  VARIABLE ;<br/>
 * <br/>
 * funcexpr ::=    abs ( expr )<br/>
 *               | sqrt ( expr )<br/>
 *               | log ( expr )<br/>
 *               | exp ( expr )<br/>
 *               | sin ( expr )<br/>
 *               | cos ( expr )<br/>
 *               | tan ( expr )<br/>
 *               | rint ( expr )<br/>
 *               | floor ( expr )<br/>
 *               | pow ( expr , expr )<br/>
 *               | ceil ( expr )<br/>
 *               | ifelse ( boolexpr , expr (if true) , expr (if false) )<br/>
 *               ;<br/>
 * <br/>
 * boolexpr ::=    BOOLEAN<br/>
 *               | true<br/>
 *               | false<br/>
 *               | expr &amp;lt; expr<br/>
 *               | expr &amp;lt;= expr<br/>
 *               | expr &amp;gt; expr<br/>
 *               | expr &amp;gt;= expr<br/>
 *               | expr = expr<br/>
 *               | ( boolexpr )<br/>
 *               | ! boolexpr<br/>
 *               | boolexpr &amp; boolexpr<br/>
 *               | boolexpr | boolexpr<br/>
 *               ;<br/>
 * <br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-expr &lt;adams.core.base.BaseString&gt; [-expr ...] (property: expressions)
 * &nbsp;&nbsp;&nbsp;The expressions that make up the formula: expr0 + expr1 + expre2 + -- +
 * &nbsp;&nbsp;&nbsp;exprN.
 * &nbsp;&nbsp;&nbsp;default: A0*exp(A1*X), A2*exp(A3*X)
 * </pre>
 *
 * <pre>-partial-expr &lt;adams.core.base.BaseString&gt; [-partial-expr ...] (property: partialDerivationExpressions)
 * &nbsp;&nbsp;&nbsp;The partial derivation expressions to use.
 * &nbsp;&nbsp;&nbsp;default: exp(A1*X), A0*exp(A1*X)*X, exp(A3*X), A2*exp(A3*X)*X
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 2537 $
 */
public class NonlinearExpression
  extends NonlinearFunction
  implements GrammarSupplier {

  /** for serialization. */
  private static final long serialVersionUID = 685629825790681289L;

  /** the array with expressions. */
  protected BaseString[] m_Expressions;

  /** the array with the partial derivation expressions. */
  protected BaseString[] m_PartialExpressions;

  /**
   * Returns a short description of the function.
   *
   * @return		a description of the function
   */
  public String globalInfo() {
    return
        "Combines the provided expressions in the following form:\n\n"
      + "  f(x) = expr0 + expr1 + expr2 + ... + exprN\n\n"
      + "Expressions for the partial derivations must be provided as well."
      + "Notes:\n"
      + "- 'x' must be upper case in the expressions.\n"
      + "- the coefficients can be used in the expressions as 'AN' with 'N' starting from 0.\n"
      + "\n"
      + "The expressions use the following grammar:\n\n"
      + getGrammar();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return new MathematicalExpression().getGrammar();
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Expressions = new BaseString[0];
  }

  /**
   * Adds options to the internal list of options. Derived classes must
   * override this method to add additional options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"expr", "expressions",
	new BaseString[]{
	    new BaseString("A0*exp(A1*X)"),
	    new BaseString("A2*exp(A3*X)")});

    m_OptionManager.add(
	"partial-expr", "partialDerivationExpressions",
	new BaseString[]{
	    new BaseString("exp(A1*X)"),
	    new BaseString("A0*exp(A1*X)*X"),
	    new BaseString("exp(A3*X)"),
	    new BaseString("A2*exp(A3*X)*X")});

  }

  /**
   * Sets the expressions to use.
   *
   * @param value 	the expressions
   */
  public void setExpressions(BaseString[] value) {
    m_Expressions = value.clone();
  }

  /**
   * Returns the expressions to use.
   *
   * @return 		the expressions
   */
  public BaseString[] getExpressions() {
    return m_Expressions.clone();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionsTipText() {
    return "The expressions that make up the formula: expr0 + expr1 + expre2 + -- + exprN.";
  }

  /**
   * Sets the partial derivation expressions to use.
   *
   * @param value 	the expressions
   */
  public void setPartialDerivationExpressions(BaseString[] value) {
    m_PartialExpressions = value.clone();
  }

  /**
   * Returns the partial derivation expressions to use.
   *
   * @return 		the expressions
   */
  public BaseString[] getPartialDerivationExpressions() {
    return m_PartialExpressions.clone();
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String partialDerivationExpressionsTipText() {
    return "The partial derivation expressions to use.";
  }

  /**
   * Calculates the partial derivatives based on the x value and the
   * coefficients.
   *
   * @param x		the x value
   * @param a		the coefficents
   * @return		the partial derivatives
   */
  public double[] calcDerivatives(double x, double[] a) {
    double[]	result;
    int		i;
    HashMap	symbols;

    if (a.length != m_PartialExpressions.length)
      throw new IllegalStateException(
	  "Number of coefficients differs from number of partial derivative expressions: "
	  + a.length + " != " + m_PartialExpressions.length);

    result  = new double[m_PartialExpressions.length];
    symbols = new HashMap();
    symbols.put("X", new Double(x));
    for (i = 0; i < a.length; i++)
      symbols.put("A" + i, new Double(a[i]));

    for (i = 0; i < m_PartialExpressions.length; i++) {
      try {
	result[i] = MathematicalExpression.evaluate(m_PartialExpressions[i].stringValue(), symbols);
      }
      catch (Exception e) {
	result[i] = Double.NaN;
	getSystemErr().println("Error evaluating '" + m_PartialExpressions[i] + "':");
	getSystemErr().printStackTrace(e);
      }
    }

    getDebugging().println("x=" + x + " -> " + Utils.arrayToString(a) + " -> dyda=" + Utils.arrayToString(result));

    return result;
  }

  /**
   * Calculates the y value based on the x value and the coefficients.
   *
   * @param x		the x value
   * @param a		the coefficents
   * @return		the y value
   */
  public double calcY(double x, double[] a) {
    double	result;
    int		i;
    HashMap	symbols;

    result  = 0.0;
    symbols = new HashMap();
    symbols.put("X", new Double(x));
    for (i = 0; i < a.length; i++)
      symbols.put("A" + i, new Double(a[i]));

    for (i = 0; i < m_Expressions.length; i++) {
      try {
	result += MathematicalExpression.evaluate(m_Expressions[i].stringValue(), symbols);
      }
      catch (Exception e) {
	a[i]    = Double.NaN;
	result += a[i];
	getSystemErr().println("Error evaluating '" + m_Expressions[i] + "':");
	getSystemErr().printStackTrace(e);
      }
    }

    getDebugging().println("x=" + x + " -> " + Utils.arrayToString(a) + " -> y=" + result);

    return result;
  }
}
