/*
 * Flow.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;

import java.util.ArrayList;
import java.util.List;
import java.util.Vector;

import adams.core.Utils;
import adams.core.Variables;
import adams.core.VariablesHandler;
import adams.core.io.FlowFile;
import adams.db.LogEntry;
import adams.db.MutableLogEntryHandler;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorExecution;
import adams.flow.core.ActorHandlerInfo;
import adams.flow.core.ActorUtils;
import adams.flow.core.InstantiatableActor;

/**
 <!-- globalinfo-start -->
 * Container object for actors, used for executing a flow.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: Flow
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-actor &lt;adams.flow.core.AbstractActor&gt; [-actor ...] (property: actors)
 * &nbsp;&nbsp;&nbsp;All the actors that define this flow.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-error-handling &lt;ACTORS_ALWAYS_STOP_ON_ERROR|ACTORS_DECIDE_TO_STOP_ON_ERROR&gt; (property: errorHandling)
 * &nbsp;&nbsp;&nbsp;Defines how errors are handled that occur during execution of the flow;
 * &nbsp;&nbsp;&nbsp;ACTORS_DECIDE_TO_STOP_ON_ERROR stops the flow only if the actor has the '
 * &nbsp;&nbsp;&nbsp;stopFlowOnError' flag set.
 * &nbsp;&nbsp;&nbsp;default: ACTORS_ALWAYS_STOP_ON_ERROR
 * </pre>
 *
 * <pre>-log-errors (property: logErrors)
 * &nbsp;&nbsp;&nbsp;If set to true, errors are logged and can be retrieved after execution.
 * </pre>
 *
 * <pre>-execute-on-error &lt;adams.core.io.FlowFile&gt; (property: executeOnError)
 * &nbsp;&nbsp;&nbsp;The external flow to execute in case the flow finishes with an error; allows
 * &nbsp;&nbsp;&nbsp;the user to call a clean-up flow.
 * &nbsp;&nbsp;&nbsp;default: ${CWD}
 * </pre>
 *
 * <pre>-execute-on-finish &lt;adams.core.io.FlowFile&gt; (property: executeOnFinish)
 * &nbsp;&nbsp;&nbsp;The external flow to execute in case the flow finishes normal, without any
 * &nbsp;&nbsp;&nbsp;errors.
 * &nbsp;&nbsp;&nbsp;default: ${CWD}
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4020 $
 */
public class Flow
  extends MutableConnectedControlActor
  implements InstantiatableActor, MutableLogEntryHandler, StorageHandler, VariablesHandler {

  /** for serialization. */
  private static final long serialVersionUID = 723059748204261319L;

  /**
   * Enum for the error handling within the flow.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4020 $
   */
  public enum ErrorHandling {
    /** actors always stop on an error. */
    ACTORS_ALWAYS_STOP_ON_ERROR,
    /** actors only stop on errors if the "stopFlowOnError" flag is set. */
    ACTORS_DECIDE_TO_STOP_ON_ERROR
  }

  /** the error handling. */
  protected ErrorHandling m_ErrorHandling;

  /** whether to store errors (as LogEntry records). */
  protected boolean m_LogErrors;

  /** for storing the LogEntry records. */
  protected ArrayList<LogEntry> m_LogEntries;

  /** the storage for temporary data. */
  protected transient Storage m_Storage;

  /** the variables manager. */
  protected Variables m_VariablesManager;

  /** the external flow to execute in case of an abnormal stop. */
  protected FlowFile m_ExecuteOnError;

  /** the external actor to execute in case of an abnormal stop. */
  protected AbstractActor m_ExecuteOnErrorActor;

  /** the external flow to execute in case the flow finishes normal. */
  protected FlowFile m_ExecuteOnFinish;

  /** the external actor to execute in case the flow finishes normal. */
  protected AbstractActor m_ExecuteOnFinishActor;

  /** the actor that was executed after the flow finished. */
  protected AbstractActor m_AfterExecuteActor;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Container object for actors, used for executing a flow.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"error-handling", "errorHandling",
	ErrorHandling.ACTORS_ALWAYS_STOP_ON_ERROR);

    m_OptionManager.add(
	"log-errors", "logErrors",
	false);

    m_OptionManager.add(
	"execute-on-error", "executeOnError",
	new FlowFile("."));

    m_OptionManager.add(
	"execute-on-finish", "executeOnFinish",
	new FlowFile("."));
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_LogEntries           = new ArrayList<LogEntry>();
    m_VariablesManager     = new Variables();
    m_ExecuteOnErrorActor  = null;
    m_ExecuteOnFinishActor = null;
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_AfterExecuteActor = null;
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;
    String	value;

    result = "";

    variable = getOptionManager().getVariableForProperty("executeOnError");
    value    = "";
    if (variable != null)
      value = variable;
    else if (!m_ExecuteOnError.isDirectory())
      value = m_ExecuteOnError.toString();
    if (value.length() > 0)
      result += "on error: " + value;

    variable = getOptionManager().getVariableForProperty("executeOnFinish");
    value    = "";
    if (variable != null)
      value = variable;
    else if (!m_ExecuteOnFinish.isDirectory())
      value = m_ExecuteOnFinish.toString();
    if (value.length() > 0)
      result += "on finish: " + value;

    return result;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String actorsTipText() {
    return "All the actors that define this flow.";
  }

  /**
   * Sets how errors are handled.
   *
   * @param value 	the error handling
   */
  public void setErrorHandling(ErrorHandling value) {
    m_ErrorHandling = value;
    reset();
  }

  /**
   * Returns how errors are handled.
   *
   * @return 		the error handling
   */
  public ErrorHandling getErrorHandling() {
    return m_ErrorHandling;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String errorHandlingTipText() {
    return
        "Defines how errors are handled that occur during execution of the "
      + "flow; " + ErrorHandling.ACTORS_DECIDE_TO_STOP_ON_ERROR + " stops the "
      + "flow only if the actor has the 'stopFlowOnError' flag set.";
  }

  /**
   * Sets whether to log errors.
   *
   * @param value 	true if to log errors
   */
  public void setLogErrors(boolean value) {
    m_LogErrors = value;
    reset();
  }

  /**
   * Returns whether errors are logged.
   *
   * @return 		true if errors are logged
   */
  public boolean getLogErrors() {
    return m_LogErrors;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String logErrorsTipText() {
    return "If set to true, errors are logged and can be retrieved after execution.";
  }

  /**
   * Sets the external flow to execute in case the flow finishes with an error.
   *
   * @param value 	the external flow
   */
  public void setExecuteOnError(FlowFile value) {
    m_ExecuteOnError = value;
    reset();
  }

  /**
   * Returns the external flow to execute in case the flow finishes with an error.
   *
   * @return 		the external flow
   */
  public FlowFile getExecuteOnError() {
    return m_ExecuteOnError;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String executeOnErrorTipText() {
    return
        "The external flow to execute in case the flow finishes with an "
      + "error; allows the user to call a clean-up flow.";
  }

  /**
   * Sets the external flow to execute in case the flow finishes without
   * any errors.
   *
   * @param value 	the external flow
   */
  public void setExecuteOnFinish(FlowFile value) {
    m_ExecuteOnFinish = value;
    reset();
  }

  /**
   * Returns the external flow to execute in case the flow finishes without
   * any errors.
   *
   * @return 		the external flow
   */
  public FlowFile getExecuteOnFinish() {
    return m_ExecuteOnFinish;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String executeOnFinishTipText() {
    return
        "The external flow to execute in case the flow finishes normal, without "
      + "any errors.";
  }

  /**
   * Returns some information about the actor handler, e.g., whether it can
   * contain standalones and the actor execution.
   *
   * @return		the info
   */
  public ActorHandlerInfo getActorHandlerInfo() {
    return new ActorHandlerInfo(true, ActorExecution.SEQUENTIAL, false);
  }

  /**
   * Ignored.
   *
   * @param value	ignored
   */
  public synchronized void setVariables(Variables value) {
  }

  /**
   * Returns the Variables instance to use.
   *
   * @return		the scope handler
   */
  public synchronized Variables getVariables() {
    m_Variables = m_VariablesManager;
    return m_VariablesManager;
  }

  /**
   * Returns the full name of the actor, i.e., the concatenated names of all
   * parents. Used in error messages.
   *
   * @return		the full name
   */
  public String getFullName() {
    m_FullName = null;
    return super.getFullName();
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String		result;
    Vector<String>	errors;

    result = super.setUp();

    getOptionManager().updateVariablesInstance();

    m_LogEntries.clear();

    if (result == null) {
      if (!m_ExecuteOnError.isDirectory() && m_ExecuteOnError.exists()) {
	errors = new Vector<String>();
	m_ExecuteOnErrorActor = ActorUtils.read(m_ExecuteOnError.getAbsolutePath(), errors);
	if (!errors.isEmpty())
	  result = "Error loading execute-on-error actor '" + m_ExecuteOnError.getAbsolutePath() + "':\n" + Utils.flatten(errors, "\n");
	else if (m_ExecuteOnErrorActor == null)
	  result = "Error loading execute-on-error actor '" + m_ExecuteOnError.getAbsolutePath() + "'!";
	else
	  result = m_ExecuteOnErrorActor.setUp();
      }
    }

    if (result == null) {
      if (!m_ExecuteOnFinish.isDirectory() && m_ExecuteOnFinish.exists()) {
	errors = new Vector<String>();
	m_ExecuteOnFinishActor = ActorUtils.read(m_ExecuteOnFinish.getAbsolutePath(), errors);
	if (!errors.isEmpty())
	  result = "Finish loading execute-on-finish actor '" + m_ExecuteOnFinish.getAbsolutePath() + "':\n" + Utils.flatten(errors, "\n");
	else if (m_ExecuteOnFinishActor == null)
	  result = "Finish loading execute-on-finish actor '" + m_ExecuteOnFinish.getAbsolutePath() + "'!";
	else
	  result = m_ExecuteOnFinishActor.setUp();
      }
    }

    return result;
  }

  /**
   * Removes all currently stored LogEntry records.
   */
  public void clearLogEntries() {
    m_LogEntries.clear();
  }

  /**
   * Returns the stored LogEntry records.
   *
   * @return		the stored entries
   */
  public List<LogEntry> getLogEntries() {
    return m_LogEntries;
  }

  /**
   * Adds the LogEntry record to the internal list (in case logging is enabled).
   *
   * @param entry	the record to add
   * @see		#getLogErrors()
   */
  public void addLogEntry(LogEntry entry) {
    if (m_LogErrors)
      m_LogEntries.add(entry);
  }

  /**
   * Returns the specified LogEntry record.
   *
   * @param index	the index of the record to return
   * @return		the requested LogEntry
   */
  public LogEntry getLogEntry(int index) {
    return m_LogEntries.get(index);
  }

  /**
   * Removes the specified LogEntry record from the internal list.
   *
   * @param index	the index of the record to delete
   * @return		the deleted LogEntry
   */
  public LogEntry removeLogEntry(int index) {
    return m_LogEntries.remove(index);
  }

  /**
   * Returns the number of stored LogEntry records.
   *
   * @return		the number of records
   */
  public int countLogEntries() {
    return m_LogEntries.size();
  }

  /**
   * Returns the storage container.
   *
   * @return		the container
   */
  public synchronized Storage getStorage() {
    if (m_Storage == null)
      m_Storage = new Storage();

    return m_Storage;
  }

  /**
   * Executes the actor.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;

    result = super.doExecute();

    // do we execute an actor?
    m_AfterExecuteActor = null;
    if (m_Stopped) {
      if (m_ExecuteOnErrorActor != null)
	m_AfterExecuteActor = m_ExecuteOnErrorActor;
    }
    else if (result == null) {
      if (m_ExecuteOnFinishActor != null)
	m_AfterExecuteActor = m_ExecuteOnFinishActor;
    }
    if (m_AfterExecuteActor != null) {
      m_AfterExecuteActor.execute();
      m_AfterExecuteActor.wrapUp();
    }

    return result;
  }

  /**
   * Cleans up after the execution has finished. Also removes graphical
   * components.
   */
  public void cleanUp() {
    m_LogEntries.clear();
    if (m_Storage != null) {
      m_Storage.clear();
      m_Storage = null;
    }

    if (m_AfterExecuteActor != null) {
      m_AfterExecuteActor.destroy();
      m_AfterExecuteActor = null;
    }

    m_VariablesManager.cleanUp();

    super.cleanUp();
  }

  /**
   * Frees up memory in a "destructive" non-reversible way.
   * <p/>
   * Cleans up the variables.
   *
   * @see 		#m_VariablesManager
   */
  public void destroy() {
    m_VariablesManager.cleanUp();

    super.destroy();
  }
}
