/*
 * AbstractGlobalActor.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.core;

import java.util.HashSet;

/**
 * Abstract ancestor for all actors that access global actors.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3946 $
 */
public abstract class AbstractGlobalActor
  extends AbstractActor {

  /** for serialization. */
  private static final long serialVersionUID = -6652513967046303107L;

  /** the global name. */
  protected GlobalActorReference m_GlobalName;

  /** the global actor. */
  protected AbstractActor m_GlobalActor;

  /** the helper class. */
  protected GlobalActorHelper m_Helper;

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "global", "globalName",
	    new GlobalActorReference("unknown"));
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Helper = new GlobalActorHelper();
  }

  /**
   * Sets the name of the global actor to use.
   *
   * @param value 	the global name
   */
  public void setGlobalName(GlobalActorReference value) {
    m_GlobalName = value;
    reset();
  }

  /**
   * Returns the name of the global actor in use.
   *
   * @return 		the global name
   */
  public GlobalActorReference getGlobalName() {
    return m_GlobalName;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String globalNameTipText() {
    return "The name of the global actor to use.";
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("globalName");

    if (variable != null)
      return variable;
    else if ((m_GlobalName != null) && (m_GlobalName.toString().length() > 0))
      return m_GlobalName.toString();
    else
      return null;
  }

  /**
   * Tries to find the global actor referenced by its global name.
   *
   * @return		the global actor or null if not found
   */
  protected AbstractActor findGlobalActor() {
    return m_Helper.findGlobalActorRecursive(this, getGlobalName());
  }

  /**
   * Checks whether a reference to the global actor is currently available.
   *
   * @return		true if a reference is available
   * @see		#getGlobalActor()
   */
  public boolean hasGlobalActor() {
    return (m_GlobalActor != null);
  }

  /**
   * Returns the currently set global actor.
   *
   * @return		the actor, can be null
   */
  public AbstractActor getGlobalActor() {
    return m_GlobalActor;
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String		result;
    HashSet<String>	variables;

    result = super.setUp();

    if (result == null) {
      m_GlobalActor = findGlobalActor();
      if (m_GlobalActor == null) {
        result = "Couldn't find global actor '" + getGlobalName() + "'!";
      }
      else {
	variables = findVariables(m_GlobalActor);
	m_DetectedVariables.addAll(variables);
	if (m_DetectedVariables.size() > 0)
	  getVariables().addVariableChangeListener(this);
      }
    }

    return result;
  }

  /**
   * Executes the global actor. Derived classes might need to override this
   * method to ensure atomicity.
   *
   * @return		null if no error, otherwise error message
   */
  protected abstract String executeGlobalActor();

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;

    result = null;

    if (!m_GlobalActor.getSkip() && !m_GlobalActor.isStopped()) {
      synchronized(m_GlobalActor) {
	if (isDebugOn())
	  debug("Executing global actor - start: " + m_GlobalActor);
	result = executeGlobalActor();
	if (isDebugOn())
	  debug("Executing global actor - end: " + result);
      }
    }

    return result;
  }

  /**
   * Returns whether the actor has finished.
   *
   * @return		true if finished
   */
  public boolean isFinished() {
    if (m_GlobalActor == null)
      return true;
    else
      return m_GlobalActor.isFinished();
  }

  /**
   * Stops the execution. No message set.
   */
  public void stopExecution() {
    try {
      if (m_GlobalActor != null) {
	  m_GlobalActor.notifyAll();
      }
    }
    catch (Exception e) {
      // ignored
    }

    super.stopExecution();
  }

  /**
   * Cleans up after the execution has finished. Graphical output is left
   * untouched.
   */
  public void wrapUp() {
    if (m_GlobalActor != null) {
      synchronized(m_GlobalActor) {
	m_GlobalActor.wrapUp();
      }
    }

    super.wrapUp();
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void cleanUp() {
    super.cleanUp();

    if (m_GlobalActor != null) {
      m_GlobalActor.cleanUp();
      m_GlobalActor = null;
    }
  }
}