/**
 * UpdateGlobalActorName.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.processor;

import java.lang.reflect.Method;

import adams.core.option.AbstractArgumentOption;
import adams.core.option.BooleanOption;
import adams.core.option.ClassOption;
import adams.core.option.OptionTraverser;
import adams.flow.core.AbstractActor;
import adams.flow.core.GlobalActorReference;

/**
 <!-- globalinfo-start -->
 * Updates all occurrences of the old global actor name with the new one.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-old-name &lt;java.lang.String&gt; (property: oldName)
 * &nbsp;&nbsp;&nbsp;The old global actor name to replace with the new one.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-new-name &lt;java.lang.String&gt; (property: newName)
 * &nbsp;&nbsp;&nbsp;The new global actor name that replaces the old one.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3958 $
 */
public class UpdateGlobalActorName
  extends AbstractModifyingProcessor {

  /** for serialization. */
  private static final long serialVersionUID = 3445744354536414039L;

  /** the old name. */
  protected String m_OldName;

  /** the new name. */
  protected String m_NewName;

  /**
   * Returns a string describing the object.
   *
   * @return 		a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Updates all occurrences of the old global actor name with the new one.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	"old-name", "oldName",
	"");

    m_OptionManager.add(
	"new-name", "newName",
	"");
  }

  /**
   * Sets the old name to replace.
   *
   * @param value 	the old name
   */
  public void setOldName(String value) {
    m_OldName = value;
    reset();
  }

  /**
   * Returns the the old name to replace.
   *
   * @return 		the old name
   */
  public String getOldName() {
    return m_OldName;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String oldNameTipText() {
    return "The old global actor name to replace with the new one.";
  }

  /**
   * Sets the new name that replaces the old one.
   *
   * @param value 	the new name
   */
  public void setNewName(String value) {
    m_NewName = value;
    reset();
  }

  /**
   * Returns the the new name that replaces the old one.
   *
   * @return 		the new name
   */
  public String getNewName() {
    return m_NewName;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String newNameTipText() {
    return "The new global actor name that replaces the old one.";
  }

  /**
   * Performs the actual processing.
   *
   * @param actor	the actor to process
   */
  protected void processActor(AbstractActor actor) {
    actor.getOptionManager().traverse(new OptionTraverser() {
      public void handleClassOption(ClassOption option) {
	// ignored
      }
      public void handleBooleanOption(BooleanOption option) {
	// ignored
      }
      public void handleArgumentOption(AbstractArgumentOption option) {
	if (option.getBaseClass().equals(GlobalActorReference.class)) {
	  GlobalActorReference current = (GlobalActorReference) option.getCurrentValue();
	  if (current.getValue().equals(m_OldName)) {
	    // TODO
	    System.out.println("Updating: " + m_OldName + " -> " + m_NewName);
	    Method method = option.getDescriptor().getWriteMethod();
	    try {
	      method.invoke(option.getOptionHandler(), new GlobalActorReference(m_NewName));
	      m_Modified = true;
	    }
	    catch (Exception e) {
	      getSystemErr().println("Failed to update global actor reference '" + option.getProperty() + "' of '" + option.getOptionHandler().getClass().getName() + "':");
	      getSystemErr().printStackTrace(e);
	    }
	  }
	}
      }
      public boolean canRecurse(Class cls) {
        return true;
      }
    });

    if (m_Modified)
      m_ModifiedActor = actor;
  }
}
