/*
 * SelectFile.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.source;

import java.io.File;

import adams.core.base.BaseString;
import adams.core.io.PlaceholderDirectory;
import adams.core.io.PlaceholderFile;
import adams.flow.core.InteractiveActor;
import adams.gui.chooser.BaseFileChooser;
import adams.gui.core.ExtensionFileFilter;

/**
 <!-- globalinfo-start -->
 * Pops up a file chooser dialog, prompting the user to select one or more files. The files then get forwarded as strings.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SelectFile
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-output-array (property: outputArray)
 * &nbsp;&nbsp;&nbsp;Whether to output the files as array or one-by-one.
 * </pre>
 *
 * <pre>-stop-if-canceled (property: stopFlowIfCanceled)
 * &nbsp;&nbsp;&nbsp;If enabled, the flow gets stopped in case the user cancels the dialog.
 * </pre>
 *
 * <pre>-custom-stop-message &lt;java.lang.String&gt; (property: customStopMessage)
 * &nbsp;&nbsp;&nbsp;The custom stop message to use in case a user cancelation stops the flow
 * &nbsp;&nbsp;&nbsp;(default is the full name of the actor)
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-file-chooser-title &lt;java.lang.String&gt; (property: fileChooserTitle)
 * &nbsp;&nbsp;&nbsp;The title for the file chooser dialog.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-initial-dir &lt;adams.core.io.PlaceholderDirectory&gt; (property: initialDirectory)
 * &nbsp;&nbsp;&nbsp;The initial directory for the file chooser.
 * &nbsp;&nbsp;&nbsp;default: ${CWD}
 * </pre>
 *
 * <pre>-extension &lt;adams.core.base.BaseString&gt; [-extension ...] (property: extensions)
 * &nbsp;&nbsp;&nbsp;The extensions available through the file chooser.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-absolute (property: absoluteFileNames)
 * &nbsp;&nbsp;&nbsp;If enabled, absolute file names instead of relative ones are output.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3695 $
 */
public class SelectFile
  extends AbstractArrayProvider
  implements InteractiveActor {

  /** for serialization. */
  private static final long serialVersionUID = 8200691218381875131L;

  /** the title of the file chooser dialog. */
  protected String m_FileChooserTitle;

  /** the initial directory. */
  protected PlaceholderDirectory m_InitialDirectory;

  /** the extensions to offer in the file chooser. */
  protected BaseString[] m_Extensions;

  /** whether to use absolute file/dir names. */
  protected boolean m_AbsoluteFileNames;

  /** whether to stop the flow if canceled. */
  protected boolean m_StopFlowIfCanceled;

  /** the custom stop message to use if flow gets stopped due to cancelation. */
  protected String m_CustomStopMessage;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Pops up a file chooser dialog, prompting the user to select one or "
      + "more files. The files then get forwarded as strings.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "stop-if-canceled", "stopFlowIfCanceled",
	    false);

    m_OptionManager.add(
	    "custom-stop-message", "customStopMessage",
	    "");

    m_OptionManager.add(
	    "file-chooser-title", "fileChooserTitle",
	    "");

    m_OptionManager.add(
	    "initial-dir", "initialDirectory",
	    new PlaceholderDirectory("."));

    m_OptionManager.add(
	    "extension", "extensions",
	    new BaseString[0]);

    m_OptionManager.add(
	    "absolute", "absoluteFileNames",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    result = "directory: ";
    variable = getOptionManager().getVariableForProperty("initialDirectory");
    if (variable != null)
      result += variable;
    else
      result += m_InitialDirectory.toString();

    if (m_AbsoluteFileNames)
      result += " (absolute file names)";

    if (m_StopFlowIfCanceled)
      result += " [stops flow if canceled]";

    return result;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputArrayTipText() {
    return "Whether to output the files as array or one-by-one.";
  }

  /**
   * Sets the title for the file chooser dialog.
   *
   * @param value	the title
   */
  public void setFileChooserTitle(String value) {
    m_FileChooserTitle = value;
    reset();
  }

  /**
   * Returns the title for the file chooser dialog.
   *
   * @return 		the title
   */
  public String getFileChooserTitle() {
    return m_FileChooserTitle;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String fileChooserTitleTipText() {
    return "The title for the file chooser dialog.";
  }

  /**
   * Sets the initial directory.
   *
   * @param value	the initial directory
   */
  public void setInitialDirectory(PlaceholderDirectory value) {
    m_InitialDirectory = value;
    reset();
  }

  /**
   * Returns the initial directory.
   *
   * @return 		the initial directory
   */
  public PlaceholderDirectory getInitialDirectory() {
    return m_InitialDirectory;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String initialDirectoryTipText() {
    return "The initial directory for the file chooser.";
  }

  /**
   * Sets the extensions to offer in the file chooser.
   *
   * @param value	the extensions
   */
  public void setExtensions(BaseString[] value) {
    m_Extensions = value;
    reset();
  }

  /**
   * Returns the extension on offer in the file chooser.
   *
   * @return 		the extensions
   */
  public BaseString[] getExtensions() {
    return m_Extensions;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String extensionsTipText() {
    return "The extensions available through the file chooser.";
  }

  /**
   * Sets whether to output absolute file names or not.
   *
   * @param value	if true absolute file names are output
   */
  public void setAbsoluteFileNames(boolean value) {
    m_AbsoluteFileNames = value;
    reset();
  }

  /**
   * Returns whether to output absolute file names or not.
   *
   * @return 		true if absolute files are output
   */
  public boolean getAbsoluteFileNames() {
    return m_AbsoluteFileNames;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String absoluteFileNamesTipText() {
    return "If enabled, absolute file names instead of relative ones are output.";
  }

  /**
   * Sets whether to stop the flow if dialog canceled.
   *
   * @param value	if true flow gets stopped if dialog canceled
   */
  public void setStopFlowIfCanceled(boolean value) {
    m_StopFlowIfCanceled = value;
    reset();
  }

  /**
   * Returns whether to stop the flow if dialog canceled.
   *
   * @return 		true if the flow gets stopped if dialog canceled
   */
  public boolean getStopFlowIfCanceled() {
    return m_StopFlowIfCanceled;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String stopFlowIfCanceledTipText() {
    return "If enabled, the flow gets stopped in case the user cancels the dialog.";
  }

  /**
   * Sets the custom message to use when stopping the flow.
   *
   * @param 		the stop message
   */
  public void setCustomStopMessage(String value) {
    m_CustomStopMessage = value;
    reset();
  }

  /**
   * Returns the custom message to use when stopping the flow.
   *
   * @return		the stop message
   */
  public String getCustomStopMessage() {
    return m_CustomStopMessage;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return		tip text for this property suitable for
   *             	displaying in the GUI or for listing the options.
   */
  public String customStopMessageTipText() {
    return
        "The custom stop message to use in case a user cancelation stops the "
      + "flow (default is the full name of the actor)";
  }

  /**
   * Returns the base class of the items.
   *
   * @return		the class
   */
  protected Class getItemClass() {
    return String.class;
  }

  /**
   * Performs the interaction with the user.
   *
   * @return		true if successfully interacted
   */
  public boolean doInteract() {
    boolean		result;
    int			retVal;
    File[]		files;
    BaseFileChooser	fileChooser;
    ExtensionFileFilter	filter;

    result = false;

    m_Queue.clear();

    fileChooser = new BaseFileChooser();
    fileChooser.resetChoosableFileFilters();
    for (BaseString ext: m_Extensions) {
      filter = new ExtensionFileFilter(ext.getValue().toUpperCase() + " files", ext.getValue());
      fileChooser.addChoosableFileFilter(filter);
    }
    if (m_FileChooserTitle.length() > 0)
      fileChooser.setDialogTitle(m_FileChooserTitle);
    fileChooser.setCurrentDirectory(m_InitialDirectory);
    fileChooser.setFileSelectionMode(BaseFileChooser.FILES_ONLY);
    fileChooser.setAcceptAllFileFilterUsed(true);
    fileChooser.setMultiSelectionEnabled(true);
    retVal = fileChooser.showOpenDialog(null);
    if (retVal == BaseFileChooser.APPROVE_OPTION) {
      result = true;
      files  = fileChooser.getSelectedFiles();
      for (File file: files) {
	if (m_AbsoluteFileNames)
	  m_Queue.add(file.getAbsolutePath());
	else
	  m_Queue.add(new PlaceholderFile(file).toString());
      }
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    if (!isHeadless()) {
      if (!doInteract()) {
	if (m_StopFlowIfCanceled) {
	  if ((m_CustomStopMessage == null) || (m_CustomStopMessage.trim().length() == 0))
	    stopExecution("Flow canceled: " + getFullName());
	  else
	    stopExecution(m_CustomStopMessage);
	}
      }
    }

    return m_StopMessage;
  }
}
