/*
 * GlobalActorReferenceEditor.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.gui.goe;

import java.awt.BorderLayout;
import java.awt.FlowLayout;
import java.awt.FontMetrics;
import java.awt.Graphics;
import java.awt.Rectangle;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyAdapter;
import java.awt.event.KeyEvent;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JTextField;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;
import javax.swing.tree.TreePath;

import adams.core.Utils;
import adams.core.option.AbstractOption;
import adams.flow.core.GlobalActorReference;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.BaseTreeNode;
import adams.gui.core.MouseUtils;
import adams.gui.flow.tree.Node;
import adams.gui.goe.globalactorstree.GlobalActorsNode;
import adams.gui.goe.globalactorstree.GlobalActorsTree;

/**
 * A PropertyEditor for GlobalActorReference objects.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3584 $
 */
public class GlobalActorReferenceEditor
  extends AbstractPropertyEditorSupport
  implements CustomStringRepresentationHandler {

  /** The text field with the value. */
  protected JTextField m_TextValue;

  /** The tree displaying all the global actors. */
  protected GlobalActorsTree m_Tree;

  /**
   * Returns the Compound as string.
   *
   * @param option	the current option
   * @param object	the Compound object to convert
   * @return		the generated string
   */
  public static String toString(AbstractOption option, Object object) {
    return ((GlobalActorReference) object).getValue();
  }

  /**
   * Returns a Compound generated from the string.
   *
   * @param option	the current option
   * @param str		the string to convert to a Compound
   * @return		the generated Compound
   */
  public static Object valueOf(AbstractOption option, String str) {
    return new GlobalActorReference(str);
  }

  /**
   * Returns a custom string representation of the object.
   *
   * @param obj		the object to turn into a string
   * @return		the string representation
   */
  public String toCustomStringRepresentation(Object obj) {
    return toString(null, obj);
  }

  /**
   * Returns an object created from the custom string representation.
   *
   * @param str		the string to turn into an object
   * @return		the object
   */
  public Object fromCustomStringRepresentation(String str) {
    return new GlobalActorReference(str);
  }

  /**
   * Returns a representation of the current property value as java source.
   *
   * @return 		a value of type 'String'
   */
  public String getJavaInitializationString() {
    String	result;

    result = "new " + GlobalActorReference.class.getName() + "(\"" + Utils.backQuoteChars(getValue().toString()) + "\")";

    return result;
  }

  /**
   * Returns the string to paint.
   *
   * @return		the string
   * @see		#paintValue(Graphics, Rectangle)
   */
  protected String getStringToPaint() {
    return "" + getValue();
  }

  /**
   * Paints a representation of the current Object.
   *
   * @param gfx 	the graphics context to use
   * @param box 	the area we are allowed to paint into
   * @see		#getStringToPaint()
   */
  public void paintValue(Graphics gfx, Rectangle box) {
    FontMetrics 	fm;
    int 		vpad;
    String 		val;

    fm   = gfx.getFontMetrics();
    vpad = (box.height - fm.getHeight()) / 2;
    val  = getStringToPaint();
    gfx.drawString(val, 2, fm.getHeight() + vpad);
  }

  /**
   * Parses the given string and returns the generated object. The string
   * has to be a valid one, i.e., the isValid(String) check has been
   * performed already and succeeded.
   *
   * @param s		the string to parse
   * @return		the generated object, or null in case of an error
   */
  protected GlobalActorReference parse(String s) {
    GlobalActorReference	result;

    try {
      result = new GlobalActorReference();
      result.setValue(s);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Updates the value of the text field if possble, using the last component
   * on the given tree path.
   *
   * @param path	the path to use for updating the value
   * @return		true if the text field got updated
   */
  protected boolean updateValue(TreePath path) {
    boolean		result;
    BaseTreeNode 	node;
    GlobalActorsNode 	global;

    result = false;

    if (path != null) {
      node = (BaseTreeNode) path.getLastPathComponent();
      if (node instanceof GlobalActorsNode) {
	global = (GlobalActorsNode) node;
	if (global.isGlobalActor()) {
	  m_TextValue.setText(global.getLabel());
	  result = true;
	}
      }
    }

    return result;
  }

  /**
   * Gets the custom editor component.
   *
   * @return 		the editor
   */
  protected JComponent createCustomEditor() {
    JPanel	panelTree;
    JPanel	panelAll;
    JPanel	panel;
    JLabel	label;
    JPanel 	panelButtons;
    JButton 	buttonOK;
    JButton 	buttonClose;

    panelTree = new JPanel(new BorderLayout(0, 5));
    panelTree.setBorder(BorderFactory.createEmptyBorder(5, 0, 0, 0));
    m_Tree    = new GlobalActorsTree();
    m_Tree.getSelectionModel().addTreeSelectionListener(new TreeSelectionListener() {
      public void valueChanged(TreeSelectionEvent e) {
	updateValue(e.getPath());
      }
    });
    m_Tree.addMouseListener(new MouseAdapter() {
      public void mousePressed(MouseEvent e) {
        final TreePath selPath = m_Tree.getPathForLocation(e.getX(), e.getY());
        if (MouseUtils.isDoubleClick(e)) {
          if (updateValue(selPath)) {
            e.consume();
            acceptInput();
          }
        }
        if (!e.isConsumed())
          super.mousePressed(e);
      }
    });
    panelTree.add(new BaseScrollPane(m_Tree), BorderLayout.CENTER);

    panelTree.add(new JLabel("Select global actor:"), BorderLayout.NORTH);

    panelAll = new JPanel(new BorderLayout());
    panelTree.add(panelAll, BorderLayout.SOUTH);
    panel    = new JPanel(new FlowLayout(FlowLayout.LEFT));
    panelAll.add(panel, BorderLayout.CENTER);

    m_TextValue = new JTextField(20);
    m_TextValue.addKeyListener(new KeyAdapter() {
      public void keyPressed(KeyEvent e) {
	if (e.getKeyCode() == KeyEvent.VK_ENTER) {
	  e.consume();
	  acceptInput();
	}
	else if (e.getKeyCode() == KeyEvent.VK_ESCAPE) {
	  e.consume();
	  discardInput();
	}
	else {
	  super.keyPressed(e);
	}
      }
    });
    label = new JLabel("Manual reference");
    label.setDisplayedMnemonic('M');
    label.setLabelFor(m_TextValue);
    panel.add(label);
    panel.add(m_TextValue);

    panelButtons = new JPanel(new FlowLayout(FlowLayout.RIGHT));
    panelAll.add(panelButtons, BorderLayout.SOUTH);

    buttonOK = new JButton("OK");
    buttonOK.setMnemonic('O');
    buttonOK.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	acceptInput();
      }
    });
    panelButtons.add(buttonOK);

    buttonClose = new JButton("Cancel");
    buttonClose.setMnemonic('C');
    buttonClose.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	discardInput();
      }
    });
    panelButtons.add(buttonClose);

    return panelTree;
  }

  /**
   * Checks whether the string is valid.
   *
   * @param s		the string to check
   * @return		true if the string is valid
   */
  protected boolean isValid(String s) {
    return ((GlobalActorReference) getValue()).isValid(s);
  }

  /**
   * Checks whether the string is the same as the currently used one.
   *
   * @param s		the string to check
   * @return		true if the strings are the same
   */
  protected boolean isUnchanged(String s) {
    return s.equals(((GlobalActorReference) getValue()).getValue());
  }

  /**
   * Accepts the input and closes the dialog.
   */
  protected void acceptInput() {
    String 	s;

    s = m_TextValue.getText();
    if (isValid(s) && !isUnchanged(s))
      setValue(parse(s));
    closeDialog();
  }

  /**
   * Discards the input and closes the dialog.
   */
  protected void discardInput() {
    closeDialog();
  }

  /**
   * Locates all the global actors for the node.
   *
   * @param node	the node to start the search from
   * @return		the located global actors (including GlobalActors nodes)
   */
  protected Vector<String> findGlobalActors() {
    Vector<String>	result;
    Vector<Node>	globals;
    int			i;
    Node		child;

    result = new Vector<String>();

    globals = FlowHelper.findGlobalActors(m_CustomEditor);
    for (Node global: globals) {
      result.add(global.getFullName());
      for (i = 0; i < global.getChildCount(); i++) {
	child = (Node) global.getChildAt(i);
	result.add(child.getFullName());
      }
    }

    return result;
  }

  /**
   * Initializes the display of the value.
   */
  protected void initForDisplay() {
    if (!m_TextValue.getText().equals("" + getValue()))
      m_TextValue.setText("" + getValue());
    m_TextValue.setToolTipText(((GlobalActorReference) getValue()).getTipText());

    m_Tree.setTree(FlowHelper.getTree(m_CustomEditor));
    m_Tree.setItems(findGlobalActors());
    m_Tree.expandAll();
    m_Tree.selectGlobalActor(m_TextValue.getText());

    m_TextValue.grabFocus();
  }
}
