/*
 * GlobalActorsTree.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.goe.globalactorstree;

import java.util.Enumeration;

import javax.swing.tree.TreeCellRenderer;
import javax.swing.tree.TreePath;

import adams.gui.core.BaseTreeNode;
import adams.gui.core.dotnotationtree.DotNotationTree;
import adams.gui.flow.tree.Node;
import adams.gui.flow.tree.Tree;

/**
 * Displays classes in a tree structure.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3584 $
 */
public class GlobalActorsTree
  extends DotNotationTree<GlobalActorsNode> {

  /** for serialization. */
  private static final long serialVersionUID = 1489354474021395304L;

  /** the underlying flow. */
  protected Tree m_FlowTree;

  /**
   * Initializes the tree with no classes.
   */
  public GlobalActorsTree() {
    super();
    setSorted(true);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_FlowTree = null;
  }

  /**
   * Sets the underlying flow.
   *
   * @param value	the flow
   */
  public void setTree(Tree value) {
    m_FlowTree = value;
  }

  /**
   * Returns the underlying flow.
   *
   * @return		the flow
   */
  public Tree getFlow() {
    return m_FlowTree;
  }

  /**
   * Returns the default renderer to use.
   *
   * @return		the renderer
   */
  protected TreeCellRenderer getDefaultRenderer() {
    return new GlobalActorsTreeRenderer();
  }

  /**
   * Returns the label used for the root node.
   *
   * @return		the label
   */
  protected String getRootNodeLabel() {
    if ((m_FlowTree != null) && (m_FlowTree.getModel().getRoot() != null))
      return ((Node) m_FlowTree.getModel().getRoot()).getActor().getName();
    else
      return "Flow";
  }

  /**
   * Creates a new node with the specified label.
   *
   * @param label	the label to use for the node
   * @return		the new node
   */
  protected GlobalActorsNode newNode(String label) {
    GlobalActorsNode	result;
    String		classname;
    Node		global;

    classname = null;
    if (m_FlowTree != null) {
      global = m_FlowTree.locate(label);
      if (global != null)
	classname = global.getActor().getClass().getName();
    }

    result = new GlobalActorsNode(label);
    result.setClassname(classname);

    return result;
  }

  /**
   * Post-processes a leaf after being added, i.e., info node generators
   * are applied.
   *
   * @param node	the node to process
   * @param item	the full item string
   */
  protected void postAddLeaf(GlobalActorsNode node, String item) {
    String		classname;
    Node		global;

    classname = null;
    if (m_FlowTree != null) {
      global = m_FlowTree.locate(item);
      if (global != null)
	classname = global.getActor().getClass().getName();
      node.setClassname(classname);
    }

    super.postAddLeaf(node, item);
  }

  /**
   * Selects the global actor identified by the name.
   *
   * @param name	the name of the global actor
   */
  public GlobalActorsNode findGlobalActor(String name) {
    GlobalActorsNode	result;
    BaseTreeNode	node;
    Enumeration		enm;
    GlobalActorsNode	child;

    result = null;

    if ((name != null) && (name.length() > 0)) {
      node = (BaseTreeNode) getTreeModel().getRoot();
      if (node != null) {
	enm = node.breadthFirstEnumeration();
	while (enm.hasMoreElements()) {
	  node = (BaseTreeNode) enm.nextElement();
	  if (node instanceof GlobalActorsNode) {
	    child = (GlobalActorsNode) node;
	    if (child.isGlobalActor() && child.getLabel().equals(name)) {
	      result = child;
	      break;
	    }
	  }
	}
      }
    }

    return result;
  }

  /**
   * Selects the global actor identified by the name.
   *
   * @param name	the name of the global actor
   */
  public void selectGlobalActor(String name) {
    GlobalActorsNode	node;
    TreePath		path;

    node = findGlobalActor(name);
    if (node != null) {
      path = new TreePath(node.getPath());
      setSelectionPath(path);
    }
  }
}
