/*
 * AbstractColorProvider.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.core;

import java.awt.Color;
import java.util.Vector;

import adams.core.ShallowCopySupporter;
import adams.core.option.OptionHandlingObject;
import adams.core.option.OptionUtils;

/**
 * A class for providing colors.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3932 $
 */
public abstract class AbstractColorProvider
  extends OptionHandlingObject
  implements ShallowCopySupporter<AbstractColorProvider> {

  /** for serialization. */
  private static final long serialVersionUID = 1159553726314921425L;

  /** the minimum value for R, G and B before restarting the color provider. */
  public final static int MIN_VALUE = 64;

  /** the current index. */
  protected int m_Index;

  /** whether it is the first iteration. */
  protected boolean m_FirstIteration;

  /** contains the current colors. */
  protected Vector<Color> m_Colors;

  /** contains the default colors. */
  protected Vector<Color> m_DefaultColors;

  /** recycled, i.e., returned colors. */
  protected Vector<Color> m_RecycledColors;

  /** excluded colors (already used). */
  protected Vector<Color> m_ExcludedColors;

  /** whether darkening of colors is allowed. */
  protected boolean m_AllowDarkening;

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_DefaultColors  = new Vector<Color>();
    m_Colors         = new Vector<Color>();
    m_RecycledColors = new Vector<Color>();
    m_ExcludedColors = new Vector<Color>();
    m_AllowDarkening = true;
  }

  /**
   * Finishes the initialization.
   */
  protected void finishInit() {
    super.finishInit();

    resetColors();
  }

  /**
   * Returns the next color.
   *
   * @return		the next color
   */
  public Color next() {
    Color	result;

    if (m_DefaultColors.size() == 0)
      throw new IllegalStateException("No more colors left!");

    result = null;

    while (result == null) {
      if (m_RecycledColors.size() > 0) {
	result = m_RecycledColors.firstElement();
	m_RecycledColors.remove(0);
      }
      else {
	result = m_Colors.get(m_Index);

	// create next color
	if (!m_FirstIteration) {
	  result = result.darker();
	  m_Colors.set(m_Index, result);
	}

	// increment index
	m_Index++;
	if (m_Index == m_Colors.size()) {
	  m_Index = 0;
	  if (m_AllowDarkening)
	    m_FirstIteration = false;
	}
      }

      if (m_ExcludedColors.contains(result))
	result = null;
    }

    // too dark? -> restart
    if (    (result.getRed() < MIN_VALUE)
	 && (result.getGreen() < MIN_VALUE)
	 && (result.getBlue() < MIN_VALUE) ) {
      resetColors();
      result = m_Colors.get(m_Index);
    }

    if (isDebugOn())
      debug("next color: " + result);

    return result;
  }

  /**
   * Resets the colors.
   */
  public void resetColors() {
    m_Index          = 0;
    m_FirstIteration = true;

    m_Colors.clear();
    m_Colors.addAll(m_DefaultColors);
    m_RecycledColors.clear();
    m_ExcludedColors.clear();

    if (isDebugOn())
      debug("reset of colors");
  }

  /**
   * "Recycles" the specified colors, i.e., makes it available for future use.
   *
   * @param c		the color to re-use
   */
  public void recycle(Color c) {
    m_RecycledColors.add(c);
  }

  /**
   * "Excludes" the specified colors, i.e., makes it unavailable for future use.
   *
   * @param c		the color to exclude
   */
  public void exclude(Color c) {
    m_ExcludedColors.add(c);
  }

  /**
   * Returns a shallow copy of itself.
   *
   * @return		the shallow copy
   */
  public AbstractColorProvider shallowCopy() {
    return shallowCopy(false);
  }

  /**
   * Returns a shallow copy of itself.
   *
   * @param expand	whether to expand variables to their current values
   * @return		the shallow copy
   */
  public AbstractColorProvider shallowCopy(boolean expand) {
    return (AbstractColorProvider) OptionUtils.shallowCopy(this, expand);
  }
}
