/**
 * PausableThreadPoolExecutor.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 * Copyright (C) Sun/Oracle (based on example from Javadoc: java.util.concurrent.ThreadPoolExecutor)
 */
package adams.multiprocess;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.ReentrantLock;

import adams.core.Pausable;
import adams.core.annotation.MixedCopyright;

/**
 * The thread pool executor for the JobRunner class.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 3678 $
 * @see JobRunner
 */
@MixedCopyright
public class PausableFixedThreadPoolExecutor
  extends ThreadPoolExecutor
  implements Pausable {

  /** whether the executor is paused. */
  protected boolean m_IsPaused;

  /** for pausing. */
  protected ReentrantLock m_PauseLock = new ReentrantLock();

  /** for resuming. */
  protected Condition m_Unpaused = m_PauseLock.newCondition();

  /**
   * Initializes the thread pool.
   *
   * @param numThreads	the maximum number of threads to use
   */
  public PausableFixedThreadPoolExecutor(int numThreads) {
    super(numThreads, numThreads,
        0L, TimeUnit.MILLISECONDS,
        new LinkedBlockingQueue<Runnable>());
  }

  protected void beforeExecute(Thread t, Runnable r) {
    super.beforeExecute(t, r);
    m_PauseLock.lock();
    try {
      while (m_IsPaused)
	m_Unpaused.await();
    }
    catch (InterruptedException ie) {
      t.interrupt();
    }
    finally {
      m_PauseLock.unlock();
    }
  }

  /**
   * Returns whether the object is currently paused.
   *
   * @return		true if object is paused
   */
  public boolean isPaused() {
    return m_IsPaused;
  }

  /**
   * Pauses the execution.
   */
  public void pauseExecution() {
    m_PauseLock.lock();
    try {
      m_IsPaused = true;
    }
    finally {
      m_PauseLock.unlock();
    }
  }

  /**
   * Resumes the execution.
   */
  public void resumeExecution() {
    m_PauseLock.lock();
    try {
      m_IsPaused = false;
      m_Unpaused.signalAll();
    }
    finally {
      m_PauseLock.unlock();
    }
  }
}