/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FlowRunner.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow;

import java.lang.reflect.Method;
import java.util.Vector;

import adams.core.Pausable;
import adams.core.Stoppable;
import adams.core.Utils;
import adams.core.VariablesHandler;
import adams.core.io.PlaceholderFile;
import adams.core.management.ProcessUtils;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.ArrayConsumer;
import adams.core.option.OptionUtils;
import adams.db.AbstractDatabaseObjectWithOptionHandling;
import adams.env.Environment;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorUtils;

/**
 <!-- globalinfo-start -->
 * Executes flows from command-line.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-driver &lt;java.lang.String&gt; (property: driver)
 * &nbsp;&nbsp;&nbsp;The JDBC driver.
 * &nbsp;&nbsp;&nbsp;default: com.mysql.jdbc.Driver
 * </pre>
 *
 * <pre>-url &lt;java.lang.String&gt; (property: URL)
 * &nbsp;&nbsp;&nbsp;The database URL.
 * </pre>
 *
 * <pre>-user &lt;java.lang.String&gt; (property: user)
 * &nbsp;&nbsp;&nbsp;The database user.
 * </pre>
 *
 * <pre>-password &lt;java.lang.String&gt; (property: password)
 * &nbsp;&nbsp;&nbsp;The password of the database user.
 * </pre>
 *
 * <pre>-headless (property: headless)
 * &nbsp;&nbsp;&nbsp;If set to true, the actor is run in headless mode without GUI components.
 * </pre>
 *
 * <pre>-file &lt;adams.core.io.PlaceholderFile&gt; (property: file)
 * &nbsp;&nbsp;&nbsp;The file to load the actor from.
 * &nbsp;&nbsp;&nbsp;default: .
 * </pre>
 *
 * <pre>-actor &lt;adams.flow.core.AbstractActor [options]&gt; (property: actor)
 * &nbsp;&nbsp;&nbsp;The actor to execute (if not loaded from file).
 * &nbsp;&nbsp;&nbsp;default: adams.flow.control.Flow
 * </pre>
 *
 * <pre>-local-placeholder &lt;adams.core.base.BaseString&gt; [-local-placeholder ...] (property: localPlaceholders)
 * &nbsp;&nbsp;&nbsp;The local placeholders of the flow (format: key=value).
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-clean-up (property: cleanUp)
 * &nbsp;&nbsp;&nbsp;If set to true, then a clean up is performed after execution, removing any
 * &nbsp;&nbsp;&nbsp;graphical output as well.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class FlowRunner
  extends AbstractDatabaseObjectWithOptionHandling
  implements Stoppable, Pausable {

  /** for serialization. */
  private static final long serialVersionUID = 5693250462014974198L;

  /** the method for stopping all engines. */
  public final static String METHOD_STOPALLENGINES = "stopAllEngines";

  /** the flow file to execute. */
  protected PlaceholderFile m_File;

  /** the actor to execute. */
  protected AbstractActor m_Actor;

  /** the actual actor to execute. */
  protected AbstractActor m_ActualActor;

  /** the last actor that was executed. */
  protected AbstractActor m_LastActor;

  /** whether the execution is to be headless, i.e., no GUI components. */
  protected boolean m_Headless;

  /** the directory to use as the project's home directory. */
  protected String m_Home;

  /** whether to clean up after execution, i.e., removing graphical output
   * automatically. */
  protected boolean m_CleanUp;

  /** whether the flow was interrupted by the user. */
  protected boolean m_InterruptedByUser;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Executes flows from command-line.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "home", "home",
	    "");

    m_OptionManager.add(
	    "headless", "headless",
	    false);

    m_OptionManager.add(
	    "file", "file",
	    new PlaceholderFile("."));

    m_OptionManager.add(
	    "actor", "actor",
	    new adams.flow.control.Flow());

    m_OptionManager.add(
	    "clean-up", "cleanUp",
	    false);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_ActualActor = null;
    m_LastActor   = null;
    m_dbc         = null;
  }

  /**
   * Overrides the automatic detection of the project's home directory and uses
   * the specified directory instead. No placeholders allowed, should be
   * absolute.
   *
   * @param value	the directory to use
   */
  public void setHome(String value) {
    m_Home = value;
    reset();
  }

  /**
   * Returns the directory to use as home directory instead of the automatically
   * determined one.
   *
   * @return		the directory to use
   */
  public String getHome() {
    return m_Home;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String homeTipText() {
    return "The directory to use as the project's home directory, overriding the automatically determined one.";
  }

  /**
   * Sets whether the actor is to be run in headless mode, i.e., suppressing
   * GUI components.
   *
   * @param value	if true then GUI components will be suppressed
   */
  public void setHeadless(boolean value) {
    m_Headless = value;
  }

  /**
   * Returns whether the actor is run in headless mode.
   *
   * @return		true if GUI components are suppressed
   */
  public boolean isHeadless() {
    return m_Headless;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String headlessTipText() {
    return "If set to true, the actor is run in headless mode without GUI components.";
  }

  /**
   * Sets the file to load the actor from.
   *
   * @param value 	the file
   */
  public void setFile(PlaceholderFile value) {
    m_File = value;
    reset();
  }

  /**
   * Returns the file to load the actor from.
   *
   * @return 		the file
   */
  public PlaceholderFile getFile() {
    return m_File;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String fileTipText() {
    return "The file to load the actor from.";
  }

  /**
   * Sets the actor to use (instead of loading it from file).
   *
   * @param value 	the actor
   */
  public void setActor(AbstractActor value) {
    m_Actor = value;
    reset();
  }

  /**
   * Returns the actor to execute, if no file specified.
   *
   * @return 		the file
   */
  public AbstractActor getActor() {
    return m_Actor;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String actorTipText() {
    return "The actor to execute (if not loaded from file).";
  }

  /**
   * Sets whether to clean up after execution, i.e., removing graphical
   * output.
   *
   * @param value	if true then a clean up is performed after execution
   */
  public void setCleanUp(boolean value) {
    m_CleanUp = value;
  }

  /**
   * Returns whether to perform a clean up after the execution and remove
   * graphical output.
   *
   * @return		true if clean up is performed after execution
   */
  public boolean isCleanUp() {
    return m_CleanUp;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String cleanUpTipText() {
    return
        "If set to true, then a clean up is performed after execution, "
      + "removing any graphical output as well.";
  }

  /**
   * Returns the instance of the last actor that was executed.
   *
   * @return		the actor or null if no actor has been run yet
   */
  public AbstractActor getLastActor() {
    return m_LastActor;
  }

  /**
   * Executes the actor if possible.
   *
   * @return		the error if one occurred, otherwise null (= everything OK)
   */
  public String execute() {
    String		result;
    Vector<String>	errors;

    result              = null;
    m_InterruptedByUser = false;

    if (isDebugOn())
      debug("PID: " + ProcessUtils.getVirtualMachinePID());

    // connect
    establishDatabaseConnection();

    // clean up last run
    if (m_LastActor != null)
      m_LastActor.destroy();

    // file has precedence over actor
    if (!m_File.isDirectory()) {
      errors = new Vector<String>();
      m_ActualActor = ActorUtils.read(m_File.getAbsolutePath(), errors);
      if (!errors.isEmpty()) {
	result = "Failed to load actor from '" + m_File + "'!\n" + Utils.flatten(errors, "\n");
	return result;
      }
      if (m_ActualActor instanceof VariablesHandler)
	ActorUtils.updateVariablesWithFlowFilename((VariablesHandler) m_ActualActor, m_File);
    }
    else {
      m_ActualActor = m_Actor.shallowCopy(true);
    }
    m_LastActor = m_ActualActor;
    if (getDebugLevel() > 1)
      debug("Actor command-line: " + m_ActualActor.toCommandLine(), 2);

    try {
      // initialize actor
      m_ActualActor.setHeadless(isHeadless());
      if (isDebugOn() && m_ActualActor.isHeadless())
	debug("Running in headless mode");

      result = m_ActualActor.setUp();
      if (isDebugOn())
	debug("setUp() result: " + result);

      // execute actor
      if (result == null) {
	result = m_ActualActor.execute();
	if (isDebugOn())
	  debug("execute() result: " + result);
	if (m_ActualActor.hasStopMessage()) {
	  debug("stop message: " + m_ActualActor.getStopMessage());
	  if (result == null)
	    result = m_ActualActor.getStopMessage();
	}
      }

      // finish up
      m_ActualActor.wrapUp();
      if (isDebugOn())
	debug("wrapUp() finished");

      // clean up?
      if (m_CleanUp) {
	m_ActualActor.cleanUp();
	if (isDebugOn())
	  debug("cleanUp() finished");
      }

      // any errors?
      if (result != null) {
	if (!m_File.isDirectory())
	  result = "Error executing flow '" + m_File + "': " + result;
	else
	  result = "Error executing actor: " + result;
      }
    }
    catch (Exception e) {
      result = e.toString();
      e.printStackTrace();
    }

    m_ActualActor = null;

    // interrupted by user?
    if (m_InterruptedByUser && (result == null))
      result = "Flow interrupted by user!";

    return result;
  }

  /**
   * Pauses the execution.
   */
  public void pauseExecution() {
    if ((m_ActualActor != null) && (m_ActualActor instanceof Pausable))
      ((Pausable) m_ActualActor).pauseExecution();
  }

  /**
   * Returns whether the object is currently paused.
   *
   * @return		true if object is paused
   */
  public boolean isPaused() {
    if ((m_ActualActor != null) && (m_ActualActor instanceof Pausable))
      return ((Pausable) m_ActualActor).isPaused();
    else
      return false;
  }

  /**
   * Resumes the execution.
   */
  public void resumeExecution() {
    if ((m_ActualActor != null) && (m_ActualActor instanceof Pausable))
      ((Pausable) m_ActualActor).resumeExecution();
  }

  /**
   * Stops the execution.
   */
  public void stopExecution() {
    if (m_ActualActor != null) {
      m_InterruptedByUser = true;
      m_ActualActor.stopExecution();
    }
  }

  /**
   * Instantiates the flow with the given options.
   *
   * @param classname	the classname of the flow to instantiate
   * @param options	the options for the flow
   * @return		the instantiated flow or null if an error occurred
   */
  public static FlowRunner forName(String classname, String[] options) {
    FlowRunner	result;

    try {
      result = (FlowRunner) OptionUtils.forName(FlowRunner.class, classname, options);
    }
    catch (Exception e) {
      e.printStackTrace();
      result = null;
    }

    return result;
  }

  /**
   * Instantiates the flow from the given commandline
   * (i.e., classname and optional options).
   *
   * @param cmdline	the classname (and optional options) of the
   * 			flow to instantiate
   * @return		the instantiated flow
   * 			or null if an error occurred
   */
  public static FlowRunner forCommandLine(String cmdline) {
    return (FlowRunner) AbstractOptionConsumer.fromString(ArrayConsumer.class, cmdline);
  }

  /**
   * Stops all engines of the specified ScriptingEngine class.
   *
   * @param engines	the scripting engine to use
   */
  protected static void stopAllEngines(Class[] engines) {
    int		i;
    Method	method;

    for (i = 0; i < engines.length; i++) {
      try {
	method = engines[i].getMethod(METHOD_STOPALLENGINES, new Class[0]);
	method.invoke(null, new Object[0]);
      }
      catch (Exception e) {
	System.err.println("Failed to call " + engines[i].getName() + "." + METHOD_STOPALLENGINES + ":");
	e.printStackTrace();
      }
    }
  }

  /**
   * Runs the flow from commandline.
   *
   * @param env		the environment class to use
   * @param flow	the flow class to execute
   * @param engines	the class array of the scripting engines
   * @param args	the commandline arguments, use -help to display all
   */
  public static void runFlow(Class env, Class flow, Class[] engines, String[] args) {
    FlowRunner	flowInst;
    String	result;

    Environment.setEnvironmentClass(env);
    Environment.setHome(OptionUtils.getOption(args, "-home"));

    try {
      if (OptionUtils.helpRequested(args)) {
	System.out.println("Help requested...\n");
	flowInst = forName(flow.getName(), new String[0]);
	System.out.println("\n" + OptionUtils.list(flowInst));
      }
      else {
	flowInst = forName(flow.getName(), args);
	ArrayConsumer.setOptions(flowInst, args);
	result = flowInst.execute();
	stopAllEngines(engines);
	if (result == null) {
	  System.out.println("\nFinished execution!");
	}
	else {
	  System.err.println("\n" + result);
	  System.exit(1);
	}
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Runs the flow with the given options.
   * Use "-f &lt;file&gt;" to supply a flow setup file to execute.
   *
   * @param args	the options to use
   */
  public static void main(String[] args) {
    runFlow(Environment.class, FlowRunner.class, new Class[]{adams.gui.scripting.ScriptingEngine.class}, args);
  }
}
