/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * MathExpression.java
 * Copyright (C) 2009-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.HashMap;

import adams.flow.core.Token;
import adams.parser.GrammarSupplier;
import adams.parser.MathematicalExpression;
import adams.parser.MathematicalExpressionText;

/**
 <!-- globalinfo-start -->
 * Evaluates a mathematical expression.<br/>
 * The input value can be accessed via 'X'.<br/>
 * Variables are supported as well, e.g.: pow(X,&#64;{exp}) with '&#64;{exp}' being a variable available at execution time.<br/>
 * <br/>
 * The following grammar is used for the expressions:<br/>
 * <br/>
 * expr_list ::= expr_list expr_part | expr_part ;<br/>
 * expr_part ::= expr ;<br/>
 * <br/>
 * numexpr   ::=   ( numexpr )<br/>
 *               |  NUMBER<br/>
 *               | -numexpr<br/>
 *               | constexpr<br/>
 *               | opexpr<br/>
 *               | varexpr<br/>
 *               | funcexpr<br/>
 *               ;<br/>
 * <br/>
 * constexpr ::=   PI<br/>
 *               | E<br/>
 *               ;<br/>
 * <br/>
 * opexpr    ::=   numexpr + numexpr<br/>
 *               | numexpr - numexpr<br/>
 *               | numexpr * numexpr<br/>
 *               | numexpr &#47; numexpr<br/>
 *               | numexpr ^ numexpr<br/>
 *               ;<br/>
 * <br/>
 * varexpr  ::=  VARIABLE ;<br/>
 * <br/>
 * funcexpr ::=    abs ( numexpr )<br/>
 *               | sqrt ( numexpr )<br/>
 *               | log ( numexpr )<br/>
 *               | exp ( numexpr )<br/>
 *               | sin ( numexpr )<br/>
 *               | cos ( numexpr )<br/>
 *               | tan ( numexpr )<br/>
 *               | rint ( numexpr )<br/>
 *               | floor ( numexpr )<br/>
 *               | pow ( numexpr , numexpr )<br/>
 *               | ceil ( numexpr )<br/>
 *               | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *               ;<br/>
 * <br/>
 * boolexpr ::=    BOOLEAN<br/>
 *               | true<br/>
 *               | false<br/>
 *               | numexpr &amp;lt; numexpr<br/>
 *               | numexpr &amp;lt;= numexpr<br/>
 *               | numexpr &amp;gt; numexpr<br/>
 *               | numexpr &amp;gt;= numexpr<br/>
 *               | numexpr = numexpr<br/>
 *               | ( boolexpr )<br/>
 *               | ! boolexpr<br/>
 *               | boolexpr &amp; boolexpr<br/>
 *               | boolexpr | boolexpr<br/>
 *               ;<br/>
 * <br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Integer<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Double<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.Double<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: MathExpression
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-expression &lt;java.lang.String&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The mathematical expression to evaluate; the input value can be accessed
 * &nbsp;&nbsp;&nbsp;via 'X'.
 * &nbsp;&nbsp;&nbsp;default: X
 * </pre>
 *
 * <pre>-output-value-pair (property: outputValuePair)
 * &nbsp;&nbsp;&nbsp;If enabled, a double array with X and Y is output and not just Y.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @see adams.parser.MathematicalExpression
 */
public class MathExpression
  extends AbstractTransformer
  implements GrammarSupplier {

  /** for serialization. */
  private static final long serialVersionUID = -8477454145267616359L;

  /** the placeholder for the input value. */
  public final static String PLACEHOLDER_INPUT = "X";

  /** the mathematical expression to evaluate. */
  protected MathematicalExpressionText m_Expression;

  /** whether to output X and Y as double array or just Y. */
  protected boolean m_OutputValuePair;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Evaluates a mathematical expression.\n"
      + "The input value can be accessed via '" + PLACEHOLDER_INPUT + "'.\n"
      + "Variables are supported as well, e.g.: pow(X,@{exp}) with '@{exp}' "
      + "being a variable available at execution time.\n\n"
      + "The following grammar is used for the expressions:\n\n"
      + getGrammar();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return new MathematicalExpression().getGrammar();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "expression", "expression",
	    new MathematicalExpressionText(PLACEHOLDER_INPUT));

    m_OptionManager.add(
	    "output-value-pair", "outputValuePair",
	    false);
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("expression");

    if (variable != null)
      return variable;
    else if ((m_Expression != null) && (m_Expression.getValue().length() > 0))
      return m_Expression.stringValue();
    else
      return null;
  }

  /**
   * Sets the mathematical expression to evaluate.
   *
   * @param value	the expression
   */
  public void setExpression(MathematicalExpressionText value) {
    m_Expression = value;
    reset();
  }

  /**
   * Returns the mathematical expression to evaluate.
   *
   * @return		the expression
   */
  public MathematicalExpressionText getExpression() {
    return m_Expression;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionTipText() {
    return
        "The mathematical expression to evaluate; the input value can be "
      + "accessed via '" + PLACEHOLDER_INPUT + "'.";
  }

  /**
   * Sets whether to output a double array with X and Y or just Y.
   *
   * @param value	if true then X and Y are output
   */
  public void setOutputValuePair(boolean value) {
    m_OutputValuePair = value;
    reset();
  }

  /**
   * Returns whether to output a double array with X and Y or just Y.
   *
   * @return		true if X and Y are ouput
   */
  public boolean getOutputValuePair() {
    return m_OutputValuePair;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputValuePairTipText() {
    return "If enabled, a double array with X and Y is output and not just Y.";
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.Integer.class, java.lang.Double.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Integer.class, Double.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.lang.Double.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    if (m_OutputValuePair)
      return new Class[]{Double[].class};
    else
      return new Class[]{Double.class};
  }

  /**
   * Initializes the sub-actors for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    if (result == null) {
      if ((m_Expression == null) || (m_Expression.getValue().length() == 0))
	result = "No expression provided!";
    }

    return result;
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    HashMap	symbols;
    double	x;
    double	y;
    String	exp;

    result = null;

    exp = m_Expression.getValue();
    try {
      // get input
      if (m_InputToken.getPayload() instanceof Integer)
	x = (Integer) m_InputToken.getPayload();
      else
	x = (Double) m_InputToken.getPayload();

      // replace variables with their actual values
      exp = getVariables().expand(exp);

      // evaluate the expression
      symbols = new HashMap();
      symbols.put(PLACEHOLDER_INPUT, new Double(x));
      y = MathematicalExpression.evaluate(exp, symbols);

      if (m_OutputValuePair)
	m_OutputToken = new Token(new Double[]{x, y});
      else
	m_OutputToken = new Token(new Double(y));
    }
    catch (Exception e) {
      m_OutputToken = null;
      getSystemErr().println("Error evaluating: " + exp);
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }
}
