/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * FlowPanel.java
 * Copyright (C) 2009-2012 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.flow;

import java.awt.BorderLayout;
import java.awt.Dialog.ModalityType;
import java.awt.Dimension;
import java.io.File;
import java.util.Vector;

import javax.swing.JOptionPane;
import javax.swing.SwingUtilities;
import javax.swing.SwingWorker;
import javax.swing.event.TreeSelectionEvent;
import javax.swing.event.TreeSelectionListener;

import adams.core.Pausable;
import adams.core.Properties;
import adams.core.StatusMessageHandler;
import adams.core.Utils;
import adams.core.VariablesHandler;
import adams.core.io.FileUtils;
import adams.core.io.FilenameProposer;
import adams.core.io.PlaceholderFile;
import adams.core.option.AbstractOptionConsumer;
import adams.core.option.AbstractOptionProducer;
import adams.core.option.NestedProducer;
import adams.data.statistics.InformativeStatistic;
import adams.db.LogEntryHandler;
import adams.env.Environment;
import adams.env.FlowEditorPanelDefinition;
import adams.flow.control.Flow;
import adams.flow.core.AbstractActor;
import adams.flow.core.ActorStatistic;
import adams.flow.core.ActorUtils;
import adams.flow.core.AutomatableInteractiveActor;
import adams.flow.processor.AbstractActorProcessor;
import adams.flow.processor.GraphicalOutputProducingProcessor;
import adams.flow.processor.ManageInteractiveActors;
import adams.flow.processor.ModifyingProcessor;
import adams.flow.processor.RemoveDisabledActors;
import adams.gui.core.BaseDialog;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.GUIHelper;
import adams.gui.core.RecentFilesHandler;
import adams.gui.core.TitleGenerator;
import adams.gui.core.ToolBarPanel.ToolBarLocation;
import adams.gui.core.Undo.UndoPoint;
import adams.gui.core.UndoPanel;
import adams.gui.dialog.TextDialog;
import adams.gui.event.ActorChangeEvent;
import adams.gui.event.ActorChangeListener;
import adams.gui.event.UndoEvent;
import adams.gui.flow.tree.Node;
import adams.gui.flow.tree.Tree;
import adams.gui.goe.GenericObjectEditorDialog;
import adams.gui.sendto.SendToActionSupporter;
import adams.gui.sendto.SendToActionUtils;
import adams.gui.tools.LogEntryViewerPanel;
import adams.gui.tools.VariableManagementPanel;
import adams.gui.visualization.statistics.InformativeStatisticFactory;

/**
 * A panel for setting up, modifying, saving and loading "simple" flows.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4655 $
 */
public class FlowPanel
  extends UndoPanel
  implements StatusMessageHandler, SendToActionSupporter {

  /** for serialization. */
  private static final long serialVersionUID = -3579084888256133873L;

  /** the prefix for the title of new tabs. */
  public final static String PREFIX_NEW = "new";

  /** the prefix for the title of custom set flows. */
  public final static String PREFIX_FLOW = "Flow";

  /** the counter for the tabs. */
  protected static int m_Counter;
  static {
    m_Counter = 0;
  }

  /** the properties. */
  protected static Properties m_Properties;

  /** the owner. */
  protected FlowTabbedPane m_Owner;

  /** whether the is currently running. */
  protected boolean m_Running;

  /** whether the generation is currently being stopped. */
  protected boolean m_Stopping;

  /** whether a flow is currently being loaded, etc. using a SwingWorker. */
  protected boolean m_RunningSwingWorker;

  /** the current flow. */
  protected AbstractActor m_CurrentFlow;

  /** the last flow that was run. */
  protected AbstractActor m_LastFlow;

  /** the filename of the current flow. */
  protected File m_CurrentFile;

  /** the tree displaying the flow structure. */
  protected Tree m_Tree;

  /** the recent files handler. */
  protected RecentFilesHandler m_RecentFilesHandler;

  /** for proposing filenames for new flows. */
  protected FilenameProposer m_FilenameProposer;

  /** the last variable search performed. */
  protected String m_LastVariableSearch;

  /** the dialog for processing actors. */
  protected GenericObjectEditorDialog m_DialogProcessActors;

  /** the default toolbar location to use. */
  protected ToolBarLocation m_ToolBarLocation;

  /** the panel with the variables. */
  protected VariableManagementPanel m_PanelVariables;

  /** for generating the title of the dialog/frame. */
  protected TitleGenerator m_TitleGenerator;

  /** the current title. */
  protected String m_Title;

  /** whether to execute the flow in headless mode. */
  protected boolean m_Headless;

  /**
   * Initializes the panel with an owner.
   *
   * @param owner	the owning tabbed pane
   */
  public FlowPanel(FlowTabbedPane owner) {
    super();

    m_Owner              = owner;
    m_RecentFilesHandler = getEditor().getRecentFilesHandler();

    reset(new Flow());
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_CurrentFlow         = null;
    m_LastFlow            = null;
    m_CurrentFile         = null;
    m_RecentFilesHandler  = null;
    m_LastVariableSearch  = "";
    m_TitleGenerator      = new TitleGenerator(FlowEditorPanel.DEFAULT_TITLE, true);
    m_FilenameProposer    = new FilenameProposer(PREFIX_NEW, AbstractActor.FILE_EXTENSION, getProperties().getString("InitialDir", "%h"));
    m_DialogProcessActors = null;
    m_Title               = "";
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    Properties			props;

    super.initGUI();

    props = getProperties();

    setLayout(new BorderLayout());

    // the tree
    m_Tree = new Tree(this);
    m_Tree.setActorNameColor(props.getProperty("Tree.ActorName.Color", "black"));
    m_Tree.setActorNameSize(props.getProperty("Tree.ActorName.Size", "3"));
    m_Tree.setQuickInfoColor(props.getProperty("Tree.QuickInfo.Color", "#008800"));
    m_Tree.setQuickInfoSize(props.getProperty("Tree.QuickInfo.Size", "-2"));
    m_Tree.setAnnotationsColor(props.getProperty("Tree.Annotations.Color", "blue"));
    m_Tree.setAnnotationsSize(props.getProperty("Tree.Annotations.Size", "-2"));
    m_Tree.setInputOutputColor(props.getProperty("Tree.InputOutput.Color", "blue"));
    m_Tree.setInputOutputSize(props.getProperty("Tree.InputOutput.Size", "-2"));
    m_Tree.setPlaceholdersColor(props.getProperty("Tree.Placeholders.Color", "navy"));
    m_Tree.setPlaceholdersSize(props.getProperty("Tree.Placeholders.Size", "-2"));
    m_Tree.setStateUsesNested(props.getBoolean("Tree.StateUsesNested", true));
    m_Tree.setIconScaleFactor(props.getDouble("Tree.IconScaleFactor", 1.0));
    m_Tree.setVariableHighlightBackground(props.getProperty("VariableHighlight.Background", "#FFDD88"));
    m_Tree.setShowQuickInfo(props.getBoolean("ShowQuickInfo", true));
    m_Tree.setShowAnnotations(props.getBoolean("ShowAnnotations", true));
    m_Tree.setShowInputOutput(props.getBoolean("ShowInputOutput", false));
    m_Tree.setInputOutputPrefixes(props.getProperty("Tree.InputOutput.Prefixes", "java.lang.,java.io.,adams.core.io.,adams.flow.core.,adams.flow.container.").replace(" ", "").split(","));
    m_Tree.addActorChangeListener(new ActorChangeListener() {
      public void actorChanged(ActorChangeEvent e) {
	update();
      }
    });
    m_Tree.getSelectionModel().addTreeSelectionListener(new TreeSelectionListener() {
      public void valueChanged(TreeSelectionEvent e) {
	if (m_Tree.getSelectionPath() != null)
	  showStatus(m_Tree.getSelectedFullName());
      }
    });
    add(new BaseScrollPane(m_Tree), BorderLayout.CENTER);

    // the tabs
    m_Tree.getSelectionModel().addTreeSelectionListener(new TreeSelectionListener() {
      public void valueChanged(TreeSelectionEvent e) {
	if (getEditor() != null)
	  getEditor().getTabs().notifyTabs(m_Tree.getSelectionPaths(), m_Tree.getSelectedActors());
      }
    });
  }

  /**
   * Returns the editor this panel belongs to.
   *
   * @return		the editor, null if none set
   */
  public FlowTabbedPane getOwner() {
    return m_Owner;
  }

  /**
   * Returns the editor this panel belongs to.
   *
   * @return		the editor, null if none set
   */
  public FlowEditorPanel getEditor() {
    if (m_Owner != null)
      return m_Owner.getOwner();
    else
      return null;
  }

  /**
   * Sets whether to execute the flow in headless mode.
   *
   * @param value	if true the flow gets executed in headless mode
   */
  public void setHeadless(boolean value) {
    if (!m_Running && !m_Stopping)
      m_Headless = value;
  }

  /**
   * Returns whether the flow gets executed in headless mode.
   *
   * @return		true if the flow gets executed in headless mode
   */
  public boolean isHeadless() {
    return m_Headless;
  }

  /**
   * Updates the enabled state of the widgets.
   */
  protected void updateWidgets() {
    boolean	inputEnabled;

    inputEnabled = !m_Running && !m_Stopping;

    getTree().setEditable(inputEnabled);
  }

  /**
   * updates the enabled state etc. of all the GUI elements.
   */
  protected void update() {
    updateWidgets();
    updateTitle();
    if ((getOwner() != null) && (getOwner().getOwner() != null))
      getOwner().getOwner().update();
  }

  /**
   * Increments the flow counter and returns the new value.
   *
   * @return		the new counter value
   */
  protected synchronized int next() {
    m_Counter++;
    return m_Counter;
  }

  /**
   * Sets the title for this flow.
   *
   * @param value	the title
   */
  public void setTitle(String value) {
    m_Title = value;
  }

  /**
   * Returns the current title of the flow.
   *
   * @return		the title
   */
  public String getTitle() {
    return m_Title;
  }

  /**
   * Updates the title of the dialog/frame.
   */
  public void updateTitle() {
    int		index;

    setParentTitle(m_TitleGenerator.generate(getCurrentFile(), getTree().isModified()));

    if (getOwner() != null) {
      index = getOwner().indexOfComponent(this);
      if (index != -1)
	getOwner().setTitleAt(index, (isModified() ? "*" : "") + getTitle());
    }
  }

  /**
   * Resets the GUI to default values.
   *
   * @param actor	the actor to instantiate
   */
  public void reset(AbstractActor actor) {
    addUndoPoint("Saving undo data...", "New " + actor.getClass().getName().replaceAll(".*\\.", ""));

    cleanUp();

    m_CurrentFlow = null;
    getTree().setActor(null);
    getTree().setActor(actor);
    getTree().setModified(false);
    setCurrentFile(null);

    setTitle(PREFIX_NEW + next());

    update();

    grabFocus();
  }

  /**
   * Sets the current file.
   *
   * @param value	the file
   */
  public void setCurrentFile(File value) {
    m_CurrentFile = value;
    if (value == null)
      m_Title = "";
    else
      m_Title = value.getName().substring(0, value.getName().lastIndexOf('.'));
    getTree().setFile(value);
  }

  /**
   * Returns the current file in use.
   *
   * @return		the current file, can be null
   */
  public File getCurrentFile() {
    return m_CurrentFile;
  }

  /**
   * Loads a flow.
   *
   * @param file	the flow to load
   */
  public void load(File file) {
    SwingWorker		worker;

    file = new PlaceholderFile(file);

    m_RunningSwingWorker = true;
    final File currFile = new File(file.getAbsolutePath());
    worker = new SwingWorker() {
      protected AbstractActor m_Flow = null;
      protected Vector<String> m_Errors;
      protected Vector<String> m_Warnings;

      protected Object doInBackground() throws Exception {
	m_Errors   = new Vector<String>();
	m_Warnings = new Vector<String>();

	cleanUp();
	update();

	addUndoPoint("Saving undo data...", "Loading '" + currFile.getName() + "'");
	showStatus("Loading '" + currFile + "'...");

	m_Flow = ActorUtils.read(currFile.getAbsolutePath(), m_Errors, m_Warnings);
	if (!m_Errors.isEmpty())
	  m_Flow = null;
	setCurrentFlow(m_Flow);

	showStatus("");

        return null;
      }

      protected void done() {
	m_RunningSwingWorker = false;

	if (m_Flow == null) {
	  if (m_Errors.isEmpty())
	    GUIHelper.showErrorMessage(
		m_Owner, "Failed to load flow '" + currFile + "'!");
	  else
	    GUIHelper.showErrorMessage(
		m_Owner, "Failed to load flow '" + currFile + "':\n" + Utils.flatten(m_Errors, "\n"));
	}
	else {
	  setCurrentFile(currFile);
	  if (m_RecentFilesHandler != null)
	    m_RecentFilesHandler.addRecentFile(currFile);
	  if (!m_Warnings.isEmpty())
	    GUIHelper.showErrorMessage(
		m_Owner, "Warning(s) encountered while loading flow '" + currFile + "':\n" + Utils.flatten(m_Warnings, "\n"));
	}

	update();

        super.done();
      }
    };
    worker.execute();
  }

  /**
   * Sets the flow to work on.
   *
   * @param flow	the flow to use
   */
  public void setCurrentFlow(AbstractActor flow) {
    m_CurrentFlow = flow;

    if (flow != null) {
      getTree().setActor(flow);
      getTree().setModified(false);
    }

    setCurrentFile(null);
    setTitle(PREFIX_FLOW + next());
  }

  /**
   * Returns the current flow.
   * <p/>
   * WARNING: Recreates an actor hierarchy based on the tree. Method gets very
   * slow for large flows. If you only need the root actor, then use getCurrentRoot()
   * instead.
   *
   * @return		the current flow
   * @see		#getCurrentRoot()
   * @see		Node#getFullActor()
   */
  public AbstractActor getCurrentFlow() {
    return getTree().getActor();
  }

  /**
   * Returns the current root actor without its children.
   *
   * @return		the current root actor
   * @see		#getCurrentFlow()
   */
  public AbstractActor getCurrentRoot() {
    return getTree().getRootActor();
  }

  /**
   * Returns the last executed flow (if any).
   *
   * @return		the flow, null if not available
   */
  public AbstractActor getLastFlow() {
    return m_LastFlow;
  }

  /**
   * Sets whether the flow is modified or not.
   *
   * @param value	true if the flow is to be flagged as modified
   */
  public void setModified(boolean value) {
    getTree().setModified(value);
    update();
  }

  /**
   * Returns whether the flow is flagged as modified.
   *
   * @return		true if the flow is modified
   */
  public boolean isModified() {
    return getTree().isModified();
  }

  /**
   * Reverts a flow.
   */
  public void revert() {
    cleanUp();
    load(getCurrentFile());
  }

  /**
   * Saves the flow.
   *
   * @param file	the file to save the flow to
   */
  protected void save(final File file) {
    SwingWorker		worker;

    worker = new SwingWorker() {
      boolean m_Result;

      protected Object doInBackground() throws Exception {
	showStatus("Saving '" + file + "'...");
	m_Result = ActorUtils.write(file.getAbsolutePath(), getCurrentFlow());
	showStatus("");
        return null;
      }

      protected void done() {
	if (!m_Result)
	  GUIHelper.showErrorMessage(
	      m_Owner, "Error saving flow to '" + file.getAbsolutePath() + "'!");
	else {
	  showStatus("");
	  getTree().setModified(false);
	  if (m_RecentFilesHandler != null)
	    m_RecentFilesHandler.addRecentFile(file);
	  setCurrentFile(file);
	}

	update();

        super.done();
      }
    };
    worker.execute();
  }

  /**
   * Imports a flow.
   *
   * @param consumer	the consumer to use
   * @param file	the file to import
   */
  public void importFlow(AbstractOptionConsumer consumer, File file) {
    AbstractActor		actor;

    actor = (AbstractActor) consumer.fromFile(file);
    if (actor == null) {
      showMessage("Failed to load flow from:\n" + file, true);
    }
    else {
      getTree().setActor(actor);
      setCurrentFile(new PlaceholderFile(file.getAbsolutePath() + "." + AbstractActor.FILE_EXTENSION));
      if (!consumer.hasErrors())
	showMessage("Flow successfully imported from:\n" + file, false);
      else
	showMessage("Flow import of:\n" + file + "\nResulted in errors:\n" + Utils.flatten(consumer.getErrors(), "\n"), true);
    }
  }

  /**
   * Exports the flow.
   *
   * @param producer	the producer to use
   * @param file	the file to export to
   */
  public void exportFlow(AbstractOptionProducer producer, File file) {
    producer.produce(getCurrentFlow());
    if (!FileUtils.writeToFile(file.getAbsolutePath(), producer.toString(), false)) {
      showMessage("Failed to export flow to:\n" + file, true);
    }
    else {
      showMessage("Flow successfully exported to:\n" + file, false);
    }
  }

  /**
   * Validates the current setup.
   */
  public void validateSetup() {
    AbstractActor	actor;
    String		msg;

    actor = getCurrentFlow();
    try {
      msg   = actor.setUp();
      actor.wrapUp();
      actor.cleanUp();
    }
    catch (Exception e) {
      msg = "Actor generated exception: ";
      System.err.println(msg);
      e.printStackTrace();
      msg += e;
    }

    if (msg == null) {
      msg = "The flow passed validation!";
      showStatus(msg);
      showMessage(msg, false);
    }
    else {
      showStatus(msg);
      showMessage("The flow setup failed validation:\n" + msg, true);
    }
  }

  /**
   * Executes the flow.
   */
  public void run() {
    run(true);
  }

  /**
   * Executes the flow.
   *
   * @param showNotification	whether to show notifications about
   * 				errors/stopped/finished
   */
  public void run(boolean showNotification) {
    final boolean fShowNotification;

    fShowNotification = showNotification;
    m_Running         = true;

    SwingWorker worker = new SwingWorker() {
      String m_Output;

      protected Object doInBackground() throws Exception {
	update();
	cleanUp();

	try {
	  showStatus("Initializing");
	  m_CurrentFlow = getCurrentFlow();
	  m_CurrentFlow.setHeadless(m_Headless);
	  m_CurrentFlow = ActorUtils.removeDisabledActors(m_CurrentFlow);
	  if (m_CurrentFlow instanceof VariablesHandler)
	    ActorUtils.updateVariablesWithFlowFilename((VariablesHandler) m_CurrentFlow, m_CurrentFile);
	  m_Output      = m_CurrentFlow.setUp();
	  if ((m_Output == null) && !m_CurrentFlow.isStopped()) {
	    showStatus("Running");
	    m_Output = m_CurrentFlow.execute();
	    // did the flow get stopped by a critical actor?
	    if ((m_Output == null) && m_CurrentFlow.hasStopMessage())
	      m_Output = m_CurrentFlow.getStopMessage();
	  }
	  showStatus("Finishing up");
	  m_CurrentFlow.wrapUp();
	}
	catch (Throwable e) {
	  e.printStackTrace();
	  m_Output = Utils.throwableToString(e);
	}

	if ((m_PanelVariables != null) && (m_PanelVariables.getParentDialog() != null))
	  m_PanelVariables.getParentDialog().setVisible(false);

	return "Done!";
      }

      protected void done() {
	String	msg;
	String	errors;
	int	countErrors;

	super.done();

	m_LastFlow    = m_CurrentFlow;
	m_CurrentFlow = null;
	errors        = null;

	if (m_LastFlow instanceof LogEntryHandler) {
	  countErrors = ((LogEntryHandler) m_LastFlow).countLogEntries();
	  if (countErrors > 0)
	    errors = countErrors + " error(s) logged";
	}

	if (m_Output != null) {
	  msg = "Finished with error: " + m_Output;
	  if (errors != null)
	    msg += "(" + errors + ")";
	  showStatus(msg);
	  if (fShowNotification)
	    showMessage(m_Output, true);
	}
	else {
	  if (m_Running)
	    msg = "Flow finished.";
	  else
	    msg = "User stopped flow.";
	  if (errors != null)
	    msg += " " + errors + ".";
	  showStatus(msg);
	  if (fShowNotification) {
	    if (m_Running)
	      GUIHelper.showInformationMessage(m_Owner, msg);
	    else
	      GUIHelper.showErrorMessage(m_Owner, msg);
	  }
	}

	m_Running  = false;
	m_Stopping = false;

	update();
      }
    };
    worker.execute();
  }

  /**
   * Returns whether a flow is currently running.
   *
   * @return		true if a flow is being executed
   */
  public boolean isRunning() {
    return m_Running;
  }

  /**
   * Returns whether a flow is currently being stopped.
   *
   * @return		true if a flow is currently being stopped
   */
  public boolean isStopping() {
    return m_Stopping;
  }

  /**
   * Returns whether a swing worker is currently running.
   *
   * @return		true if a swing worker is being executed
   */
  public boolean isSwingWorkerRunning() {
    return m_RunningSwingWorker;
  }

  /**
   * Pauses/resumes the flow.
   *
   * @return		true if paused, otherwise false
   */
  public boolean pauseAndResume() {
    boolean	result;
    Pausable	pausable;

    pausable = (Pausable) m_CurrentFlow;
    if (!pausable.isPaused()) {
      showStatus("Pausing");
      pausable.pauseExecution();
      result = true;
    }
    else {
      showStatus("Resuming");
      pausable.resumeExecution();
      result = false;
    }

    update();

    return result;
  }

  /**
   * Stops the flow.
   */
  public void stop() {
    Runnable	runnable;

    showStatus("Stopping");

    m_Running  = false;
    m_Stopping = true;
    update();

    runnable = new Runnable() {
      public void run() {
	if (m_CurrentFlow != null)
	  m_CurrentFlow.stopExecution();
	update();
      }
    };
    SwingUtilities.invokeLater(runnable);
  }

  /**
   * Displays the errors from the last run.
   */
  public void displayErrors() {
    BaseDialog		dialog;
    LogEntryHandler	handler;
    LogEntryViewerPanel	panel;

    if (m_LastFlow == null)
      return;
    if (!(m_LastFlow instanceof LogEntryHandler))
      return;
    handler = (LogEntryHandler) m_LastFlow;
    if (handler.getLogEntries().size() == 0)
      return;

    if (getParentDialog() != null)
      dialog = new BaseDialog(getParentDialog(), ModalityType.MODELESS);
    else
      dialog = new BaseDialog(getParentFrame(), false);
    dialog.setTitle("Flow execution errors");
    panel = new LogEntryViewerPanel();
    panel.display(handler.getLogEntries());
    dialog.getContentPane().setLayout(new BorderLayout());
    dialog.getContentPane().add(panel, BorderLayout.CENTER);
    dialog.setSize(new Dimension(800, 600));
    dialog.setLocationRelativeTo(this);
    dialog.setVisible(true);
  }

  /**
   * Cleans up the last flow that was run.
   */
  public void cleanUp() {
    if (m_LastFlow != null) {
      showStatus("Cleaning up");
      try {
	m_LastFlow.destroy();
	m_LastFlow = null;
	showStatus("");
      }
      catch (Exception e) {
	e.printStackTrace();
	showStatus("Error cleaning up: " + e);
      }
    }
  }

  /**
   * Cleans up and closes the tab.
   */
  public void close() {
    cleanUp();
    if (m_Owner != null)
      m_Owner.remove(this);
  }

  /**
   * Displays statistics about the current flow.
   */
  public void statistics() {
    ActorStatistic			stats;
    InformativeStatisticFactory.Dialog	dialog;
    Vector<InformativeStatistic>	statsList;

    if (getTree().getSelectedNode() != null)
      stats = new ActorStatistic(getTree().getSelectedNode().getFullActor());
    else if (m_CurrentFlow != null)
      stats = new ActorStatistic(m_CurrentFlow);
    else
      stats = new ActorStatistic(getCurrentFlow());
    statsList = new Vector<InformativeStatistic>();
    statsList.add(stats);

    if (getParentDialog() != null)
      dialog = InformativeStatisticFactory.getDialog(getParentDialog(), ModalityType.DOCUMENT_MODAL);
    else
      dialog = InformativeStatisticFactory.getDialog(getParentFrame(), true);
    dialog.setStatistics(statsList);
    dialog.setTitle("Actor statistics");
    dialog.pack();
    dialog.setLocationRelativeTo(this);
    dialog.setVisible(true);
  }

  /**
   * Adds an undo point, if possible.
   *
   * @param statusMsg	the status message to display while adding the undo point
   * @param undoComment	the comment for the undo point
   */
  public void addUndoPoint(String statusMsg, String undoComment) {
    if (isUndoSupported() && getUndo().isEnabled()) {
      showStatus(statusMsg);
      getUndo().addUndo(getTree().getState(), undoComment);
      showStatus("");
    }
  }

  /**
   * peforms an undo if possible.
   */
  public void undo() {
    if (!getUndo().canUndo())
      return;

    SwingWorker worker = new SwingWorker() {
      protected Object doInBackground() throws Exception {
	showStatus("Performing Undo...");

	// add redo point
	getUndo().addRedo(getTree().getState(), getUndo().peekUndoComment());

	UndoPoint point = getUndo().undo();
	getTree().setState((Vector) point.getData());
	m_CurrentFile = getTree().getFile();

	return "Done!";
      };

      protected void done() {
        super.done();
	update();
	showStatus("");
      }
    };
    worker.execute();
  }

  /**
   * peforms a redo if possible.
   */
  public void redo() {
    if (!getUndo().canRedo())
      return;

    SwingWorker worker = new SwingWorker() {
      protected Object doInBackground() throws Exception {
	showStatus("Performing Redo...");

	// add undo point
	getUndo().addUndo(getTree().getState(), getUndo().peekRedoComment(), true);

	UndoPoint point = getUndo().redo();
	getTree().setState((Vector) point.getData());
	m_CurrentFile = getTree().getFile();

	return "Done!";
      };

      protected void done() {
        super.done();
	update();
	showStatus("");
      }
    };
    worker.execute();
  }

  /**
   * Searches for actor names in the tree.
   */
  public void find() {
    getTree().find();
  }

  /**
   * Searches for the next actor in the tree.
   */
  public void findNext() {
    getTree().findNext();
  }

  /**
   * Locates an actor based on the full actor name.
   */
  public void locateActor() {
    String	path;

    path = JOptionPane.showInputDialog("Please enter the full name of the actor (e.g., 'Flow[0].Sequence[3].Display'):");
    if (path == null)
      return;

    getTree().locateAndDisplay(path);
  }

  /**
   * Highlights variables in the tree (or hides the highlights again).
   *
   * @param highlight	whether to turn the highlights on or off
   */
  public void highlightVariables(boolean highlight) {
    String	regexp;

    if (highlight) {
      regexp = JOptionPane.showInputDialog(
	  this,
	  "Enter the regular expression for the variable name ('.*' matches all):",
	  m_LastVariableSearch);
      if (regexp == null)
	return;

      m_LastVariableSearch = regexp;
      getTree().highlightVariables(m_LastVariableSearch);
    }
    else {
      getTree().highlightVariables(null);
    }
  }

  /**
   * Cleans up the flow, e.g., removing disabled actors, unused global actors.
   *
   * @see		ActorUtils#cleanUpFlow(AbstractActor)
   */
  public void cleanUpFlow() {
    AbstractActor	cleaned;

    cleaned = ActorUtils.cleanUpFlow(getCurrentFlow());

    if (cleaned != null) {
      addUndoPoint("Saving undo data...", "Cleaning up");
      getTree().buildTree(cleaned);
      getTree().setModified(true);
      update();
    }
  }

  /**
   * Enables/disables the interactive behaviour of {@link AutomatableInteractiveActor}
   * actors.
   *
   * @param enable	true if to enable the interactive behaviour
   * @see		AutomatableInteractiveActor
   */
  public void manageInteractiveActors(boolean enable) {
    ManageInteractiveActors	processor;
    
    processor = new ManageInteractiveActors();
    processor.setEnable(enable);
    processor.process(getCurrentFlow());
    if (processor.isModified()) {
      addUndoPoint("Saving undo data...", (enable ? "Enable" : "Disable") + " interactive behaviour");
      getTree().buildTree(processor.getModifiedActor());
      getTree().setModified(true);
      update();
    }
  }

  /**
   * Processes the actors with a user-specified actor processor.
   */
  public void processActors() {
    AbstractActorProcessor		processor;
    ModifyingProcessor			modiyfing;
    GraphicalOutputProducingProcessor	graphical;
    BaseDialog				dialog;

    if (m_DialogProcessActors == null) {
      if (getParentDialog() != null)
	m_DialogProcessActors = new GenericObjectEditorDialog(getParentDialog());
      else
	m_DialogProcessActors = new GenericObjectEditorDialog(getParentFrame());
      m_DialogProcessActors.setTitle("Process actors");
      m_DialogProcessActors.setModalityType(ModalityType.DOCUMENT_MODAL);
      m_DialogProcessActors.getGOEEditor().setCanChangeClassInDialog(true);
      m_DialogProcessActors.getGOEEditor().setClassType(AbstractActorProcessor.class);
      m_DialogProcessActors.setCurrent(new RemoveDisabledActors());
    }
    m_DialogProcessActors.setLocationRelativeTo(this);
    m_DialogProcessActors.setVisible(true);
    if (m_DialogProcessActors.getResult() != GenericObjectEditorDialog.APPROVE_OPTION)
      return;
    processor = (AbstractActorProcessor) m_DialogProcessActors.getCurrent();
    processor.process(getCurrentFlow());
    if (processor instanceof ModifyingProcessor) {
      modiyfing = (ModifyingProcessor) processor;
      if (modiyfing.isModified()) {
	addUndoPoint("Saving undo data...", "Processing actors with " + processor.toString());
	getTree().buildTree(modiyfing.getModifiedActor());
	getTree().setModified(true);
	update();
      }
    }
    if (processor instanceof GraphicalOutputProducingProcessor) {
      graphical = (GraphicalOutputProducingProcessor) processor;
      if (graphical.hasGraphicalOutput()) {
	if (getParentDialog() != null)
	  dialog = new BaseDialog(getParentDialog());
	else
	  dialog = new BaseDialog(getParentFrame());
	dialog.setTitle(processor.getClass().getSimpleName());
	dialog.getContentPane().setLayout(new BorderLayout());
	dialog.getContentPane().add(graphical.getGraphicalOutput(), BorderLayout.CENTER);
	dialog.pack();
	dialog.setLocationRelativeTo(this);
	dialog.setVisible(true);
      }
    }
    showMessage("Actors processed!", false);
  }

  /**
   * Enables/disables all breakpoints in the flow (before execution).
   *
   * @param enable	if true then breakpoints get enabled
   */
  public void enableBreakpoints(boolean enable) {
    getTree().enableBreakpoints(enable);
  }

  /**
   * Displays the variables in the current running flow.
   */
  public void showVariables() {
    BaseDialog	dialog;

    if (m_PanelVariables == null) {
      m_PanelVariables = new VariableManagementPanel();
      if (getParentDialog() != null)
	dialog = new BaseDialog(getParentDialog());
      else
	dialog = new BaseDialog(getParentFrame());
      dialog.setTitle("Variables");
      dialog.getContentPane().setLayout(new BorderLayout());
      dialog.getContentPane().add(m_PanelVariables, BorderLayout.CENTER);
      dialog.pack();
      dialog.setLocationRelativeTo(null);
    }

    m_PanelVariables.setVariables(m_CurrentFlow.getVariables());
    m_PanelVariables.getParentDialog().setVisible(true);
  }

  /**
   * Displays the source code (in nested format) of the current flow.
   */
  public void showSource() {
    TextDialog 	dialog;
    String 	buffer;

    buffer = AbstractOptionProducer.toString(NestedProducer.class, getCurrentFlow());

    if (getParentDialog() != null)
      dialog = new TextDialog(getParentDialog());
    else
      dialog = new TextDialog(getParentFrame());
    dialog.setTitle(m_TitleGenerator.generate(getCurrentFile(), getTree().isModified()) + " [Source]");
    dialog.setTabSize(2);
    dialog.setContent(buffer);
    dialog.setLocationRelativeTo(getTree());
    dialog.setVisible(true);
  }

  /**
   * Displays the given message in a separate dialog.
   *
   * @param msg		the message to display
   * @param isError	whether it is an error message
   */
  protected void showMessage(String msg, boolean isError) {
    String	status;

    status = msg.replaceAll(": ", ":\n");

    if (isError)
      GUIHelper.showErrorMessage(m_Owner, status, "Error");
    else
      GUIHelper.showInformationMessage(m_Owner, status, "Status");
  }

  /**
   * Displays a message.
   *
   * @param msg		the message to display
   */
  public void showStatus(String msg) {
    if (getEditor() != null)
      getEditor().showStatus(msg);
  }

  /**
   * An undo event occurred.
   *
   * @param e		the event
   */
  public void undoOccurred(UndoEvent e) {
    update();
  }

  /**
   * Requests that this Component get the input focus, and that this
   * Component's top-level ancestor become the focused Window. This component
   * must be displayable, visible, and focusable for the request to be
   * granted.
   */
  public void grabFocus() {
    Runnable	runnable;

    runnable = new Runnable() {
      public void run() {
	getTree().grabFocus();
      }
    };
    SwingUtilities.invokeLater(runnable);
  }

  /**
   * Returns the tree.
   *
   * @return		the tree
   */
  public Tree getTree() {
    return m_Tree;
  }

  /**
   * Sets the recent files handler to use.
   *
   * @param value	the handler to use
   */
  public void setRecentFilesHandler(RecentFilesHandler value) {
    m_RecentFilesHandler = value;
  }

  /**
   * Returns the recent files handler to use.
   *
   * @return		the handler in use, null if none set
   */
  public RecentFilesHandler getRecentFilesHandler() {
    return m_RecentFilesHandler;
  }

  /**
   * Returns the classes that the supporter generates.
   *
   * @return		the classes
   */
  public Class[] getSendToClasses() {
    return new Class[]{PlaceholderFile.class};
  }

  /**
   * Checks whether something to send is available.
   *
   * @param cls		the classes to retrieve the item for
   * @return		true if an object is available for sending
   */
  public boolean hasSendToItem(Class[] cls) {
    return    SendToActionUtils.isAvailable(PlaceholderFile.class, cls)
           && !(!getTree().isModified() && (getCurrentFile() == null));
  }

  /**
   * Returns the object to send.
   *
   * @param cls		the classes to retrieve the item for
   * @return		the item to send
   */
  public Object getSendToItem(Class[] cls) {
    PlaceholderFile	result;
    AbstractActor	actor;

    result = null;

    if (SendToActionUtils.isAvailable(PlaceholderFile.class, cls)) {
      if (getTree().isModified()) {
	result = SendToActionUtils.nextTmpFile("floweditor", "flow");
	actor  = getCurrentFlow();
	ActorUtils.write(result.getAbsolutePath(), actor);
      }
      else if (getCurrentFile() != null) {
	result = new PlaceholderFile(getCurrentFile());
      }
    }

    return result;
  }

  /**
   * Returns the properties that define the editor.
   *
   * @return		the properties
   */
  public static synchronized Properties getProperties() {
    if (m_Properties == null)
      m_Properties = Environment.getInstance().read(FlowEditorPanelDefinition.KEY);

    return m_Properties;
  }
}
