/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    FileEditor.java
 *    Copyright (C) 1999-2008 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.gui.goe;

import java.awt.Component;
import java.awt.FontMetrics;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;

import javax.swing.JComponent;

import adams.core.io.PlaceholderFile;
import adams.core.option.AbstractOption;
import adams.gui.chooser.BaseFileChooser;

/**
 * A PropertyEditor for File objects that lets the user select a file.
 * <p/>
 * Based on <code>weka.gui.FileEditor</code>.
 *
 * @author Len Trigg (trigg@cs.waikato.ac.nz)
 * @version $Revision: 4586 $
 * @see weka.gui.FileEditor
 */
public class FileEditor
  extends AbstractPropertyEditorSupport
  implements CustomStringRepresentationHandler, MultiSelectionEditor {

  /** The file chooser used for selecting files. */
  protected BaseFileChooser m_FileChooser;

  /**
   * Returns the file as string.
   *
   * @param option	the current option
   * @param object	the file object to convert
   * @return		the generated string
   */
  public static String toString(AbstractOption option, Object object) {
    return ((File) object).getPath();
  }

  /**
   * Returns a file generated from the string.
   *
   * @param option	the current option
   * @param str		the string to convert to a file
   * @return		the generated file
   */
  public static Object valueOf(AbstractOption option, String str) {
    return new File(str);
  }

  /**
   * Returns a custom string representation of the object.
   *
   * @param obj		the object to turn into a string
   * @return		the string representation
   */
  public String toCustomStringRepresentation(Object obj) {
    return toString(null, obj);
  }

  /**
   * Returns an object created from the custom string representation.
   *
   * @param str		the string to turn into an object
   * @return		the object
   */
  public Object fromCustomStringRepresentation(String str) {
    return valueOf(null, str);
  }

  /**
   * Returns a representation of the current property value as java source.
   *
   * @return 		a value of type 'String'
   */
  public String getJavaInitializationString() {
    File f = (File) getValue();
    if (f == null)
      return "null";
    else
      return "new java.io.File(\"" + f.getName() + "\")";
  }

  /**
   * Gets the custom editor component.
   *
   * @return 		a value of type 'Component'
   */
  protected JComponent createCustomEditor() {
    File 	currentFile;

    currentFile = (File) getValue();
    if (currentFile != null)
      m_FileChooser = new BaseFileChooser();
    else
      m_FileChooser = new BaseFileChooser(new File(System.getProperty("user.dir")));
    m_FileChooser.setApproveButtonText("Select");
    m_FileChooser.setApproveButtonMnemonic('S');
    m_FileChooser.setFileSelectionMode(BaseFileChooser.FILES_AND_DIRECTORIES);
    m_FileChooser.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	String cmdString = e.getActionCommand();
	if (cmdString.equals(BaseFileChooser.APPROVE_SELECTION)) {
	  File newVal = m_FileChooser.getSelectedFile();
	  setValue(newVal);
	}
	closeDialog();
      }
    });

    return m_FileChooser;
  }

  /**
   * Initializes the display of the value.
   */
  protected void initForDisplay() {
    File 	currentFile;

    currentFile = (File) getValue();
    if (currentFile != null)
      m_FileChooser.setSelectedFile(currentFile);
  }

  /**
   * Paints a representation of the current Object.
   *
   * @param gfx 	the graphics context to use
   * @param box 	the area we are allowed to paint into
   */
  public void paintValue(java.awt.Graphics gfx, java.awt.Rectangle box) {
    FontMetrics fm = gfx.getFontMetrics();
    int vpad = (box.height - fm.getHeight()) / 2 ;
    File f = (File) getValue();
    String val = "No file";
    if (f != null)
      val = f.getPath();
    gfx.drawString(val, 2, fm.getHeight() + vpad);
  }

  /**
   * Returns the selected objects.
   *
   * @return		the objects
   */
  public Object[] getSelectedObjects() {
    File[]		result;
    Component		comp;
    BaseFileChooser	chooser;
    int			retVal;
    
    // we need the parent
    comp = getCustomEditor();
    initForDisplay();
    
    chooser = new BaseFileChooser(new PlaceholderFile(System.getProperty("user.dir")));
    chooser.setApproveButtonText("Select");
    chooser.setApproveButtonMnemonic('S');
    chooser.setMultiSelectionEnabled(true);
    chooser.setFileSelectionMode(BaseFileChooser.FILES_AND_DIRECTORIES);
    retVal = chooser.showOpenDialog(comp.getParent());
    if (retVal == BaseFileChooser.APPROVE_OPTION)
      result = chooser.getSelectedFiles();
    else
      result = new File[0];
    
    return result;
  }
}
