/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * TextFileWithLineNumbersWriter.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.core.io;

import adams.core.Utils;

/**
 <!-- globalinfo-start -->
 * Writes the content to a text file, preceding each line with its line number.Makes sure that the filename is valid.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-dir &lt;adams.core.io.PlaceholderFile&gt; (property: directory)
 * &nbsp;&nbsp;&nbsp;The directory to save the outputs in.
 * &nbsp;&nbsp;&nbsp;default: .
 * </pre>
 *
 * <pre>-prefix &lt;java.lang.String&gt; (property: prefix)
 * &nbsp;&nbsp;&nbsp;The prefix for the output files.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-suffix &lt;java.lang.String&gt; (property: suffix)
 * &nbsp;&nbsp;&nbsp;The suffix for the output files.
 * &nbsp;&nbsp;&nbsp;default: .txt
 * </pre>
 *
 * <pre>-ignore-name (property: ignoreName)
 * &nbsp;&nbsp;&nbsp;If set to true, then the name of the content is ignored for generating the
 * &nbsp;&nbsp;&nbsp;filename (useful when prefix or suffix is based on variables).
 * </pre>
 *
 * <pre>-separator &lt;java.lang.String&gt; (property: separator)
 * &nbsp;&nbsp;&nbsp;The separator between line number and line content.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class TextFileWithLineNumbersWriter
  extends TextFileWriter {

  /** for serialization. */
  private static final long serialVersionUID = 4958356636746933734L;

  /** the separator between the line number and the actual line. */
  protected String m_Separator;

  /**
   * Returns a short description of the writer.
   *
   * @return		a description of the writer
   */
  public String globalInfo() {
    return
        "Writes the content to a text file, preceding each line with its line number."
      + "Makes sure that the filename is valid.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "separator", "separator",
	    "");
  }

  /**
   * Sets the separator between line number and line content.
   *
   * @param value 	the separator
   */
  public void setSeparator(String value) {
    m_Separator = value;
    reset();
  }

  /**
   * Returns the separator between line number and line content.
   *
   * @return 		the separator
   */
  public String getSeparator() {
    return m_Separator;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String separatorTipText() {
    return "The separator between line number and line content.";
  }

  /**
   * Writes the given content under the specified name.
   *
   * @param content	the content to write
   * @param name	the name under which to save the content
   * @return		if a file was generated, the filename the content was written
   * 			as, otherwise null
   */
  public String write(String content, String name) {
    String[]	lines;
    int		i;

    // add line numbers
    lines = content.split("\n");
    for (i = 0; i < lines.length; i++)
      lines[i] = (i+1) + m_Separator + lines[i];

    return super.write(Utils.flatten(lines, "\n"), name);
  }
}
