/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * AbstractOptionVisitorProducer.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.core.option;

import java.util.List;

import JSci.maths.wavelet.IllegalScalingException;
import adams.core.CleanUpHandler;
import adams.core.ConsoleObject;
import adams.core.Debuggable;
import adams.core.io.FileUtils;

/**
 * Generates output from visiting the options.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @param <O> the type of output data that gets generated
 * @param <I> the internal type used while nesting
 */
public abstract class AbstractOptionProducer<O,I>
  extends ConsoleObject
  implements Debuggable, CleanUpHandler {

  /** for serialization. */
  private static final long serialVersionUID = 4502704821224667069L;

  /** the top-level visited object. */
  protected OptionHandler m_Input;

  /** the debugging level (0 = off, >0 = on). */
  protected int m_DebugLevel;

  /** whether to use command-line flags or property names. */
  protected boolean m_UsePropertyNames;

  /** the output data. */
  protected transient O m_Output;

  /** the last data structure that was generated. */
  protected I m_LastGenerated;

  /** whether to output the current value instead of variable placeholders. */
  protected boolean m_OutputVariableValues;

  /**
   * Initializes the visitor.
   */
  public AbstractOptionProducer() {
    super();
    initialize();
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    m_DebugLevel           = 0;
    m_UsePropertyNames     = false;
    m_Output               = initOutput();
    m_OutputVariableValues = (this instanceof DebugOptionProducer);
  }

  /**
   * Resets the members.
   */
  protected void reset() {
    m_Input         = null;
    m_Output        = initOutput();
    m_LastGenerated = null;
  }

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public abstract String globalInfo();

  /**
   * Sets the debugging level (0 = off).
   *
   * @param value 	>0 if debugging output should be printed
   */
  public void setDebugLevel(int value) {
    m_DebugLevel = value;
    getDebugging().setEnabled(value > 0);
  }

  /**
   * Returns the debugging level (0 = turned off).
   *
   * @return 		true if debugging output is on
   */
  public int getDebugLevel() {
    return m_DebugLevel;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String debugLevelTipText() {
    return "The level of debugging output (0 = no output).";
  }

  /**
   * Returns true if debugging output is turned on (any level).
   *
   * @return		true if debugging output is turned on
   */
  protected boolean isDebugOn() {
    return (m_DebugLevel > 0);
  }

  /**
   * Processes the debugging message.
   *
   * @param msg		the debugging message to process
   */
  protected void debug(String msg) {
    debug(msg, 1);
  }

  /**
   * Processes the debugging message.
   *
   * @param level	the debugging level
   * @param msg		the debugging message to process
   */
  protected void debug(String msg, int level) {
    if (level <= m_DebugLevel)
      getDebugging().println(msg);
  }

  /**
   * Generates a debug string, e.g., based on the method name.
   * <p/>
   * Default implementation merely returns the string.
   *
   * @param s		the string to process
   * @return		the processed string
   */
  protected String generateDebugString(String s) {
    return s;
  }

  /**
   * Sets whether to output the values of options instead of variable
   * placeholders.
   *
   * @param value	if true then the values are output instead of variable placeholders
   */
  public void setOutputVariableValues(boolean value) {
    m_OutputVariableValues = value;
  }

  /**
   * Returns whether the values of options are output instead of variable
   * placeholders.
   *
   * @return		true if values are output instead of variable placeholders
   */
  public boolean getOutputVariableValues() {
    return m_OutputVariableValues;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String outputVariableValuesTipText() {
    return "Whether to output the values of variables instead of the variable placeholders.";
  }

  /**
   * Returns whether property names are used or just the command-line flags.
   *
   * @return		true if property names are used
   */
  public boolean getUsePropertyNames() {
    return m_UsePropertyNames;
  }

  /**
   * Returns the visited top-level object.
   *
   * @return		the visited object
   */
  public OptionHandler getInput() {
    return m_Input;
  }

  /**
   * Returns either the property name or the commandline flag, depending
   * on whether property names are to be used or not.
   *
   * @param option	the option to return the identifier for
   * @return		the identifier
   * @see		#getUsePropertyNames()
   */
  protected String getOptionIdentifier(AbstractOption option) {
    if (getUsePropertyNames())
      return option.getProperty();
    else
      return "-" + option.getCommandline();
  }

  /**
   * Initializes the output data structure.
   *
   * @return		the created data structure
   */
  protected abstract O initOutput();

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output
   * @see		#initOutput()t
   */
  public O getOutput() {
    if (m_Output == null)
      m_Output = initOutput();

    return m_Output;
  }

  /**
   * Returns the current value for the option.
   *
   * @param option	the option to get the current value for
   * @return		the current value (can be array)
   */
  protected Object getCurrentValue(AbstractOption option) {
    return option.getCurrentValue();
  }

  /**
   * Checks whether the value represents the default value for the option.
   *
   * @param option	the option to check the default value for
   * @param value	the (potential) default value
   * @return		true if the value represents the default value
   */
  protected boolean isDefaultValue(AbstractArgumentOption option, Object value) {
    return option.isDefaultValue(value);
  }

  /**
   * Hook method that gets called just before an option gets produced.
   * <p/>
   * Default implementation does nothing
   *
   * @param manager	the option manager
   * @param index	the index of the option
   */
  protected void preProduce(OptionManager manager, int index) {
  }

  /**
   * Visits a boolean option.
   *
   * @param option	the boolean option
   * @return		the last internal data structure that was generated
   */
  public abstract I processOption(BooleanOption option);

  /**
   * Visits a class option.
   *
   * @param option	the class option
   * @return		the last internal data structure that was generated
   */
  public abstract I processOption(ClassOption option);

  /**
   * Visits an argument option.
   *
   * @param option	the argument option
   * @return		the last internal data structure that was generated
   */
  public abstract I processOption(AbstractArgumentOption option);

  /**
   * Visits the option and obtains information from it.
   *
   * @param option	the current option
   * @return		the last internal data structure that was generated
   */
  public I doProduce(AbstractOption option) {
    if (option instanceof BooleanOption) {
      if (isDebugOn())
	debug(generateDebugString("produce/boolean") + ": " + option);
      return processOption((BooleanOption) option);
    }
    else if (option instanceof ClassOption) {
      if (isDebugOn())
	debug(generateDebugString("produce/class") + ": " + option);
      return processOption((ClassOption) option);
    }
    else if (option instanceof AbstractArgumentOption){
      if (isDebugOn())
	debug(generateDebugString("produce/argument") + ": " + option);
      return processOption((AbstractArgumentOption) option);
    }
    else {
      throw new IllegalScalingException("Unhandled type of option: " + option.getClass().getName());
    }
  }

  /**
   * Hook method that gets called just after an option was produced.
   * <p/>
   * Default implementation does nothing
   *
   * @param manager	the option manager
   * @param index	the index of the option
   */
  protected void postProduce(OptionManager manager, int index) {
  }

  /**
   * Hook method for performing checks on the input. May throw exceptions
   * if object doesn't pass test(s).
   * <p/>
   * Default implementation does nothing.
   *
   * @param object	the objec to check
   * @return		the checked object
   */
  protected OptionHandler checkInput(OptionHandler object) {
    if (object == null)
      throw new IllegalStateException("Input is null!");
    return object;
  }

  /**
   * Hook-method before starting visiting options.
   * <p/>
   * Default implementation merely initializes m_Output.
   *
   * @see		#initOutput()
   */
  protected void preProduce() {
    if (m_Output == null)
      m_Output = initOutput();
  }

  /**
   * Visits the options and obtains information from them.
   *
   * @param manager	the manager to traverse
   */
  protected void doProduce(OptionManager manager) {
    List<AbstractOption>	options;
    int				i;

    options = manager.getOptionsList();
    for (i = 0; i < options.size(); i++) {
      if (isDebugOn())
	debug(generateDebugString("preProduce") + ": " + manager.getOwner().getClass().getName());
      preProduce(manager, i);

      if (isDebugOn())
	debug(generateDebugString("produce") + ": " + manager.getOwner().getClass().getName());
      m_LastGenerated = doProduce(options.get(i));
      if (getDebugLevel() > 1)
	debug(generateDebugString("produce") + " --> " + m_LastGenerated, 2);

      if (isDebugOn())
	debug(generateDebugString("postProduce") + ": " + manager.getOwner().getClass().getName());
      postProduce(manager, i);
    }
  }

  /**
   * Hook-method after visiting options.
   * <p/>
   * Default implementation does nothing.
   */
  protected void postProduce() {
  }

  /**
   * Visits the option and obtains information from it.
   *
   * @param visitedObject	the option handler to visit
   * @return			the generated output
   */
  public O produce(OptionHandler visitedObject) {
    reset();
    m_Input = checkInput(visitedObject);

    if (isDebugOn())
      debug("preVisit: " + m_Input.getClass().getName());
    preProduce();

    if (isDebugOn())
      debug("doVisit: " + m_Input.getClass().getName());
    doProduce(getInput().getOptionManager());

    if (isDebugOn())
      debug("postVisit: " + m_Input.getClass().getName());
    postProduce();

    return getOutput();
  }

  /**
   * Returns the output generated from the visit.
   *
   * @return		the output, null in case of an error
   */
  public abstract String toString();

  /**
   * Writes the generated content to the specified file.
   *
   * @param filename	the file to write to
   * @return		true if successfully written
   */
  public boolean write(String filename) {
    return FileUtils.writeToFile(filename, toString(), false);
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    m_Input = null;
    m_Output        = null;
    m_LastGenerated = null;
  }

  /**
   * Uses the specified producer to generate a string representation of the
   * given option handler.
   *
   * @param cls		the producer class to use
   * @param handler	the option handler to turn into a string representation
   * @return		the string representation, null in case of an error
   */
  public static String toString(Class<? extends AbstractOptionProducer> cls, OptionHandler handler) {
    String		result;
    AbstractOptionProducer	producer;

    result = null;

    try {
      producer = (AbstractOptionProducer) cls.newInstance();
      producer.produce(handler);
      result   = producer.toString();
      producer.cleanUp();
    }
    catch (Exception e) {
      e.printStackTrace();
    }

    return result;
  }

  /**
   * Uses the specified producer to process the option handler and return the
   * generated data structure.
   *
   * @param cls		the producer class to use
   * @param handler	the handler to process
   * @return		the generated data structure
   */
  public static Object produce(Class<? extends AbstractOptionProducer> cls, OptionHandler handler) {
    Object			result;
    AbstractOptionProducer	producer;

    result = null;

    try {
      producer = (AbstractOptionProducer) cls.newInstance();
      producer.produce(handler);
      result = producer.getOutput();
      producer.cleanUp();
    }
    catch (Exception e) {
      e.printStackTrace();
    }

    return result;
  }
}
