/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * ArrayProcess.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.control;

import java.lang.reflect.Array;
import java.util.Hashtable;

import adams.flow.core.AbstractActor;
import adams.flow.core.InputConsumer;
import adams.flow.core.OutputProducer;
import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Applies all sub-actors to each of the array elements.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: ArrayProcess
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-progress (property: showProgress)
 * &nbsp;&nbsp;&nbsp;If set to true, progress information will be output to stdout ('.').
 * </pre>
 *
 * <pre>-actor &lt;adams.flow.core.AbstractActor [options]&gt; [-actor ...] (property: actors)
 * &nbsp;&nbsp;&nbsp;All the actors that define this sequence.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-array-class &lt;java.lang.String&gt; (property: arrayClass)
 * &nbsp;&nbsp;&nbsp;The class to use for the array; if none is specified, the class of the first
 * &nbsp;&nbsp;&nbsp;element is used.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class ArrayProcess
  extends SubProcess {

  /** for serialization. */
  private static final long serialVersionUID = 5975989766824652946L;

  /** the key for storing the output token in the backup. */
  public final static String BACKUP_OUTPUT = "output";

  /** the class for the array. */
  protected String m_ArrayClass;

  /** the output array. */
  protected Token m_OutputToken;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Applies all sub-actors to each of the array elements.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "array-class", "arrayClass",
	    "");
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    result = "Class: ";
    variable = getOptionManager().getVariableForProperty("arrayClass");
    if (variable != null)
      result += variable;
    else if (m_ArrayClass.length() != 0)
      result += m_ArrayClass;
    else
      result += "-from 1st element-";

    return result;
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_OutputToken = null;
  }

  /**
   * Sets the class for the array.
   *
   * @param value	the classname, use empty string to use class of first
   * 			element
   */
  public void setArrayClass(String value) {
    m_ArrayClass = value;
    reset();
  }

  /**
   * Returns the class for the array.
   *
   * @return		the classname, empty string if class of first element
   * 			is used
   */
  public String getArrayClass() {
    return m_ArrayClass;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String arrayClassTipText() {
    return
        "The class to use for the array; if none is specified, the class of "
      + "the first element is used.";
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    if (m_OutputToken != null)
      result.put(BACKUP_OUTPUT, m_OutputToken);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_OUTPUT)) {
      m_OutputToken = (Token) state.get(BACKUP_OUTPUT);
      state.remove(BACKUP_OUTPUT);
    }

    super.restoreState(state);
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		the Class of objects that can be processed
   */
  public Class[] accepts() {
    Class[]	result;
    Class[]	sub;
    int		i;

    if (active() > 0) {
      sub    = ((InputConsumer) firstActive()).accepts();
      result = new Class[sub.length];
      for (i = 0; i < sub.length; i++)
	result[i] = Array.newInstance(sub[i], 0).getClass();
      return result;
    }
    else {
      return new Class[]{Unknown.class};
    }
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		the Class of the generated tokens
   */
  public Class[] generates() {
    Class[]	result;
    Class[]	sub;
    int		i;
    Class	cls;

    if (m_ArrayClass.length() > 0) {
      try {
	cls    = Class.forName(m_ArrayClass);
	result = new Class[]{Array.newInstance(cls, 0).getClass()};
      }
      catch (Exception e) {
	// ignored
	result = new Class[0];
      }
    }
    else if (active() > 0) {
      sub    = ((OutputProducer) lastActive()).generates();
      result = new Class[sub.length];
      for (i = 0; i < sub.length; i++)
	result[i] = Array.newInstance(sub[i], 0).getClass();
    }
    else {
      result = new Class[]{Unknown.class};
    }

    return result;
  }

  /**
   * The method that accepts the input token and then processes it.
   *
   * @param token	the token to accept and process
   */
  public void input(Token token) {
    m_CurrentToken = token;
    m_OutputToken  = null;
  }

  /**
   * Executes the actor.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    AbstractActor	first;
    int			len;
    int			i;
    Token		input;
    Object		output;

    result = null;

    first = firstActive();
    if (isDebugOn())
      debug("first active actor: " + first.getFullName());

    if ((first != null) && (first instanceof InputConsumer)) {
      len = Array.getLength(m_CurrentToken.getPayload());
      for (i = 0; i < len; i++) {
	input = new Token(Array.get(m_CurrentToken.getPayload(), i));
	((InputConsumer) first).input(input);
	if (getDebugLevel() > 1)
	  debug("input token #" + (i+1) + ": " + input, 2);

	try {
	  result = m_Director.execute();
	}
	catch (Exception e) {
	  getSystemErr().printStackTrace(e);
	  result = e.toString();
	}

	if (result != null)
	  break;
      }
    }

    if (!isStopped() && (result == null) && (m_OutputTokens.size() > 0)) {
      try {
	if (m_ArrayClass.length() == 0)
	  output = Array.newInstance(m_OutputTokens.get(0).getPayload().getClass(), m_OutputTokens.size());
	else
	  output = Array.newInstance(Class.forName(m_ArrayClass), m_OutputTokens.size());
	for (i = 0; i < m_OutputTokens.size(); i++)
	  Array.set(output, i, m_OutputTokens.get(i).getPayload());
	m_OutputToken = new Token(output);
      }
      catch (Exception e) {
	result        = "Failed to generate output array: " + e;
	m_OutputToken = null;
	getSystemErr().println(result);
	getSystemErr().printStackTrace(e);
      }
    }

    if (!isStopped())
      m_OutputTokens.clear();

    return result;
  }

  /**
   * Post-execute hook.
   *
   * @return		null if everything is fine, otherwise error message
   * @see		#m_Executed
   */
  protected String postExecute() {
    String	result;

    result = super.postExecute();

    if (isStopped())
      m_OutputToken = null;

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_OutputToken != null);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result        = m_OutputToken;
    m_OutputToken = null;

    return result;
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    m_OutputToken = null;

    super.wrapUp();
  }
}
