/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * GlobalActorHelper.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.core;

import java.util.Vector;

import adams.core.ConsoleObject;
import adams.flow.control.AbstractDirectedControlActor;
import adams.flow.standalone.GlobalActors;

/**
 * Helper class for global actors.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class GlobalActorHelper
  extends ConsoleObject {

  /** for serialization. */
  private static final long serialVersionUID = -763479272812116920L;

  /**
   * Checks a control actor's children whether they contain the global actor
   * that we're looking for.
   *
   * @param group	the group to check
   * @param name	the name of the global actor
   * @return		the global actor or null if not found
   */
  public AbstractActor findGlobalActor(ActorHandler group, GlobalActorReference name) {
    AbstractActor		result;
    int				i;
    GlobalActors		global;
    int				index;
    AbstractExternalActor	external;

    result = null;

    for (i = 0; i < group.size(); i++) {
      if (group.get(i) instanceof GlobalActors) {
	global = (GlobalActors) group.get(i);
	index  = global.indexOf(name.toString());
	if (index > -1) {
	  result = global.get(index);
	  break;
	}
      }
      else if (group.get(i) instanceof AbstractExternalActor) {
	external = (AbstractExternalActor) group.get(i);
	if (external.getExternalActor() instanceof ActorHandler) {
	  result = findGlobalActor((ActorHandler) external.getExternalActor(), name);
	  if (result != null)
	    break;
	}
      }
    }

    return result;
  }

  /**
   * Tries to find the global actor referenced by its global name.
   *
   * @param root	the root to search in
   * @param name	the name of the global actor
   * @return		the global actor or null if not found
   */
  public AbstractActor findGlobalActor(AbstractActor root, GlobalActorReference name) {
    AbstractActor	result;

    result = null;

    if (root == null) {
      getSystemErr().println("No root container found!");
    }
    else if (!(root instanceof AbstractDirectedControlActor)) {
      getSystemErr().println(
	  "Root is not a container ('" + root.getFullName() + "'/"
	  + root.getClass().getName() + ")!");
      root = null;
    }

    if (root != null)
      result = findGlobalActor((ActorHandler) root, name);

    return result;
  }

  /**
   * Tries to find the referenced global actor. First all possible actor
   * handlers are located recursively (up to the root) that allow also
   * singletons. This list of actors is then searched for the global actor.
   *
   * @param actor	the actor to start from
   * @param name	the name of the global actor
   * @return		the global actor or null if not found
   * @see		ActorUtils#findActorHandlers(AbstractActor, boolean)
   */
  public AbstractActor findGlobalActorRecursive(AbstractActor actor, GlobalActorReference name) {
    AbstractActor		result;
    Vector<ActorHandler>	handlers;
    int				i;

    result   = null;
    handlers = ActorUtils.findActorHandlers(actor, true);
    for (i = 0; i < handlers.size(); i++) {
      result = findGlobalActor(handlers.get(i), name);
      if (result != null)
	break;
    }

    return result;
  }
}
