/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SequenceToCollection.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.transformer;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Hashtable;
import java.util.Vector;

import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Turns a sequence of tokens into a collection.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;adams.flow.core.Unknown<br/>
 * - generates:<br/>
 * &nbsp;&nbsp;&nbsp;java.util.ArrayList<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: SequenceToCollection
 * </pre>
 * 
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default: 
 * </pre>
 * 
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded 
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 * 
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 * 
 * <pre>-size &lt;int&gt; (property: collectionSize)
 * &nbsp;&nbsp;&nbsp;The size of the collection(s).
 * &nbsp;&nbsp;&nbsp;default: 1
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 * 
 * <pre>-collection-class &lt;java.lang.String&gt; (property: collectionClass)
 * &nbsp;&nbsp;&nbsp;The class to use for the collection.
 * &nbsp;&nbsp;&nbsp;default: java.util.ArrayList
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4712 $
 */
public class SequenceToCollection
  extends AbstractTransformer {

  /** for serialization. */
  private static final long serialVersionUID = 1363005679574784724L;

  /** the key for storing the current elements in the backup. */
  public final static String BACKUP_ELEMENTS = "elements";

  /** the buffered elements of the array that still need to be broadcasted. */
  protected Vector m_Elements;

  /** the length of the arrays. */
  protected int m_CollectionSize;

  /** the class for the array. */
  protected String m_CollectionClass;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Turns a sequence of tokens into a collection.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "size", "collectionSize",
	    1, 1, null);

    m_OptionManager.add(
	    "collection-class", "collectionClass",
	    ArrayList.class.getName());
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    variable = getOptionManager().getVariableForProperty("collectionSize");
    if (variable != null)
      result = "Size: " + variable;
    else
      result = "Size: " + m_CollectionSize;

    result += ", Class: ";
    variable = getOptionManager().getVariableForProperty("collectionClass");
    if (variable != null)
      result += variable;
    else if (m_CollectionClass.length() != 0)
      result += m_CollectionClass;
    else
      result += "-none-";

    return result;
  }

  /**
   * Sets the size of the collection.
   *
   * @param value	the size
   */
  public void setCollectionSize(int value) {
    if (value < 1) {
      getSystemErr().println("Collection(s) must have a size of at least 1, provided: " + value + "!");
    }
    else {
      m_CollectionSize = value;
      reset();
    }
  }

  /**
   * Returns the length of the collection.
   *
   * @return		the size
   */
  public int getCollectionSize() {
    return m_CollectionSize;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String collectionSizeTipText() {
    return "The size of the collection(s).";
  }

  /**
   * Sets the class for the collection.
   *
   * @param value	the classname, use empty string to use class of first
   * 			element
   */
  public void setCollectionClass(String value) {
    if (value.length() > 0) {
      m_CollectionClass = value;
      reset();
    }
    else {
      getSystemErr().println("Class cannot be empty!");
    }
  }

  /**
   * Returns the class for the collection.
   *
   * @return		the classname
   */
  public String getCollectionClass() {
    return m_CollectionClass;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String collectionClassTipText() {
    return "The class to use for the collection.";
  }

  /**
   * Removes entries from the backup.
   */
  protected void pruneBackup() {
    super.pruneBackup();

    pruneBackup(BACKUP_ELEMENTS);
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    result.put(BACKUP_ELEMENTS, m_Elements);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_ELEMENTS)) {
      m_Elements = (Vector) state.get(BACKUP_ELEMENTS);
      state.remove(BACKUP_ELEMENTS);
    }

    super.restoreState(state);
  }

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_Elements    = new Vector();
    m_OutputToken = null;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->adams.flow.core.Unknown.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		<!-- flow-generates-start -->java.util.ArrayList.class<!-- flow-generates-end -->
   */
  public Class[] generates() {
    try {
      return new Class[]{Class.forName(m_CollectionClass)};
    }
    catch (Exception e) {
      return new Class[]{Collection.class};
    }
  }

  /**
   * Executes the flow item.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String	result;
    Collection	coll;

    result = null;

    try {
      m_Elements.add(m_InputToken.getPayload());
      debug("Buffered elements: " + m_Elements.size());
      if (m_Elements.size() == m_CollectionSize) {
	coll = (Collection) Class.forName(m_CollectionClass).newInstance();
	debug("Collection type: " + coll.getClass().getComponentType());
	coll.addAll(m_Elements);
	m_OutputToken = new Token(coll);
	m_Elements.clear();
	debug("Collection generated");
      }
    }
    catch (Exception e) {
      getSystemErr().printStackTrace(e);
      result = e.toString();
    }

    return result;
  }
}
