/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * DataContainerFileChooser.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.chooser;

import java.awt.BorderLayout;
import java.awt.Component;
import java.io.File;
import java.util.Collections;
import java.util.Hashtable;
import java.util.Vector;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JLabel;
import javax.swing.JPanel;

import adams.core.io.PlaceholderFile;
import adams.data.container.DataContainer;
import adams.data.io.input.AbstractDataContainerReader;
import adams.data.io.output.AbstractDataContainerWriter;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.goe.GenericObjectEditor;
import adams.gui.goe.GenericObjectEditorDialog;

/**
 * A specialized JFileChooser that lists all available file Readers and Writers
 * for data containers.
 * <p/>
 * Based on <code>weka.gui.ConverterFileChooser</code>
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 * @see	    weka.gui.ConverterFileChooser
 * @param <T> the type of container
 */
public abstract class AbstractDataContainerFileChooser<T extends DataContainer>
  extends AbstractExtensionFileFilterFileChooser<AbstractDataContainerFileChooser.DataContainerFileExtensionFilter> {

  /** for serialization. */
  private static final long serialVersionUID = -5373058011025481738L;

  /**
   * A custom filter class that stores the associated class along the
   * description and extensions.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4584 $
   */
  public static class DataContainerFileExtensionFilter
    extends ExtensionFileFilter {

    /** for serialization. */
    private static final long serialVersionUID = 5863117558505811134L;

    /** the classname. */
    protected String m_Classname;

    /**
     * Constructs a filter that matches all files.
     *
     * @param classname		the classname this filter is for
     */
    public DataContainerFileExtensionFilter(String classname) {
      super();

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extension		the extensions of the files (no dot!)
     */
    public DataContainerFileExtensionFilter(String classname, String description, String extension) {
      super(description, extension);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extensions	the extensions of the files (no dot!)
     */
    public DataContainerFileExtensionFilter(String classname, String description, String[] extensions) {
      super(description, extensions);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extension		the extensions of the files (no dot!)
     * @param caseSensitive	if true then the filter is case-sensitive
     */
    public DataContainerFileExtensionFilter(String classname, String description, String extension, boolean caseSensitive) {
      super(description, extension, caseSensitive);

      m_Classname = classname;
    }

    /**
     * Constructs a filter that matches files with the given extension, not
     * case-sensitive.
     *
     * @param classname		the classname this filter is for
     * @param description	the display string
     * @param extensions	the extensions of the files (no dot!)
     * @param caseSensitive	if true then the filter is case-sensitive
     */
    public DataContainerFileExtensionFilter(String classname, String description, String[] extensions, boolean caseSensitive) {
      super(description, extensions, caseSensitive);

      m_Classname = classname;
    }

    /**
     * Returns the associated classname.
     *
     * @return		the classname
     */
    public String getClassname() {
      return m_Classname;
    }
  }

  /** the file filters for the readers. */
  protected static Hashtable<Class,Vector<DataContainerFileExtensionFilter>> m_ReaderFileFilters = new Hashtable<Class,Vector<DataContainerFileExtensionFilter>>();

  /** the file filters for the writers. */
  protected static Hashtable<Class,Vector<DataContainerFileExtensionFilter>> m_WriterFileFilters = new Hashtable<Class,Vector<DataContainerFileExtensionFilter>>();

  /** the configure button. */
  protected JButton m_ConfigureButton;

  /** the checkbox for bringing up the GenericObjectEditor. */
  protected JCheckBox m_CheckBoxOptions;

  /** the note about the options dialog. */
  protected JLabel m_LabelOptions;

  /** the GOE for displaying the options of a reader/writer. */
  protected GenericObjectEditor m_Editor;

  /**
   * Constructs a FileChooser pointing to the user's default directory.
   */
  protected AbstractDataContainerFileChooser() {
    super();
  }

  /**
   * Constructs a FileChooser using the given File as the path.
   *
   * @param currentDirectory	the path to start in
   */
  protected AbstractDataContainerFileChooser(File currentDirectory) {
    super(currentDirectory);
  }

  /**
   * Constructs a FileChooser using the given path.
   *
   * @param currentDirectory	the path to start in
   */
  protected AbstractDataContainerFileChooser(String currentDirectory) {
    super(currentDirectory);
  }

  /**
   * Further initializations.
   */
  protected void initialize() {
    JPanel	panel;
    JPanel	panel2;

    super.initialize();

    m_CheckBoxOptions = new JCheckBox("Invoke options dialog");
    m_CheckBoxOptions.setMnemonic('I');
    m_LabelOptions = new JLabel("<html><br>Note:<br><br>Some file formats offer additional<br>options which can be customized<br>when invoking the options dialog.</html>");
    panel = new JPanel(new BorderLayout());
    panel.add(m_CheckBoxOptions, BorderLayout.NORTH);
    panel2 = new JPanel(new BorderLayout());
    panel2.add(m_LabelOptions, BorderLayout.NORTH);
    panel2.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));
    panel.add(panel2, BorderLayout.CENTER);
    setAccessory(panel);

    m_Editor = new GenericObjectEditor(false);
  }

  /**
   * initializes the DataContainerFileExtensionFilters.
   *
   * @param chooser	the chooser instance to use as reference
   * @param reader	if true then the reader filters are initialized
   * @param classnames	the classnames of the converters
   */
  protected static void initFilters(AbstractDataContainerFileChooser chooser, boolean reader, String[] classnames) {
    int					i;
    String 				classname;
    Class 				cls;
    String[] 				ext;
    String 				desc;
    Object		 		converter;
    DataContainerFileExtensionFilter 	filter;

    if (reader)
      m_ReaderFileFilters.put(chooser.getClass(), new Vector<DataContainerFileExtensionFilter>());
    else
      m_WriterFileFilters.put(chooser.getClass(), new Vector<DataContainerFileExtensionFilter>());

    for (i = 0; i < classnames.length; i++) {
      classname = (String) classnames[i];

      // get data from converter
      try {
	cls       = Class.forName(classname);
	converter = cls.newInstance();
	if (reader) {
	  desc = ((AbstractDataContainerReader) converter).getFormatDescription();
	  ext  = ((AbstractDataContainerReader) converter).getFormatExtensions();
	}
	else {
	  desc = ((AbstractDataContainerWriter) converter).getFormatDescription();
	  ext  = ((AbstractDataContainerWriter) converter).getFormatExtensions();
	}
      }
      catch (Exception e) {
	cls       = null;
	converter = null;
	ext       = new String[0];
	desc      = "";
      }

      if (converter == null)
	continue;

      // reader?
      if (reader) {
	filter = new DataContainerFileExtensionFilter(classname, desc, ext);
	m_ReaderFileFilters.get(chooser.getClass()).add(filter);
      }
      else {
	filter = new DataContainerFileExtensionFilter(classname, desc, ext);
	m_WriterFileFilters.get(chooser.getClass()).add(filter);
      }
    }

    if (reader)
      Collections.sort(m_ReaderFileFilters.get(chooser.getClass()));
    else
      Collections.sort(m_WriterFileFilters.get(chooser.getClass()));
  }

  /**
   * Returns the file filters for opening files.
   *
   * @return		the file filters
   */
  protected Vector<DataContainerFileExtensionFilter> getOpenFileFilters() {
    return m_ReaderFileFilters.get(getClass());
  }

  /**
   * Returns the file filters for writing files.
   *
   * @return		the file filters
   */
  protected Vector<DataContainerFileExtensionFilter> getSaveFileFilters() {
    return m_WriterFileFilters.get(getClass());
  }

  /**
   * initializes the GUI.
   *
   * @param dialogType		the type of dialog to setup the GUI for
   */
  protected void initGUI(int dialogType) {
    super.initGUI(dialogType);

    // initial setup
    if (dialogType == OPEN_DIALOG) {
      m_Editor.setClassType(AbstractDataContainerReader.class);
      m_Editor.setValue(getDefaultReader());
    }
    else {
      m_Editor.setClassType(AbstractDataContainerWriter.class);
      m_Editor.setValue(getDefaultWriter());
    }
    restoreLastFilter(dialogType);
  }

  /**
   * Returns the default reader.
   *
   * @return		the default reader
   */
  protected abstract AbstractDataContainerReader<T> getDefaultReader();

  /**
   * Returns the default writer.
   *
   * @return		the default writer
   */
  protected abstract AbstractDataContainerWriter<T> getDefaultWriter();

  /**
   * Pops up an "Open File" file chooser dialog.
   *
   * @param parent		the parent of this file chooser
   * @return			the result of the user's action
   */
  public int showOpenDialog(Component parent) {
    int result = super.showOpenDialog(parent);

    if (result == APPROVE_OPTION) {
      // bring up options dialog?
      if (m_CheckBoxOptions.isSelected()) {
	m_Editor.setValue(m_CurrentHandler);
	GenericObjectEditorDialog dialog = GenericObjectEditorDialog.createDialog(this, m_Editor);
	dialog.setVisible(true);
	result = dialog.getResult();
	if (result == GenericObjectEditorDialog.APPROVE_OPTION)
	  m_CurrentHandler = m_Editor.getValue();
      }
    }

    return result;
  }

  /**
   * Pops up an "Save File" file chooser dialog.
   *
   * @param parent		the parent of this file chooser
   * @return			the result of the user's action
   */
  public int showSaveDialog(Component parent) {
    int result = super.showSaveDialog(parent);

    if (result == APPROVE_OPTION) {
      // bring up options dialog?
      if (m_CheckBoxOptions.isSelected()) {
	m_Editor.setValue(m_CurrentHandler);
	GenericObjectEditorDialog dialog = GenericObjectEditorDialog.createDialog(this, m_Editor);
	dialog.setVisible(true);
	result = dialog.getResult();
	if (result == GenericObjectEditorDialog.APPROVE_OPTION)
	  m_CurrentHandler = m_Editor.getValue();
      }
    }

    return result;
  }

  /**
   * returns the reader that was chosen by the user, can be null in case the
   * user aborted the dialog or the save dialog was shown.
   *
   * @return		the chosen reader, if any
   */
  public AbstractDataContainerReader getReader() {
    configureCurrentHandlerHook(OPEN_DIALOG);

    if (m_CurrentHandler instanceof AbstractDataContainerWriter)
      return null;
    else
      return (AbstractDataContainerReader) m_CurrentHandler;
  }

  /**
   * returns the writer that was chosen by the user, can be null in case the
   * user aborted the dialog or the open dialog was shown.
   *
   * @return		the chosen writer, if any
   */
  public AbstractDataContainerWriter getWriter() {
    configureCurrentHandlerHook(SAVE_DIALOG);

    if (m_CurrentHandler instanceof AbstractDataContainerReader)
      return null;
    else
      return (AbstractDataContainerWriter) m_CurrentHandler;
  }

  /**
   * sets the current converter according to the current filefilter.
   */
  protected void updateCurrentHandlerHook() {
    String	classname;
    Object	newHandler;
    boolean	onlyFiles;

    try {
      // determine current converter
      classname  = ((DataContainerFileExtensionFilter) getFileFilter()).getClassname();
      newHandler = Class.forName(classname).newInstance();

      if (m_CurrentHandler == null) {
	m_CurrentHandler = newHandler;
      }
      else {
	if (!m_CurrentHandler.getClass().equals(newHandler.getClass()))
	  m_CurrentHandler = newHandler;
      }

      // files or directories?
      if (m_DialogType == OPEN_DIALOG)
	onlyFiles = ((AbstractDataContainerReader) m_CurrentHandler).isInputFile();
      else
	onlyFiles = ((AbstractDataContainerWriter) m_CurrentHandler).isOutputFile();
      if (onlyFiles)
	setFileSelectionMode(FILES_ONLY);
      else
	setFileSelectionMode(DIRECTORIES_ONLY);
    }
    catch (Exception e) {
      m_CurrentHandler = null;
      e.printStackTrace();
    }
  }

  /**
   * configures the current converter.
   *
   * @param dialogType		the type of dialog to configure for
   */
  protected void configureCurrentHandlerHook(int dialogType) {
    PlaceholderFile	selFile;
    String		classname;
    File		currFile;
    boolean		onlyFiles;

    selFile = getSelectedPlaceholderFile();

    if (m_CurrentHandler == null) {
      classname = ((DataContainerFileExtensionFilter) getFileFilter()).getClassname();
      try {
	m_CurrentHandler = Class.forName(classname).newInstance();
      }
      catch (Exception e) {
	e.printStackTrace();
	m_CurrentHandler = null;
      }

      // none found?
      if (m_CurrentHandler == null)
	return;
    }

    // wrong type?
    if (m_CurrentHandler instanceof AbstractDataContainerReader)
      onlyFiles = ((AbstractDataContainerReader) m_CurrentHandler).isInputFile();
    else
      onlyFiles = ((AbstractDataContainerWriter) m_CurrentHandler).isOutputFile();
    if ((onlyFiles && selFile.isDirectory()) || (!onlyFiles && !selFile.isDirectory()))
      return;

    try {
      if (m_CurrentHandler instanceof AbstractDataContainerReader)
	currFile = ((AbstractDataContainerReader) m_CurrentHandler).getInput();
      else
	currFile = ((AbstractDataContainerWriter) m_CurrentHandler).getOutput();
      if ((currFile == null) || (!currFile.getAbsolutePath().equals(selFile.getAbsolutePath()))) {
	if (m_CurrentHandler instanceof AbstractDataContainerReader)
	  ((AbstractDataContainerReader) m_CurrentHandler).setInput(selFile);
	else
	  ((AbstractDataContainerWriter) m_CurrentHandler).setOutput(selFile);
      }
    }
    catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Returns whether the filters have already been initialized.
   *
   * @return		true if the filters have been initialized
   */
  protected boolean getFiltersInitialized() {
    return (m_ReaderFileFilters.containsKey(getClass()));
  }
}
