/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * AbstractEditPostProcessor.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.flow.tree.postprocessor;

import javax.swing.JOptionPane;

import adams.core.ClassLister;
import adams.core.ConsoleObject;
import adams.flow.core.AbstractActor;
import adams.gui.flow.tree.Tree;

/**
 * Ancestor for post-processors for edits in the tree.
 * 
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4701 $
 */
public abstract class AbstractEditPostProcessor
  extends ConsoleObject {

  /** for serialization. */
  private static final long serialVersionUID = -1710283049516306816L;

  /**
   * Checks whether this post processor scheme applies to the current situation.
   * 
   * @param parent	the parent actor
   * @param oldActor	the old actor
   * @param newActor	the new, updated actor
   * @return		true if this post processor applies to the situation
   */
  public abstract boolean applies(AbstractActor parent, AbstractActor oldActor, AbstractActor newActor);
  
  /**
   * Post-processes the tree.
   * 
   * @param tree	the tree to post-process
   * @param parent	the parent actor
   * @param oldActor	the old actor
   * @param newActor	the new, updated actor
   * @return		true if tree got modified
   */
  public abstract boolean postProcess(Tree tree, AbstractActor parent, AbstractActor oldActor, AbstractActor newActor);

  /**
   * Returns a list with classnames of post-processors.
   *
   * @return		the post-processor classnames
   */
  public static String[] getPostProcessors() {
    return ClassLister.getSingleton().getClassnames(AbstractEditPostProcessor.class);
  }
  
  /**
   * Applies all the post-processors, if applicable.
   * 
   * @param tree	the tree that was modified
   * @param parent	the parent actor of the modified actor
   * @param oldActor	the old actor
   * @param newActor	the new, updated actor
   * @return		true if tree got modified
   */
  public static boolean apply(Tree tree, AbstractActor parent, AbstractActor oldActor, AbstractActor newActor) {
    boolean			result;
    String[]			processors;
    AbstractEditPostProcessor	proc;
    boolean			confirmed;
    
    result = false;
    
    confirmed = false;
    processors = getPostProcessors();
    for (String processor: processors) {
      try {
	proc = (AbstractEditPostProcessor) Class.forName(processor).newInstance();
	if (proc.applies(parent, oldActor, newActor)) {
	  if (!confirmed) {
	    if (JOptionPane.showConfirmDialog(tree, "Propagate changes throughout the tree (if applicable)?") == JOptionPane.YES_OPTION)
	      confirmed = true;
	    else
	      break;
	  }
	  result = result | proc.postProcess(tree, parent, oldActor, newActor);
	}
      }
      catch (Exception e) {
	System.err.println("Error applying edit post-processor '" + processor + "':");
	e.printStackTrace();
      }
    }

    return result;
  }
}
