/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ContentPanel.java
 * Copyright (C) 2008-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.visualization.core.plot;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.text.DecimalFormat;
import java.util.HashSet;
import java.util.Iterator;
import java.util.Vector;

import javax.swing.JMenuItem;
import javax.swing.JPopupMenu;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.gui.core.BasePanel;
import adams.gui.core.MouseUtils;
import adams.gui.event.PaintEvent.PaintMoment;
import adams.gui.visualization.core.AxisPanel;
import adams.gui.visualization.core.PlotPanel;
import adams.gui.visualization.core.PopupMenuCustomizer;
import adams.gui.visualization.core.axis.Tick;
import adams.gui.visualization.core.axis.Visibility;

/**
 * A specialized panel that can notify listeners of paint updates.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4760 $
 */
public class ContentPanel
  extends BasePanel
  implements ChangeListener {

  /** the owner of the panel. */
  protected PlotPanel m_Owner;

  /** the panel itself. */
  protected ContentPanel m_Self;

  /** for serialization. */
  private static final long serialVersionUID = 1785560399953428177L;

  /** for outputting the values. */
  protected DecimalFormat m_Format;

  /** whether zooming is enabled. */
  protected boolean m_ZoomingEnabled;

  /** the color of the zoom box. */
  protected Color m_ZoomBoxColor;

  /** whether the zoom box is currently been drawn. */
  protected boolean m_Zooming;

  /** whether dragging has happened at all. */
  protected boolean m_Dragged;

  /** the top left corner of the zoom box. */
  protected Point m_ZoomTopLeft;

  /** the bottom right corner of the zoom box. */
  protected Point m_ZoomBottomRight;

  /** an optional customizer for the right-click popup. */
  protected PopupMenuCustomizer m_PopupMenuCustomizer;

  /** whether panning is enabled. */
  protected boolean m_PanningEnabled;

  /** whether the graph is currently moved around. */
  protected boolean m_Panning;

  /** the starting mouse position of panning. */
  protected Point m_PanningStart;

  /** the original pixel offset for the left axis. */
  protected int m_LeftPixelOffset;

  /** the original pixel offset for the right axis. */
  protected int m_RightPixelOffset;

  /** the original pixel offset for the top axis. */
  protected int m_TopPixelOffset;

  /** the original pixel offset for the bottom axis. */
  protected int m_BottomPixelOffset;

  /** for post-processing the tiptext. */
  protected TipTextCustomizer m_TipTextCustomizer;

  /** the hit detectors. */
  protected HashSet<AbstractHitDetector> m_HitDetectors;

  /**
   * Initializes the panel.
   *
   * @param owner	the plot panel this panel belongs to
   */
  public ContentPanel(PlotPanel owner) {
    super();

    m_Owner = owner;
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Self                = this;
    m_Format              = new DecimalFormat("0.0E0;-0.0E0");
    m_Zooming             = false;
    m_Dragged             = false;
    m_ZoomBoxColor        = Color.GRAY;
    m_PopupMenuCustomizer = null;
    m_Panning             = false;
    m_TipTextCustomizer   = null;
    m_HitDetectors        = new HashSet<AbstractHitDetector>();
    m_ZoomingEnabled      = true;
    m_PanningEnabled      = true;
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    setToolTipText("");  // in order to enable the tooltip

    addMouseListener(new MouseAdapter() {
      // start zoom
      public void mousePressed(MouseEvent e) {
        super.mousePressed(e);

        if (e.getButton() == MouseEvent.BUTTON1) {
          // get top/left coordinates for zoom
          if (!e.isShiftDown()) {
            if (m_ZoomingEnabled) {
              m_Zooming     = true;
              m_Dragged     = false;
              m_ZoomTopLeft = e.getPoint();
            }
          }
          // get start position of panning
          else if (e.isShiftDown()) {
            if (m_PanningEnabled) {
              m_Panning           = true;
              m_PanningStart      = e.getPoint();
              m_LeftPixelOffset   = getOwner().getAxis(Axis.LEFT).getPixelOffset();
              m_RightPixelOffset  = getOwner().getAxis(Axis.RIGHT).getPixelOffset();
              m_TopPixelOffset    = getOwner().getAxis(Axis.TOP).getPixelOffset();
              m_BottomPixelOffset = getOwner().getAxis(Axis.TOP).getPixelOffset();
            }
          }
        }
      }

      // perform zoom/panning
      public void mouseReleased(MouseEvent e) {
        super.mouseReleased(e);

        if (e.getButton() == MouseEvent.BUTTON1) {
          // get bottom/right coordinates for zoom
          if (m_Zooming && m_Dragged) {
            m_Zooming         = false;
            m_Dragged         = false;
            m_ZoomBottomRight = e.getPoint();

            addZoom(
        	(int) m_ZoomTopLeft.getY(),
        	(int) m_ZoomTopLeft.getX(),
        	(int) m_ZoomBottomRight.getY(),
        	(int) m_ZoomBottomRight.getX());
          }
          else if (m_Panning) {
            m_Panning  = false;
            int deltaX = (int) e.getX() - (int) m_PanningStart.getX();
            int deltaY = (int) m_PanningStart.getY() - (int) e.getY();

            // update pixel offset
            getOwner().getAxis(Axis.LEFT).setPixelOffset(m_LeftPixelOffset + deltaY);
            getOwner().getAxis(Axis.RIGHT).setPixelOffset(m_RightPixelOffset + deltaY);
            getOwner().getAxis(Axis.TOP).setPixelOffset(m_TopPixelOffset + deltaX);
            getOwner().getAxis(Axis.BOTTOM).setPixelOffset(m_BottomPixelOffset + deltaX);

            repaint();
          }
        }
        
        m_Zooming = false;
        m_Panning = false;
      }

      // popup menu/hits
      public void mouseClicked(MouseEvent e) {
        if (!MouseUtils.isPrintScreenClick(e)) {
          if (MouseUtils.isRightClick(e)) {
            JPopupMenu menu = getPopupMenu(e);
            if (menu != null)
              menu.show(m_Self, e.getX(), e.getY());
          }
          else if ((e.getButton() == MouseEvent.BUTTON1) && (e.getClickCount() == 1)) {
            detectHits(e);
          }
        }
      }
    });
    addMouseMotionListener(new MouseMotionAdapter() {
      // for zooming
      public void mouseDragged(MouseEvent e) {
        super.mouseDragged(e);

        // update zoom box
        if (m_Zooming && !e.isShiftDown()) {
          m_Dragged         = true;
          m_ZoomBottomRight = e.getPoint();

          repaint();
        }
        else if (m_Panning && e.isShiftDown()) {
          int deltaX = (int) e.getX() - (int) m_PanningStart.getX();
          int deltaY = (int) m_PanningStart.getY() - (int) e.getY();

          // update pixel offset
          getOwner().getAxis(Axis.LEFT).setPixelOffset(m_LeftPixelOffset + deltaY);
          getOwner().getAxis(Axis.RIGHT).setPixelOffset(m_RightPixelOffset + deltaY);
          getOwner().getAxis(Axis.TOP).setPixelOffset(m_TopPixelOffset + deltaX);
          getOwner().getAxis(Axis.BOTTOM).setPixelOffset(m_BottomPixelOffset + deltaX);

          repaint();
        }
      }
    });

    // making it focusable
    setFocusable(true);
    addMouseListener(new MouseAdapter() {
      public void mouseClicked(MouseEvent e) {
        requestFocusInWindow();
        super.mouseClicked(e);
      }
    });
  }

  /**
   * Returns the plot panel this panel belongs to.
   *
   * @return		the owning panel
   */
  public PlotPanel getOwner() {
    return m_Owner;
  }

  /**
   * Sets whether zooming is enabled or not.
   *
   * @param value	if true then zooming is enabled
   */
  public void setZoomingEnabled(boolean value) {
    m_ZoomingEnabled = value;
    if (!m_ZoomingEnabled)
      clearZoom();
  }

  /**
   * Returns whether zooming is enabled.
   *
   * @return		true if zooming is enabled
   */
  public boolean isZoomingEnabled() {
    return m_ZoomingEnabled;
  }

  /**
   * Pops the current zoom.
   */
  public void popZoom() {
    if (getOwner().isZoomed()) {
      getOwner().getAxis(Axis.TOP).popZoom();
      getOwner().getAxis(Axis.LEFT).popZoom();
      getOwner().getAxis(Axis.BOTTOM).popZoom();
      getOwner().getAxis(Axis.RIGHT).popZoom();
    }
  }

  /**
   * Clears the zoom.
   */
  public void clearZoom() {
    getOwner().getAxis(Axis.TOP).clearZoom();
    getOwner().getAxis(Axis.LEFT).clearZoom();
    getOwner().getAxis(Axis.BOTTOM).clearZoom();
    getOwner().getAxis(Axis.RIGHT).clearZoom();
  }

  /**
   * Sets whether panning is enabled or not.
   *
   * @param value	if true then panning is enabled
   */
  public void setPanningEnabled(boolean value) {
    m_PanningEnabled = value;
    // clear panning
    if (!m_PanningEnabled)
      clearPanning();
  }

  /**
   * Returns whether panning is enabled.
   *
   * @return		true if panning is enabled
   */
  public boolean isPanningEnabled() {
    return m_PanningEnabled;
  }

  /**
   * Clears the panning.
   */
  public void clearPanning() {
    getOwner().getAxis(Axis.TOP).clearPanning();
    getOwner().getAxis(Axis.LEFT).clearPanning();
    getOwner().getAxis(Axis.BOTTOM).clearPanning();
    getOwner().getAxis(Axis.RIGHT).clearPanning();
  }

  /**
   * swaps points if necessary, s.t., first is the smaller one.
   *
   * @param value1	the first value
   * @param value2	the second value
   * @return		the ordered values
   */
  protected int[] order(int value1, int value2) {
    int[]	result;

    if (value1 < value2)
      result = new int[]{value1, value2};
    else
      result = new int[]{value2, value1};

    return result;
  }

  /**
   * swaps points if necessary, s.t., first is the smaller one.
   *
   * @param value1	the first value
   * @param value2	the second value
   * @return		the ordered values
   */
  protected double[] order(double value1, double value2) {
    double[]	result;

    if (value1 < value2)
      result = new double[]{value1, value2};
    else
      result = new double[]{value2, value1};

    return result;
  }

  /**
   * Adds a zoom.
   *
   * @param top	the top value
   * @param left	the left value
   * @param bottom	the bottom value
   * @param right	the right value
   */
  public void addZoom(double top, double left, double bottom, double right) {
    double[] 	y;
    double[] 	x;

    y = order(top, bottom);
    x = order(left, right);

    // update axes
    if (   getOwner().getAxis(Axis.LEFT).canZoom(y[0], y[1])
	&& getOwner().getAxis(Axis.RIGHT).canZoom(y[0], y[1])
	&& getOwner().getAxis(Axis.TOP).canZoom(x[0], x[1])
	&& getOwner().getAxis(Axis.BOTTOM).canZoom(x[0], x[1]) ) {
      getOwner().getAxis(Axis.LEFT).pushZoom(y[0], y[1]);
      getOwner().getAxis(Axis.RIGHT).pushZoom(y[0], y[1]);
      getOwner().getAxis(Axis.TOP).pushZoom(x[0], x[1]);
      getOwner().getAxis(Axis.BOTTOM).pushZoom(x[0], x[1]);
    }
    else {
      System.err.println("No further zoom possible!");
    }

    repaint();
  }

  /**
   * Adds a zoom.
   *
   * @param top	the top position
   * @param left	the left position
   * @param bottom	the bottom position
   * @param right	the right position
   */
  public void addZoom(int top, int left, int bottom, int right) {
    int[] 	y;
    int[] 	x;

    y = order(top, bottom);
    x = order(left, right);

    // update axes
    if (   getOwner().getAxis(Axis.LEFT).canZoom(
	    getOwner().getAxis(Axis.LEFT).posToValue(y[1]),
	    getOwner().getAxis(Axis.LEFT).posToValue(y[0]))
	&& getOwner().getAxis(Axis.RIGHT).canZoom(
	    getOwner().getAxis(Axis.RIGHT).posToValue(y[1]),
	    getOwner().getAxis(Axis.RIGHT).posToValue(y[0]))
	&& getOwner().getAxis(Axis.TOP).canZoom(
	    getOwner().getAxis(Axis.TOP).posToValue(x[0]),
	    getOwner().getAxis(Axis.TOP).posToValue(x[1]))
	&& getOwner().getAxis(Axis.BOTTOM).canZoom(
	    getOwner().getAxis(Axis.BOTTOM).posToValue(x[0]),
	    getOwner().getAxis(Axis.BOTTOM).posToValue(x[1])) ) {
      getOwner().getAxis(Axis.LEFT).pushZoom(
	  getOwner().getAxis(Axis.LEFT).posToValue(y[1]),
	  getOwner().getAxis(Axis.LEFT).posToValue(y[0]));
      getOwner().getAxis(Axis.RIGHT).pushZoom(
	  getOwner().getAxis(Axis.RIGHT).posToValue(y[1]),
	  getOwner().getAxis(Axis.RIGHT).posToValue(y[0]));
      getOwner().getAxis(Axis.TOP).pushZoom(
	  getOwner().getAxis(Axis.TOP).posToValue(x[0]),
	  getOwner().getAxis(Axis.TOP).posToValue(x[1]));
      getOwner().getAxis(Axis.BOTTOM).pushZoom(
	  getOwner().getAxis(Axis.BOTTOM).posToValue(x[0]),
	  getOwner().getAxis(Axis.BOTTOM).posToValue(x[1]));
    }
    else {
      System.err.println("No further zoom possible!");
    }

    repaint();
  }

  /**
   * Sets the color for the zoom box.
   *
   * @param value	the color to use
   */
  public void setZoomBoxColor(Color value) {
    m_ZoomBoxColor = value;
    if (m_Zooming)
      repaint();
  }

  /**
   * Returns the color for the zoom box currently in use.
   *
   * @return		the color in use
   */
  public Color getZoomBoxColor() {
    return m_ZoomBoxColor;
  }

  /**
   * Sets the class to customize the right-click popup menu.
   *
   * @param value	the customizer
   */
  public void setPopupMenuCustomizer(PopupMenuCustomizer value) {
    m_PopupMenuCustomizer = value;
  }

  /**
   * Returns the current customizer, can be null.
   *
   * @return		the customizer
   */
  public PopupMenuCustomizer getPopupMenuCustomizer() {
    return m_PopupMenuCustomizer;
  }

  /**
   * Returns the popup menu, potentially customized.
   *
   * @param e		the mouse event
   * @return		the popup menu
   * @see		#m_PopupMenuCustomizer
   */
  public JPopupMenu getPopupMenu(MouseEvent e) {
    JPopupMenu		result;
    JMenuItem		item;

    result = null;

    if (m_ZoomingEnabled) {
      if (result == null)
	result = new JPopupMenu();

      item = new JMenuItem("Zoom out");
      item.setEnabled(getOwner().isZoomed());
      item.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  popZoom();
	  repaint();
	}
      });
      result.add(item);

      item = new JMenuItem("Clear zoom");
      item.setEnabled(getOwner().isZoomed());
      item.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  clearZoom();
	  repaint();
	}
      });
      result.add(item);
    }

    if (m_PanningEnabled) {
      if (result == null)
	result = new JPopupMenu();

      item = new JMenuItem("Undo panning");
      item.setEnabled(getOwner().isPanned());
      item.addActionListener(new ActionListener() {
	public void actionPerformed(ActionEvent e) {
	  clearPanning();
	  repaint();
	}
      });
      result.add(item);
    }

    // customize it?
    if (m_PopupMenuCustomizer != null)
      m_PopupMenuCustomizer.customizePopupMenu(e, result);

    return result;
  }

  /**
   * Clears the background.
   *
   * @param g		the graphics context
   */
  protected void clearBackground(Graphics g) {
    g.setColor(getOwner().getBackgroundColor());
    g.fillRect(0, 0, getWidth() - 1, getHeight() - 1);
  }

  /**
   * Paints the coordinates grid according to the ticks of the axes.
   *
   * @param g		the graphics context
   */
  protected void paintCoordinatesGrid(Graphics g) {
    AxisPanel		panel;
    Vector<Tick>	ticks;
    Tick		tick;
    int		i;
    boolean		vertical;
    int		pos;

    g.setColor(getOwner().getGridColor());

    for (Axis axis: Axis.values()) {
      if (!getOwner().getAxis(axis).getShowGridLines())
        continue;
      if (getOwner().getAxisVisibility(axis) == Visibility.INVISIBLE)
        continue;

      panel    = getOwner().getAxis(axis);
      ticks    = panel.getAxisModel().getTicks();
      vertical = ((axis == Axis.LEFT) || (axis == Axis.RIGHT));

      for (i = 0; i < ticks.size(); i++) {
        tick = ticks.get(i);

        if (tick.isMinor() && panel.getShowOnlyMajorGridLines())
          continue;

        pos = panel.correctPosition(tick.getPosition());
        if (vertical)
          g.drawLine(0, pos, getWidth(), pos);
        else
          g.drawLine(pos, 0, pos, getHeight());
      }
    }
  }

  /**
   * Paints the zoom box, if necessary (i.e., currently zooming/dragging).
   *
   * @param g		the graphics context
   */
  protected void paintZoomBox(Graphics g) {
    int	topX;
    int	bottomX;
    int	topY;
    int	bottomY;
    int	tmp;

    if (m_Zooming && m_Dragged) {
      g.setColor(m_ZoomBoxColor);

      topX    = (int) m_ZoomTopLeft.getX();
      topY    = (int) m_ZoomTopLeft.getY();
      bottomX = (int) m_ZoomBottomRight.getX();
      bottomY = (int) m_ZoomBottomRight.getY();

      // swap necessary?
      if (topX > bottomX) {
        tmp     = topX;
        topX    = bottomX;
        bottomX = tmp;
      }
      if (topY > bottomY) {
        tmp     = topY;
        topY    = bottomY;
        bottomY = tmp;
      }

      g.drawRect(
          topX,
          topY,
          (bottomX - topX + 1),
          (bottomY - topY + 1));
    }
  }

  /**
   * paints the panel and notifies all listeners.
   *
   * @param g		the graphics context
   */
  public void paintComponent(Graphics g) {
    super.paintComponent(g);

    // background
    clearBackground(g);
    getOwner().notifyPaintListeners(g, PaintMoment.BACKGROUND);

    // grid
    paintCoordinatesGrid(g);
    getOwner().notifyPaintListeners(g, PaintMoment.GRID);

    // other paint moments
    getOwner().notifyPaintListeners(g, PaintMoment.PRE_PAINT);
    getOwner().notifyPaintListeners(g, PaintMoment.PAINT);
    getOwner().notifyPaintListeners(g, PaintMoment.POST_PAINT);

    // paint zoom box
    paintZoomBox(g);
  }

  /**
   * Sets the class for customizing the tip text.
   *
   * @param value	the customizer
   */
  public void setTipTextCustomizer(TipTextCustomizer value) {
    m_TipTextCustomizer = value;
  }

  /**
   * Returns the current tip text customizer, can be null.
   *
   * @return		the customizer
   */
  public TipTextCustomizer getTipTextCustomizer() {
    return m_TipTextCustomizer;
  }

  /**
   * Returns the values as tooltip.
   *
   * @param event	the event that triggered this method
   * @return		the tool tip
   */
  public String getToolTipText(MouseEvent event) {
    String	result;
    AxisPanel	panel;
    int	count;

    result = null;

    count = 0;
    for (Axis axis: Axis.values()) {
      if (!getOwner().hasToolTipAxis(axis))
        continue;
      if (count == 0)
        result = "";
      else
        result += ", ";

      panel = getOwner().getAxis(axis);
      m_Format.applyPattern(panel.getNumberFormat());

      result += axis.getDisplayShort() + ": ";
      if ((axis == Axis.LEFT) || (axis == Axis.RIGHT))
        result += panel.valueToDisplay(panel.posToValue(event.getY()));
      else
        result += panel.valueToDisplay(panel.posToValue(event.getX()));
      count++;
    }

    // post-process tiptext?
    if (m_TipTextCustomizer != null)
      result = m_TipTextCustomizer.processTipText(getOwner(), event.getPoint(), result);

    return result;
  }

  /**
   * Removes all hit detectors.
   */
  public void clearHitDetectors() {
    m_HitDetectors.clear();
  }

  /**
   * Adds the detector to the internal list of detectors.
   *
   * @param detector		the detector to add
   */
  public void addHitDetector(AbstractHitDetector detector) {
    m_HitDetectors.add(detector);
  }

  /**
   * Removes the detector from the internal list of detectors.
   *
   * @param detector		the detector to remover
   */
  public void removeHitDetector(AbstractHitDetector detector) {
    m_HitDetectors.remove(detector);
  }

  /**
   * Runs the mouseevent through all registered hit detectors.
   *
   * @param e		the mouse event for the detectors to analyze
   */
  protected void detectHits(MouseEvent e) {
    Iterator<AbstractHitDetector>	iter;
    AbstractHitDetector		detector;

    iter = m_HitDetectors.iterator();
    while (iter.hasNext()) {
      detector = iter.next();
      if (detector.isEnabled())
        detector.detect(e);
    }
  }

  /**
   * In case an axis changes its type, e.g., log instead of percentage.
   *
   * @param e		the event
   */
  public void stateChanged(ChangeEvent e) {
    repaint();
  }
}