/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * AbstractTableAction.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.visualization.report.reportfactory;

import javax.swing.Icon;
import javax.swing.JPopupMenu;

import adams.data.report.AbstractField;
import adams.data.report.Report;
import adams.db.ReportProvider;
import adams.gui.action.AbstractBaseAction;
import adams.gui.chooser.AbstractReportFileChooser;
import adams.gui.core.GUIHelper;
import adams.gui.visualization.report.ReportFactory;

/**
 * Ancestor for actions populating the popup menu of the Table class of the 
 * {@link ReportFactory}.
 * 
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4707 $
 * @param <T> the type of table we are dealing with
 */
public abstract class AbstractTableAction
  extends AbstractBaseAction {

  /** for serialization. */
  private static final long serialVersionUID = 3352294109225825118L;

  /** the placeholder for a separator in the popup menu. */
  public final static String SEPARATOR = "-";

  /** the key for storing the table in the action's hashtable. */
  public final static String KEY_TABLE = "table";

  /** the key for storing the row in the action's hashtable. */
  public final static String KEY_ROW = "row";

  /** the key for storing the field in the action's hashtable. */
  public final static String KEY_FIELD = "field";

  /** the key for storing the value in the action's hashtable. */
  public final static String KEY_VALUE = "value";
  
  /**
   * Defines an <code>AbstractTableAction</code> object with a default
   * description string and default icon.
   */
  public AbstractTableAction() {
    super();
  }

  /**
   * Defines an <code>AbstractTableAction</code> object with the specified
   * description string and a default icon.
   *
   * @param name	the description
   */
  public AbstractTableAction(String name) {
    super(name);
  }

  /**
   * Defines an <code>AbstractTableAction</code> object with the specified
   * description string and a the specified icon.
   *
   * @param name	the description
   * @param icon	the icon
   */
  public AbstractTableAction(String name, Icon icon) {
    super(name, icon);
  }

  /**
   * Defines an <code>AbstractTableAction</code> object with the specified
   * description string and a the specified icon.
   *
   * @param name	the description
   * @param icon	the icon file (without path)
   */
  public AbstractTableAction(String name, String icon) {
    super(name, icon);
  }

  /**
   * Sets the table to use.
   * 
   * @param value	the table
   */
  protected void setTable(ReportFactory.Table value) {
    putValue(KEY_TABLE, value);
  }
  
  /**
   * Returns the currently table.
   * 
   * @return		the table, null if not applicable
   */
  protected ReportFactory.Table getTable() {
    return (ReportFactory.Table) getValue(KEY_TABLE);
  }

  /**
   * Sets the row to use.
   * 
   * @param value	the row
   */
  protected void setRow(Integer value) {
    putValue(KEY_ROW, value);
  }

  /**
   * Returns the currently selected row in the table.
   * 
   * @return		the row, null if not applicable
   */
  protected Integer getRow() {
    return (Integer) getValue(KEY_ROW);
  }

  /**
   * Sets the field to use.
   * 
   * @param value	the field
   */
  protected void setField(AbstractField value) {
    putValue(KEY_FIELD, value);
  }

  /**
   * Returns the currently selected field in the table.
   * 
   * @return		the field, null if not applicable
   */
  protected AbstractField getField() {
    return (AbstractField) getValue(KEY_FIELD);
  }

  /**
   * Sets the value to use.
   * 
   * @param value	the value
   */
  protected void setValue(String value) {
    putValue(KEY_VALUE, value);
  }

  /**
   * Returns the currently selected value in the table.
   * 
   * @return		the value, null if not applicable
   */
  protected String getValue() {
    return (String) getValue(KEY_VALUE);
  }

  /**
   * Returns the underlying report provider.
   * 
   * @return		the report provider, null if not available
   */
  protected ReportProvider getReportProvider() {
    if (getTable() != null)
      return getTable().getReportProvider();
    else
      return null;
  }

  /**
   * Sets the underlying report.
   * 
   * @param value	the report to set
   */
  protected void setReport(Report value) {
    if (getTable() != null)
      getTable().setReport(value);
  }

  /**
   * Returns the underlying report.
   * 
   * @return		the report, null if not available
   */
  protected Report getReport() {
    if (getTable() != null)
      return getTable().getReport();
    else
      return null;
  }
  
  /**
   * Returns the underlying report file chooser.
   * 
   * @return		the file chooser, null if not available
   */
  protected AbstractReportFileChooser getFileChooser() {
    if (getTable() != null)
      return getTable().getFileChooser();
    else
      return null;
  }

  /**
   * Parses a string as double. If parsing fails, an error dialog is
   * popped up.
   *
   * @param s		the string to parse
   * @return		the double or null in case of an error
   */
  protected Double parseDouble(String s) {
    Double	result;

    try {
      result = Double.parseDouble(s);
    }
    catch (Exception e) {
      result = null;
      e.printStackTrace();
      GUIHelper.showErrorMessage(
	  getTable(), "Error parsing value '" + s + "':\n" + e);
    }

    return result;
  }

  /**
   * Parses a string as boolean. If parsing fails, an error dialog is
   * popped up.
   *
   * @param s		the string to parse
   * @return		the boolean or null in case of an error
   */
  protected Boolean parseBoolean(String s) {
    Boolean 	result;

    try {
      result = Boolean.parseBoolean(s);
    }
    catch (Exception e) {
      result = null;
      e.printStackTrace();
      GUIHelper.showErrorMessage(
	  getTable(), "Error parsing value '" + s + "':\n" + e);
    }

    return result;
  }
  
  /**
   * Checks whether the action is applicable and should be added to the popup
   * menu.
   * 
   * @param table	the table the popup menu is for
   * @param row		the currently selected row
   * @param field	the field in the specified row
   * @param value	the current value
   * @return		true if the action is applicable, i.e., should be 
   * 			included in the popup menu
   */
  public abstract boolean isApplicable(ReportFactory.Table table, int row, AbstractField field, String value);
  
  /**
   * Assembles the popup menu for the table.
   * 
   * @param actions	the classnames of the actions, "-" indicates a separator
   * @param table	the table the popup menu is for
   * @param row		the row in the table this popup menu is for
   * @return		the popup menu
   * @see		#SEPARATOR
   */
  public static JPopupMenu createPopup(String[] actions, ReportFactory.Table table, int row) {
    JPopupMenu		result;
    AbstractTableAction	taction;
    AbstractField	field;
    Object		value;
    String		valueStr;
    boolean		wasSeparator;
    
    result = new JPopupMenu();

    field        = table.getFieldAt(row);
    value        = table.getValueAt(row, 2);
    if (value == null)
      valueStr = "";
    else
      valueStr = "" + value;
    wasSeparator = false;
    for (String action: actions) {
      if (action.equals(SEPARATOR)) {
	if (!wasSeparator) {
	  result.addSeparator();
	  wasSeparator = true;
	}
      }
      else {
	try {
	  taction = (AbstractTableAction) Class.forName(action).newInstance();
	  if (taction.isApplicable(table, row, field, valueStr)) {
	    taction.setTable(table);
	    taction.setRow(row);
	    taction.setField(field);
	    taction.setValue(valueStr);
	    wasSeparator = false;
	    result.add(taction);
	  }
	}
	catch (Exception e) {
	  System.err.println("Failed to instantiate table action '" + action + "':");
	  e.printStackTrace();
	}
      }
    }
    
    return result;
  }
}
