/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * NamedCounter.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */
package adams.core;

import java.io.Serializable;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Set;

/**
 * Provides counters that can be referenced by a name.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4856 $
 */
public class NamedCounter
  implements Serializable {

  /** for serialization. */
  private static final long serialVersionUID = -690032882282626773L;

  /** for storing the counts. */
  protected Hashtable<String,Integer> m_Counts;

  /**
   * Initializes the counter.
   */
  public NamedCounter() {
    super();
    m_Counts = new Hashtable<String,Integer>();
  }

  /**
   * Clears the counter.
   */
  public synchronized void clear() {
    m_Counts.clear();
  }

  /**
   * Clears only the specific counter.
   * 
   * @param name	the name of the counter
   */
  public synchronized void clear(String name) {
    m_Counts.put(name, 0);
  }

  /**
   * Returns the current counter value.
   *
   * @param name	the name of the counter
   * @return		the current counter value
   */
  public synchronized int current(String name) {
    if (!m_Counts.containsKey(name))
      m_Counts.put(name, 0);
    return m_Counts.get(name);
  }

  /**
   * Returns the incremented counter.
   *
   * @param		the name of the counter
   * @return		the incremented counter
   */
  public synchronized int next(String name) {
    m_Counts.put(name, current(name) + 1);
    return m_Counts.get(name);
  }

  /**
   * Returns whether the counter has reached the limit.
   *
   * @param name	the name of the counter
   * @param limit	the limit to check
   * @return		true if limit reached (or above)
   */
  public synchronized boolean hasReached(String name, int limit) {
    return (current(name) >= limit);
  }
  
  /**
   * Returns the currently stored names.
   * 
   * @return		the name enumeration
   */
  public synchronized Enumeration<String> names() {
    return m_Counts.keys();
  }
  
  /**
   * Returns the currently stored names.
   * 
   * @return		the name set
   */
  public synchronized Set<String> nameSet() {
    return m_Counts.keySet();
  }
  
  /**
   * Returns the current counters as string.
   *
   * @return		the string representation
   */
  public synchronized String toString() {
    return m_Counts.toString();
  }
}
