/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Expression.java
 * Copyright (C) 2011-2012 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.condition.bool;

import java.util.HashMap;

import adams.core.Variables;
import adams.data.report.Report;
import adams.data.report.ReportHandler;
import adams.flow.core.AbstractActor;
import adams.flow.core.Token;
import adams.flow.core.Unknown;
import adams.parser.BooleanExpression;
import adams.parser.BooleanExpressionText;
import adams.parser.GrammarSupplier;

/**
 <!-- globalinfo-start -->
 * Emulates an If-Then-Else construct. The 'Then' branch gets executed if the 'If' expression is 'true'.<br/>
 * In case of integer or double tokens that arrive at the input, these can be accessed in the expression via 'X'; string tokens can be accessed via expression '"X"' (surrounding double quotes are required).<br/>
 * <br/>
 * The following grammar is used for evaluating the boolean expressions:<br/>
 * <br/>
 *  expr_list ::= expr_list expr_part | expr_part ;<br/>
 *  expr_part ::= boolexpr ;<br/>
 * <br/>
 *  boolexpr ::=    ( boolexpr )<br/>
 *                | true<br/>
 *                | false<br/>
 *                | bool ( VARIABLE (must represent a boolean string: true|false) )<br/>
 *                | numexpr &lt; numexpr<br/>
 *                | numexpr &lt;= numexpr<br/>
 *                | numexpr &gt; numexpr<br/>
 *                | numexpr &gt;= numexpr<br/>
 *                | numexpr = numexpr<br/>
 *                | numexpr != numexpr<br/>
 *                | strexpr &lt; strexpr<br/>
 *                | strexpr &lt;= strexpr<br/>
 *                | strexpr &gt; strexpr<br/>
 *                | strexpr &gt;= strexpr<br/>
 *                | strexpr = strexpr<br/>
 *                | strexpr != strexpr<br/>
 *                | matches ( strexpr , regexp )<br/>
 *                | ! boolexpr<br/>
 *                | boolexpr &amp; boolexpr<br/>
 *                | boolexpr | boolexpr<br/>
 *                ;<br/>
 * <br/>
 *  numexpr   ::=   ( numexpr )<br/>
 *                | NUMBER<br/>
 *                | -numexpr<br/>
 *                | constexpr<br/>
 *                | opexpr<br/>
 *                | funcexpr<br/>
 *                ;<br/>
 * <br/>
 *  strexpr   ::=   substr ( strexpr , start )<br/>
 *                | substr ( strexpr , start , end )<br/>
 *                | lowercase ( strexpr )<br/>
 *                | uppercase ( strexpr )<br/>
 *                | string<br/>
 *                | str ( VARIABLE (must represent a string) )<br/>
 *                ;<br/>
 * <br/>
 *  constexpr ::=   PI<br/>
 *                | E<br/>
 *                ;<br/>
 * <br/>
 *  opexpr    ::=   numexpr + numexpr<br/>
 *                | numexpr - numexpr<br/>
 *                | numexpr * numexpr<br/>
 *                | numexpr &#47; numexpr<br/>
 *                | numexpr ^ numexpr<br/>
 *                ;<br/>
 * <br/>
 *  funcexpr ::=    abs ( numexpr )<br/>
 *                | sqrt ( numexpr )<br/>
 *                | log ( numexpr )<br/>
 *                | exp ( numexpr )<br/>
 *                | sin ( numexpr )<br/>
 *                | cos ( numexpr )<br/>
 *                | tan ( numexpr )<br/>
 *                | rint ( numexpr )<br/>
 *                | floor ( numexpr )<br/>
 *                | pow ( numexpr , numexpr )<br/>
 *                | ceil ( numexpr )<br/>
 *                | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *                | ifmissing ( variable , numexpr (default value if variable is missing) )<br/>
 *                | num ( VARIABLE (must represent a number) )<br/>
 *                | length ( str )<br/>
 *                ;<br/>
 * <br/>
 * Notes:<br/>
 * - Variables are either all upper case letters (e.g., "ABC") or any character   apart from "]" enclosed by "[" and "]" (e.g., "[Hello World]").<br/>
 * - 'start' and 'end' for function 'substr' are indices that start at 1.<br/>
 * - Index 'end' for function 'substr' is excluded (like Java's 'String.substring(int,int)' method)<br/>
 * - Line comments start with '#'<br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-expression &lt;adams.parser.BooleanExpressionText&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The expression to evaluate - only as long as it evaluates to 'true' the 
 * &nbsp;&nbsp;&nbsp;loop actors get executed; expressions that consists solely of a variable 
 * &nbsp;&nbsp;&nbsp;(eg '&#64;{blah}') get automatically wrapped in parentheses, since the expression 
 * &nbsp;&nbsp;&nbsp;string gets interpreted as attached variable instead.
 * &nbsp;&nbsp;&nbsp;default: true
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4813 $
 */
public class Expression
  extends AbstractBooleanCondition
  implements GrammarSupplier {

  /** for serialization. */
  private static final long serialVersionUID = -9169161144858552052L;

  /** the "If" expression to evaluate. */
  protected BooleanExpressionText m_Expression;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Emulates an If-Then-Else construct. The 'Then' branch gets executed "
      + "if the 'If' expression is 'true'.\n"
      + "In case of integer or double tokens that arrive at the input, these "
      + "can be accessed in the expression via 'X'; string tokens can be accessed "
      + "via expression '\"X\"' (surrounding double quotes are required).\n\n"
      + "The following grammar is used for evaluating the boolean expressions:\n\n"
      + getGrammar();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return new BooleanExpression().getGrammar();
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "expression", "expression",
	    new BooleanExpressionText("true"));
  }

  /**
   * Sets the expression to evaluate. Automatically wraps expressions in
   * parentheses that consists only of a variable. Otherwise, the expresssion
   * would get interpreted as attached variable for the expression option.
   *
   * @param value	the expression
   */
  public void setExpression(BooleanExpressionText value) {
    if (Variables.isPlaceholder(value.getValue()))
      value = new BooleanExpressionText("(" + value.getValue() + ")");
    m_Expression = value;
    reset();
  }

  /**
   * Returns the expression to evaluate.
   *
   * @return		the expression
   */
  public BooleanExpressionText getExpression() {
    return m_Expression;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionTipText() {
    return
        "The expression to evaluate - only as long as it evaluates to 'true' "
      + "the loop actors get executed; expressions that consists solely of a "
      + "variable (eg '@{blah}') get automatically wrapped in parentheses, "
      + "since the expression string gets interpreted as attached variable "
      + "instead.";
  }

  /**
   * Returns the quick info string to be displayed in the flow editor.
   *
   * @return		the info or null if no info to be displayed
   */
  public String getQuickInfo() {
    String	variable;

    variable = getOptionManager().getVariableForProperty("expression");

    if (variable != null)
      return variable;
    else if ((m_Expression != null) && (m_Expression.getValue().length() > 0))
      return m_Expression.stringValue();
    else
      return null;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		adams.flow.core.Unknown.class
   */
  public Class[] accepts() {
    return new Class[]{Unknown.class};
  }

  /**
   * Configures the condition.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = null;

    if ((m_Expression == null) || (m_Expression.getValue().length() == 0))
      result = "No expression provided!";

    return result;
  }

  /**
   * Evaluates whether to executed the "then" or "else" branch.
   *
   * @param owner	the owning IfThenElse actor
   * @param token	the current token passing through the IfThenElse actor
   * @return		true if to execute the "then" branch
   */
  protected boolean doEvaluate(AbstractActor owner, Token token) {
    boolean	result;
    String	exp;
    HashMap	symbols;
    Report	report;
    boolean	hasString;

    exp       = owner.getVariables().expand(getExpression().getValue());
    hasString = (exp.indexOf("\"X\"") > -1);
    symbols   = new HashMap();
    report    = null;
    if (token != null) {
      if (token.getPayload() instanceof Integer)
	symbols.put("X", ((Integer) token.getPayload()).doubleValue());
      else if (token.getPayload() instanceof Double)
	symbols.put("X", ((Double) token.getPayload()).doubleValue());
      else if ((token.getPayload() instanceof String) && hasString)
	exp = exp.replace("\"X\"", "\"" + token.getPayload() + "\"");
      else if (token.getPayload() instanceof Report)
	report = (Report) token.getPayload();
      else if (token.getPayload() instanceof ReportHandler)
	report = ((ReportHandler) token.getPayload()).getReport();
    }

    try {
      if (report != null)
	result = BooleanExpression.evaluate(exp, report);
      else
	result = BooleanExpression.evaluate(exp, symbols);
      if (getDebugLevel() >= 2)
	debug(
	    "exp: " + getExpression() + "\n"
	    + "  --> " + exp + "\n"
	    + "  = " + result, 2);
    }
    catch (Exception e) {
      getSystemErr().println("Error evaluating boolean expression: " + exp);
      getSystemErr().printStackTrace(e);
      result = false;
    }

    return result;
  }
}
