/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * AbstractContainerUpdater.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.flow.control;

import java.util.Hashtable;

import adams.flow.container.AbstractContainer;
import adams.flow.container.SequencePlotterContainer;
import adams.flow.core.AbstractActor;
import adams.flow.core.InputConsumer;
import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 * Ancestor for control actors that update a specific value of a container
 * using the defined sub-actors.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractContainerUpdater
  extends SubProcess {

  /** for serialization. */
  private static final long serialVersionUID = 7140175689043000123L;

  /** the key for storing the output token in the backup. */
  public final static String BACKUP_OUTPUT = "output";

  /** the value to modify. */
  protected String m_ContainerValueName;

  /** the output. */
  protected Token m_OutputToken;

  /**
   * Resets the scheme.
   */
  protected void reset() {
    super.reset();

    m_OutputToken = null;
  }

  /**
   * Sets the container value to update.
   *
   * @param value	the name of the value
   */
  protected void setContainerValueName(String value) {
    m_ContainerValueName = value;
    reset();
  }

  /**
   * Returns the container value to update.
   *
   * @return		the name of the value
   */
  protected String getContainerValueName() {
    return m_ContainerValueName;
  }

  /**
   * Backs up the current state of the actor before update the variables.
   *
   * @return		the backup
   */
  protected Hashtable<String,Object> backupState() {
    Hashtable<String,Object>	result;

    result = super.backupState();

    if (m_OutputToken != null)
      result.put(BACKUP_OUTPUT, m_OutputToken);

    return result;
  }

  /**
   * Restores the state of the actor before the variables got updated.
   *
   * @param state	the backup of the state to restore from
   */
  protected void restoreState(Hashtable<String,Object> state) {
    if (state.containsKey(BACKUP_OUTPUT)) {
      m_OutputToken = (Token) state.get(BACKUP_OUTPUT);
      state.remove(BACKUP_OUTPUT);
    }

    super.restoreState(state);
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		the Class of objects that can be processed
   */
  public Class[] accepts() {
    return new Class[]{AbstractContainer.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		the Class of the generated tokens
   */
  public Class[] generates() {
    return new Class[]{Unknown.class};
  }

  /**
   * The method that accepts the input token and then processes it.
   *
   * @param token	the token to accept and process
   */
  public void input(Token token) {
    m_CurrentToken = token;
    m_OutputToken  = null;
  }

  /**
   * Executes the actor.
   *
   * @return		null if everything is fine, otherwise error message
   */
  protected String doExecute() {
    String		result;
    AbstractActor	first;
    Object		input;
    Object		output;
    AbstractContainer	cont;
    AbstractContainer	contNew;
    boolean		processed;

    result = null;

    first = firstActive();
    if (isDebugOn())
      debug("first active actor: " + first.getFullName());

    cont      = null;
    processed = false;
    if ((first != null) && (first instanceof InputConsumer)) {
      cont = (SequencePlotterContainer) m_CurrentToken.getPayload();
      if (cont.hasValue(m_ContainerValueName))
	input = cont.getValue(m_ContainerValueName);
      else
	throw new IllegalStateException("Container value not present: " + m_ContainerValueName);

      ((InputConsumer) first).input(new Token(input));
      if (getDebugLevel() > 1)
	debug("input: " + input, 2);

      try {
	result    = m_Director.execute();
	processed = true;
      }
      catch (Exception e) {
	getSystemErr().printStackTrace(e);
	result = e.toString();
      }
    }

    if (processed) {
      if (m_OutputTokens.size() == 1) {
	try {
	  output  = m_OutputTokens.get(0).getPayload();
	  contNew = cont.getClone();
	  contNew.setValue(m_ContainerValueName, output);
	  m_OutputToken = new Token(contNew);
	}
	catch (Exception e) {
	  result        = "Failed to generate new container: " + e;
	  m_OutputToken = null;
	  getSystemErr().println(result);
	  getSystemErr().printStackTrace(e);
	}
      }
      else {
	result = "Last active sub-actor did not produce exactly one output: " + m_OutputTokens.size();
      }
    }

    m_OutputTokens.clear();

    return result;
  }

  /**
   * Post-execute hook.
   *
   * @return		null if everything is fine, otherwise error message
   * @see		#m_Executed
   */
  protected String postExecute() {
    String	result;

    result = super.postExecute();

    if (isStopped())
      m_OutputToken = null;

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (m_OutputToken != null);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result        = m_OutputToken;
    m_OutputToken = null;

    return result;
  }

  /**
   * Cleans up after the execution has finished.
   */
  public void wrapUp() {
    m_OutputToken = null;

    super.wrapUp();
  }
}
