/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * SubProcess.java
 * Copyright (C) 2009 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.control;


import java.util.Vector;

import adams.flow.core.AbstractActor;
import adams.flow.core.InputConsumer;
import adams.flow.core.OutputProducer;
import adams.flow.core.PauseStateHandler;
import adams.flow.core.PauseStateManager;
import adams.flow.core.Token;
import adams.flow.core.Unknown;

/**
 <!-- globalinfo-start -->
 * Encapsulates a sequence of flow items. The first actor must accept input and the last one must produce output.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D (property: debug)
 *         If set to true, scheme may output additional info to the console.
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 *         The name of the actor.
 *         default: SubProcess
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseString&gt; [-annotation ...] (property: annotations)
 *         The annotations to attach to this actor.
 * </pre>
 *
 * <pre>-skip (property: skip)
 *         If set to true, transformation is skipped and the input token is just forwarded
 *          as it is.
 * </pre>
 *
 * <pre>-progress (property: showProgress)
 *         If set to true, progress information will be output to stdout ('.').
 * </pre>
 *
 * <pre>-actor &lt;adams.flow.core.AbstractActor [options]&gt; [-actor ...] (property: actors)
 *         All the actors that define this sequence.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4665 $
 */
public class SubProcess
  extends Sequence
  implements OutputProducer, PauseStateHandler {

  /** for serialization. */
  private static final long serialVersionUID = 7433940498896052594L;

  /**
   * A specialized director for the SubProcess actor.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4665 $
   */
  public static class SubProcessDirector
    extends SequentialDirector {

    /** for serialization. */
    private static final long serialVersionUID = 1600945233224761728L;

    /**
     * Sets the group to execute.
     *
     * @param value 	the group
     */
    public void setControlActor(AbstractDirectedControlActor value) {
      if ((value instanceof SubProcess) || (value == null))
	super.setControlActor(value);
      else
	System.err.println(
	    "Group must be a SubProcess actor (provided: "
	    + ((value != null) ? value.getClass().getName() : "-null-") + ")!");
    }

    /**
     * Returns whether the final output of actors is recorded.
     *
     * @return		true
     */
    protected boolean isFinalOutputRecorded() {
      return true;
    }

    /**
     * Peforms the execution of the actors.
     *
     * @param startActor	the actor to start with
     * @return		null if everything ok, otherwise the error message
     */
    protected String doExecuteActors(AbstractActor startActor) {
      String		result;
      int		i;

      result = super.doExecuteActors(startActor);

      if (result == null) {
	for (i = 0; i < m_FinalOutput.size(); i++)
	  ((SubProcess) getControlActor()).addOutputToken(m_FinalOutput.get(i));
      }

      return result;
    }
  }

  /** for storing generated output tokens. */
  protected transient Vector<Token> m_OutputTokens;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Encapsulates a sequence of flow items. The first actor must accept "
      + "input and the last one must produce output.";
  }

  /**
   * Initializes m_OutputTokens if necessary and returns it.
   *
   * @return		m_OutputTokens
   */
  protected Vector<Token> getOutputTokens() {
    if (m_OutputTokens == null)
      m_OutputTokens = new Vector<Token>();
    return m_OutputTokens;
  }

  /**
   * Returns an instance of a director.
   *
   * @return		the director
   */
  protected SequentialDirector newDirector() {
    return new SubProcessDirector();
  }

  /**
   * Returns the pause state manager.
   * 
   * @return		the manager
   */
  public PauseStateManager getPauseStateManager() {
    return m_PauseStateManager;
  }

  /**
   * Initializes the item for flow execution.
   *
   * @return		null if everything is fine, otherwise error message
   */
  public String setUp() {
    String	result;

    result = super.setUp();

    // sub-process is used sometimes as standalone actor with no outer Flow
    // hence we need to instantiate our own pause state manager
    if (m_PauseStateManager == null)
      m_PauseStateManager = new PauseStateManager();
    
    if (result == null) {
      if (active() == 0)
	result = "No active (= non-skipped) actors!";
    }

    if (result == null) {
      if (!(firstActive() instanceof InputConsumer))
	result = "First actor ('" + firstActive().getName() + "') does not accept input!";
      else if (!(lastActive() instanceof OutputProducer))
	result = "Last actor ('" + lastActive().getName() + "') does not generate output!";
    }

    return result;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		the Class of objects that can be processed
   */
  public Class[] accepts() {
    if (active() > 0)
      return ((InputConsumer) firstActive()).accepts();
    else
      return new Class[]{Unknown.class};
  }

  /**
   * Returns the class of objects that it generates.
   *
   * @return		the Class of the generated tokens
   */
  public Class[] generates() {
    if (active() > 0)
      return ((OutputProducer) lastActive()).generates();
    else
      return new Class[]{Unknown.class};
  }

  /**
   * Adds the given token to the list of available output tokens.
   *
   * @param output	the token to add
   * @see		#getOutputTokens()
   */
  protected void addOutputToken(Token output) {
    getOutputTokens().add(output);
  }

  /**
   * Returns the generated token.
   *
   * @return		the generated token
   */
  public Token output() {
    Token	result;

    result = getOutputTokens().get(0);
    getOutputTokens().remove(0);

    return result;
  }

  /**
   * Checks whether there is pending output to be collected after
   * executing the flow item.
   *
   * @return		true if there is pending output
   */
  public boolean hasPendingOutput() {
    return (getOutputTokens().size() > 0);
  }

  /**
   * Cleans up data structures, frees up memory.
   */
  public void cleanUp() {
    super.cleanUp();

    if (m_OutputTokens != null)
      m_OutputTokens.clear();
  }
}
