/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * DisplayPanelManager.java
 * Copyright (C) 2009-2010 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.text.SimpleDateFormat;
import java.util.Date;

import javax.swing.JComponent;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JSplitPane;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import adams.core.io.FileUtils;
import adams.core.io.PlaceholderFile;
import adams.flow.core.InputConsumer;
import adams.flow.core.Token;
import adams.gui.chooser.BaseFileChooser;
import adams.gui.core.AbstractNamedHistoryPanel;
import adams.gui.core.BasePanel;
import adams.gui.core.BaseScrollPane;
import adams.gui.core.ExtensionFileFilter;
import adams.gui.core.GUIHelper;
import adams.gui.core.MenuBarProvider;
import adams.gui.print.JComponentWriter;
import adams.gui.print.JComponentWriterFileChooser;
import adams.gui.print.PNGWriter;
import adams.gui.sendto.SendToActionSupporter;
import adams.gui.sendto.SendToActionUtils;

/**
 <!-- globalinfo-start -->
 * Actor that displays a 'history' of panels created by the selected panel provider. The provider can be an actor that generates classifier errors, for instance.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input/output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;weka.classifiers.Evaluation<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: DisplayPanelManager
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-width &lt;int&gt; (property: width)
 * &nbsp;&nbsp;&nbsp;The width of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 640
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-height &lt;int&gt; (property: height)
 * &nbsp;&nbsp;&nbsp;The height of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 480
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-x &lt;int&gt; (property: x)
 * &nbsp;&nbsp;&nbsp;The X position of the dialog (&gt;=0: absolute, -1: left, -2: center, -3: right
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-y &lt;int&gt; (property: y)
 * &nbsp;&nbsp;&nbsp;The Y position of the dialog (&gt;=0: absolute, -1: top, -2: center, -3: bottom
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-writer &lt;adams.gui.print.JComponentWriter [options]&gt; (property: writer)
 * &nbsp;&nbsp;&nbsp;The writer to use for generating the graphics output.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.print.NullWriter
 * </pre>
 *
 * <pre>-provider &lt;adams.flow.sink.DisplayPanelProvider [options]&gt; (property: panelProvider)
 * &nbsp;&nbsp;&nbsp;The actor for generating the display panels.
 * &nbsp;&nbsp;&nbsp;default: adams.flow.sink.ClassifierErrors -writer adams.gui.print.NullWriter
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class DisplayPanelManager
  extends AbstractDisplay
  implements MenuBarProvider, ComponentSupplier, TextSupplier, SendToActionSupporter {

  /** for serialization. */
  private static final long serialVersionUID = 3365817040968234289L;

  /**
   * A history panel that keeps track of named AbstractDisplayPanel objects, e.g.,
   * containing experiments results.
   *
   * @author  fracpete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4584 $
   */
  public static class DisplayPanelHistoryPanel
    extends AbstractNamedHistoryPanel<AbstractDisplayPanel> {

    /** for serialization. */
    private static final long serialVersionUID = 1704390033157269580L;

    /** the text area to display the result in. */
    protected BasePanel m_Panel;

    /**
     * Initializes the members.
     */
    protected void initialize() {
      super.initialize();

      m_Panel = null;
    }

    /**
     * Sets the panel to display the results in.
     *
     * @param value	the text area to use
     */
    public void setPanel(BasePanel value) {
      m_Panel = value;
    }

    /**
     * Displays the specified entry.
     *
     * @param name	the name of the entry, can be null to clear display
     */
    protected void updateEntry(String name) {
      m_Panel.removeAll();

      if (name != null) {
        // update panel
        if (hasEntry(name)) {
          m_Panel.add(getEntry(name));
          m_Panel.getParent().invalidate();
          m_Panel.getParent().validate();
          m_Panel.getParent().repaint();
        }
      }
    }

    /**
     * Removes the specified entry.
     *
     * @param name	the name of the entry
     * @return		the entry that was stored under this name or null if
     * 			no entry was stored with this name
     */
    public AbstractDisplayPanel removeEntry(String name) {
      AbstractDisplayPanel	result;

      result = super.removeEntry(name);
      if (result != null)
	result.cleanUp();

      return result;
    }
  }

  /**
   * Represents a panel with a history on the left and the displayed panel
   * on the right.
   *
   * @author FracPete (fracpete at waikato dot ac dot nz)
   * @version $Revision: 4584 $
   */
  public static class HistorySplitPanel
    extends BasePanel {

    /** for serialization. */
    private static final long serialVersionUID = 5121061351955687610L;

    /** the owning DisplayPanelDisplay component. */
    protected DisplayPanelManager m_Owner;

    /** the split pane for the components. */
    protected JSplitPane m_SplitPane;

    /** the history panel. */
    protected DisplayPanelHistoryPanel m_History;

    /** the actual panel for displaying the other panels. */
    protected BasePanel m_Panel;

    /** the format for the dates. */
    protected SimpleDateFormat m_Format;

    /**
     * Initializes the split pane.
     *
     * @param owner		the owning TextDisplay
     */
    public HistorySplitPanel(DisplayPanelManager owner) {
      super(new BorderLayout());

      m_Owner   = owner;
      m_Format  = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS");

      m_SplitPane = new JSplitPane();
      add(m_SplitPane, BorderLayout.CENTER);

      m_Panel = new BasePanel(new BorderLayout());
      if (owner.getPanelProvider().displayPanelRequiresScrollPane())
	m_SplitPane.setBottomComponent(new BaseScrollPane(m_Panel));
      else
	m_SplitPane.setBottomComponent(m_Panel);

      m_History = new DisplayPanelHistoryPanel();
      m_History.setPanel(m_Panel);
      m_SplitPane.setTopComponent(m_History);

      m_SplitPane.setResizeWeight(0.1);
      m_SplitPane.setDividerLocation(150);
    }

    /**
     * Returns the owner of this history panel.
     *
     * @return		the owning TextDisplay component
     */
    public DisplayPanelManager getOwner() {
      return m_Owner;
    }

    /**
     * Removes all entries.
     */
    public void clear() {
      m_History.clear();
      m_Panel.removeAll();
    }

    /**
     * Returns the number of results.
     *
     * @return		the number of results
     */
    public int count() {
      return m_History.count();
    }

    /**
     * Returns the underlying history panel.
     *
     * @return		the panel
     */
    public DisplayPanelHistoryPanel getHistory() {
      return m_History;
    }

    /**
     * Adds the given text.
     *
     * @param result	the text to add
     */
    public synchronized void addResult(AbstractDisplayPanel result) {
      String	baseID;
      String	id;
      int	count;

      // determine a unique ID
      synchronized(m_Format) {
	baseID = m_Format.format(new Date());
      }
      id     = baseID;
      count  = 1;
      while (m_History.hasEntry(id)) {
	count++;
	id = baseID + " (" + count + ")";
      }

      // add result
      m_History.addEntry(id, result);

      // select this entry immediately
      m_History.setSelectedIndex(count() - 1);
    }
  }

  /** the history panel. */
  protected HistorySplitPanel m_HistoryPanel;

  /** the actor to use for generating panels. */
  protected DisplayPanelProvider m_PanelProvider;

  /** the menu bar, if used. */
  protected JMenuBar m_MenuBar;

  /** the "clear" menu item. */
  protected JMenuItem m_MenuItemFileClear;

  /** the "save as" menu item. */
  protected JMenuItem m_MenuItemFileSaveAs;

  /** the "exit" menu item. */
  protected JMenuItem m_MenuItemFileClose;

  /** the filedialog for loading/saving flows. */
  protected JComponentWriterFileChooser m_ComponentFileChooser;

  /** the filedialog for loading/saving flows. */
  protected BaseFileChooser m_TextFileChooser;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Actor that displays a 'history' of panels created by the selected "
      + "panel provider. The provider can be an actor that generates classifier "
      + "errors, for instance.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "provider", "panelProvider",
	    new ImageViewer());
  }

  /**
   * Returns a quick info about the actor, which will be displayed in the GUI.
   *
   * @return		null if no info available, otherwise short string
   */
  public String getQuickInfo() {
    String	result;
    String	variable;

    result = super.getQuickInfo() + ", provider: ";

    variable = getOptionManager().getVariableForProperty("panelProvider");
    if (variable != null)
      result += variable;
    else
      result += m_PanelProvider.getClass().getName().replace("adams.flow.sink.", "");

    return result;
  }

  /**
   * Returns the default width for the dialog.
   *
   * @return		the default width
   */
  protected int getDefaultWidth() {
    return 640;
  }

  /**
   * Returns the default height for the dialog.
   *
   * @return		the default height
   */
  protected int getDefaultHeight() {
    return 480;
  }

  /**
   * Sets the panel provider to use for generating the panels.
   *
   * @param value	the panel provider to use
   */
  public void setPanelProvider(DisplayPanelProvider value) {
    m_PanelProvider = value;
    reset();
  }

  /**
   * Returns the panel provider in use for generating the panels.
   *
   * @return		the panel provider in use
   */
  public DisplayPanelProvider getPanelProvider() {
    return m_PanelProvider;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String panelProviderTipText() {
    return "The actor for generating the display panels.";
  }

  /**
   * Returns the currently selected panel.
   *
   * @return		the panel or null if none selected
   */
  protected BasePanel getSelectedPanel() {
    BasePanel	result;
    int		index;

    result = null;
    index  = m_HistoryPanel.getHistory().getSelectedIndex();
    if (index != -1)
      result = m_HistoryPanel.getHistory().getEntry(index);

    return result;
  }

  /**
   * Returns the current panel.
   *
   * @return		the current panel, can be null
   */
  public JComponent supplyComponent() {
    JComponent	result;

    result = null;

    if (m_PanelProvider instanceof ComponentSupplier)
      result = ((ComponentSupplier) getSelectedPanel()).supplyComponent();

    return result;
  }

  /**
   * Supplies the text.
   *
   * @return		the text, null if none available
   */
  public String supplyText() {
    String	result;

    result = null;

    if (m_PanelProvider instanceof TextSupplier)
      result = ((TextSupplier) getSelectedPanel()).supplyText();

    return result;
  }

  /**
   * Clears the content of the panel.
   */
  public void clearPanel() {
    m_HistoryPanel.clear();
  }

  /**
   * Creates the panel to display in the dialog.
   *
   * @return		the panel
   */
  protected BasePanel newPanel() {
    HistorySplitPanel	result;

    result         = new HistorySplitPanel(this);
    m_HistoryPanel = result;

    return result;
  }

  /**
   * Hook method before the frame gets created.
   *
   * @param panel	the panel to display in the frame
   */
  protected void preCreateFrame(BasePanel panel) {
    super.preCreateFrame(panel);

    m_ComponentFileChooser = new JComponentWriterFileChooser();

    m_TextFileChooser = new BaseFileChooser();
    m_TextFileChooser.addChoosableFileFilter(new ExtensionFileFilter("Text files", "txt"));
    m_TextFileChooser.setDefaultExtension("txt");
    m_TextFileChooser.setAutoAppendExtension(true);
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->weka.classifiers.Evaluation.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    if ((m_PanelProvider != null) && (m_PanelProvider instanceof InputConsumer))
      return ((InputConsumer) m_PanelProvider).accepts();
    else
      return new Class[]{Object.class};
  }

  /**
   * Displays the token (the panel and dialog have already been created at
   * this stage).
   *
   * @param token	the token to display
   */
  protected void display(Token token) {
    synchronized(m_HistoryPanel) {
      m_HistoryPanel.addResult(m_PanelProvider.createDisplayPanel(token));
    }
  }

  /**
   * Assembles the menu bar.
   *
   * @return		the menu bar
   */
  protected JMenuBar createMenuBar() {
    JMenuBar	result;
    JMenu	menu;
    JMenuItem	menuitem;

    result = new JMenuBar();

    // File
    menu = new JMenu("File");
    result.add(menu);
    menu.setMnemonic('F');
    menu.addChangeListener(new ChangeListener() {
      public void stateChanged(ChangeEvent e) {
	updateMenu();
      }
    });

    // File/Clear
    menuitem = new JMenuItem("Clear");
    menu.add(menuitem);
    menuitem.setMnemonic('l');
    menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed N"));
    menuitem.setIcon(GUIHelper.getIcon("new.gif"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	clear();
      }
    });
    m_MenuItemFileClear = menuitem;

    // File/Save As
    menuitem = new JMenuItem("Save as...");
    menu.add(menuitem);
    menuitem.setMnemonic('a');
    menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed S"));
    menuitem.setIcon(GUIHelper.getIcon("save.gif"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	saveAs();
      }
    });
    m_MenuItemFileSaveAs = menuitem;

    // File/Send to
    menu.addSeparator();
    if (SendToActionUtils.addSendToSubmenu(this, menu))
      menu.addSeparator();

    // File/Close
    menuitem = new JMenuItem("Close");
    menu.add(menuitem);
    menuitem.setMnemonic('C');
    menuitem.setAccelerator(GUIHelper.getKeyStroke("ctrl pressed Q"));
    menuitem.setIcon(GUIHelper.getIcon("exit.png"));
    menuitem.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	close();
      }
    });
    m_MenuItemFileClose = menuitem;

    return result;
  }

  /**
   * Creates a menu bar (singleton per panel object). Can be used in frames.
   *
   * @return		the menu bar
   */
  public JMenuBar getMenuBar() {
    if (m_MenuBar == null) {
      m_MenuBar = createMenuBar();
      updateMenu();
    }

    return m_MenuBar;
  }

  /**
   * updates the enabled state of the menu items.
   */
  protected void updateMenu() {
    if (m_MenuBar == null)
      return;

    m_MenuItemFileSaveAs.setEnabled(
      (    (m_PanelProvider instanceof ComponentSupplier)
        || (m_PanelProvider instanceof TextSupplier) )
      && (getSelectedPanel() != null));

    m_MenuItemFileClear.setEnabled(m_HistoryPanel.count() > 0);
  }

  /**
   * Clears the display.
   */
  protected void clear() {
    m_HistoryPanel.clear();
  }

  /**
   * Saves the setups.
   */
  protected void saveAs() {
    int			retVal;
    JComponentWriter	writer;

    if (m_PanelProvider instanceof TextSupplier) {
      retVal = m_TextFileChooser.showSaveDialog(m_HistoryPanel);
      if (retVal != BaseFileChooser.APPROVE_OPTION)
	return;

      FileUtils.writeToFile(
	  m_TextFileChooser.getSelectedFile().getAbsolutePath(),
	  supplyText(),
	  false);
    }
    else if (m_PanelProvider instanceof ComponentSupplier) {
      retVal = m_ComponentFileChooser.showSaveDialog(m_HistoryPanel);
      if (retVal != BaseFileChooser.APPROVE_OPTION)
	return;

      writer = m_ComponentFileChooser.getWriter();
      writer.setComponent(supplyComponent());
      try {
	writer.toOutput();
      }
      catch (Exception e) {
	getSystemErr().println("Error saving panel to '" + writer.getFile() + "': ");
	getSystemErr().printStackTrace(e);
      }
    }
  }

  /**
   * Closes the dialog or frame.
   */
  protected void close() {
    m_HistoryPanel.closeParent();
  }

  /**
   * Removes all graphical components.
   */
  protected void cleanUpGUI() {
    if (m_HistoryPanel != null)
      m_HistoryPanel.clear();

    m_MenuBar              = null;
    m_MenuItemFileClear    = null;
    m_MenuItemFileSaveAs   = null;
    m_MenuItemFileClose    = null;

    super.cleanUpGUI();
  }

  /**
   * Returns the classes that the supporter generates.
   *
   * @return		the classes
   */
  public Class[] getSendToClasses() {
    if (m_PanelProvider instanceof ComponentSupplier)
      return new Class[]{PlaceholderFile.class, JComponent.class};
    else if (m_PanelProvider instanceof TextSupplier)
      return new Class[]{String.class};
    else
      return new Class[]{String.class};
  }

  /**
   * Checks whether something to send is available.
   *
   * @param cls		the classes to retrieve the item for
   * @return		true if an object is available for sending
   */
  public boolean hasSendToItem(Class[] cls) {
    if (SendToActionUtils.isAvailable(JComponent.class, cls)) {
      if (m_PanelProvider instanceof ComponentSupplier)
	return (supplyComponent() != null);
      else
	return false;
    }

    if (SendToActionUtils.isAvailable(new Class[]{PlaceholderFile.class, String.class}, cls)) {
      if (m_PanelProvider instanceof ComponentSupplier)
	return (supplyComponent() != null);
      else if (m_PanelProvider instanceof TextSupplier)
	return ((supplyText() != null) && (supplyText().length() > 0));
    }

    return false;
  }

  /**
   * Returns the object to send.
   *
   * @param cls		the classes to retrieve the item for
   * @return		the item to send
   */
  public Object getSendToItem(Class[] cls) {
    Object		result;
    JComponent		comp;
    PNGWriter		writer;

    result = null;

    if (SendToActionUtils.isAvailable(new Class[]{PlaceholderFile.class, String.class}, cls)) {
      if (m_PanelProvider instanceof ComponentSupplier) {
	comp = supplyComponent();
	if (comp != null) {
	  result = SendToActionUtils.nextTmpFile("actor-" + getName(), "png");
	  writer = new PNGWriter();
	  writer.setFile((PlaceholderFile) result);
	  writer.setComponent(comp);
	  try {
	    writer.generateOutput();
	  }
	  catch (Exception e) {
	    getSystemErr().println("Failed to write image to " + result + ":");
	    getSystemErr().printStackTrace(e);
	    result = null;
	  }
	}
      }
      if (m_PanelProvider instanceof TextSupplier) {
	result = supplyText();
      }
    }
    else if (SendToActionUtils.isAvailable(JComponent.class, cls)) {
      if (m_PanelProvider instanceof ComponentSupplier)
	result = supplyComponent();
    }

    return result;
  }
}
