/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ImageViewer.java
 * Copyright (C) 2010-2011 University of Waikato, Hamilton, New Zealand
 */

package adams.flow.sink;

import java.awt.BorderLayout;
import java.awt.Color;
import java.awt.image.BufferedImage;
import java.io.File;

import javax.swing.JPanel;

import adams.core.io.PlaceholderFile;
import adams.data.image.AbstractImage;
import adams.flow.core.Token;
import adams.gui.core.BasePanel;
import adams.gui.visualization.image.ImagePanel;

/**
 <!-- globalinfo-start -->
 * Actor for displaying an image.
 * <p/>
 <!-- globalinfo-end -->
 *
 <!-- flow-summary-start -->
 * Input&#47;output:<br/>
 * - accepts:<br/>
 * &nbsp;&nbsp;&nbsp;java.lang.String<br/>
 * &nbsp;&nbsp;&nbsp;java.io.File<br/>
 * &nbsp;&nbsp;&nbsp;java.awt.image.BufferedImage<br/>
 * <p/>
 <!-- flow-summary-end -->
 *
 <!-- options-start -->
 * Valid options are: <p/>
 *
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 *
 * <pre>-name &lt;java.lang.String&gt; (property: name)
 * &nbsp;&nbsp;&nbsp;The name of the actor.
 * &nbsp;&nbsp;&nbsp;default: ImageViewer
 * </pre>
 *
 * <pre>-annotation &lt;adams.core.base.BaseText&gt; (property: annotations)
 * &nbsp;&nbsp;&nbsp;The annotations to attach to this actor.
 * &nbsp;&nbsp;&nbsp;default:
 * </pre>
 *
 * <pre>-skip (property: skip)
 * &nbsp;&nbsp;&nbsp;If set to true, transformation is skipped and the input token is just forwarded
 * &nbsp;&nbsp;&nbsp;as it is.
 * </pre>
 *
 * <pre>-stop-flow-on-error (property: stopFlowOnError)
 * &nbsp;&nbsp;&nbsp;If set to true, the flow gets stopped in case this actor encounters an error;
 * &nbsp;&nbsp;&nbsp; useful for critical actors.
 * </pre>
 *
 * <pre>-width &lt;int&gt; (property: width)
 * &nbsp;&nbsp;&nbsp;The width of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 640
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-height &lt;int&gt; (property: height)
 * &nbsp;&nbsp;&nbsp;The height of the dialog.
 * &nbsp;&nbsp;&nbsp;default: 480
 * &nbsp;&nbsp;&nbsp;minimum: 1
 * </pre>
 *
 * <pre>-x &lt;int&gt; (property: x)
 * &nbsp;&nbsp;&nbsp;The X position of the dialog (&gt;=0: absolute, -1: left, -2: center, -3: right
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-y &lt;int&gt; (property: y)
 * &nbsp;&nbsp;&nbsp;The Y position of the dialog (&gt;=0: absolute, -1: top, -2: center, -3: bottom
 * &nbsp;&nbsp;&nbsp;).
 * &nbsp;&nbsp;&nbsp;default: -1
 * &nbsp;&nbsp;&nbsp;minimum: -3
 * </pre>
 *
 * <pre>-writer &lt;adams.gui.print.JComponentWriter&gt; (property: writer)
 * &nbsp;&nbsp;&nbsp;The writer to use for generating the graphics output.
 * &nbsp;&nbsp;&nbsp;default: adams.gui.print.NullWriter
 * </pre>
 *
 * <pre>-zoom &lt;double&gt; (property: zoom)
 * &nbsp;&nbsp;&nbsp;The zoom level in percent.
 * &nbsp;&nbsp;&nbsp;default: 100.0
 * </pre>
 *
 * <pre>-show-properties (property: showProperties)
 * &nbsp;&nbsp;&nbsp;If enabled then the image properties get displayed.
 * </pre>
 *
 <!-- options-end -->
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4753 $
 */
public class ImageViewer
  extends AbstractGraphicalDisplay
  implements DisplayPanelProvider {

  /** for serialization. */
  private static final long serialVersionUID = 1523870513962160664L;

  /** the panel with the image. */
  protected ImagePanel m_ImagePanel;

  /** the zoom level. */
  protected double m_Zoom;

  /** the background color. */
  protected Color m_BackgroundColor;

  /** whether to display the image properties. */
  protected boolean m_ShowProperties;

  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return "Actor for displaying an image.";
  }

  /**
   * Adds options to the internal list of options.
   */
  public void defineOptions() {
    super.defineOptions();

    m_OptionManager.add(
	    "zoom", "zoom",
	    100.0);

    m_OptionManager.add(
	    "background-color", "backgroundColor",
	    getDefaultBackgroundColor());

    m_OptionManager.add(
	    "show-properties", "showProperties",
	    false);
  }

  /**
   * Returns the default width for the dialog.
   *
   * @return		the default width
   */
  protected int getDefaultWidth() {
    return 640;
  }

  /**
   * Returns the default height for the dialog.
   *
   * @return		the default height
   */
  protected int getDefaultHeight() {
    return 480;
  }

  /**
   * Returns the default background color.
   *
   * @return		the default color
   */
  protected Color getDefaultBackgroundColor() {
    return new JPanel().getBackground();
  }

  /**
   * Sets the zoom level in percent.
   *
   * @param value 	the zoom
   */
  public void setZoom(double value) {
    m_Zoom = value;
    reset();
  }

  /**
   * Returns the zoom level in percent.
   *
   * @return 		the zoom
   */
  public double getZoom() {
    return m_Zoom;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String zoomTipText() {
    return "The zoom level in percent.";
  }

  /**
   * Sets the background color.
   *
   * @param value 	the color
   */
  public void setBackgroundColor(Color value) {
    m_BackgroundColor = value;
    reset();
  }

  /**
   * Returns the background color.
   *
   * @return 		the color
   */
  public Color getBackgroundColor() {
    return m_BackgroundColor;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String backgroundColorTipText() {
    return "The background color to use.";
  }

  /**
   * Sets whether to display the image properties.
   *
   * @param value 	if true then the properties are displayed
   */
  public void setShowProperties(boolean value) {
    m_ShowProperties = value;
    reset();
  }

  /**
   * Returns whether to display the image properties.
   *
   * @return 		true if the properties are displayed
   */
  public boolean getShowProperties() {
    return m_ShowProperties;
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String showPropertiesTipText() {
    return "If enabled then the image properties get displayed.";
  }

  /**
   * Clears the content of the panel.
   */
  public void clearPanel() {
    m_ImagePanel.clear();
  }

  /**
   * Creates the panel to display in the dialog.
   *
   * @return		the panel
   */
  protected BasePanel newPanel() {
    m_ImagePanel = new ImagePanel();
    return m_ImagePanel;
  }

  /**
   * Returns the class that the consumer accepts.
   *
   * @return		<!-- flow-accepts-start -->java.lang.String.class, java.io.File.class, java.awt.image.BufferedImage.class<!-- flow-accepts-end -->
   */
  public Class[] accepts() {
    return new Class[]{String.class, File.class, BufferedImage.class, AbstractImage.class};
  }

  /**
   * Displays the token (the panel and dialog have already been created at
   * this stage).
   *
   * @param token	the token to display
   */
  protected void display(Token token) {
    if (token.getPayload() instanceof String)
      m_ImagePanel.load(new PlaceholderFile((String) token.getPayload()));
    else if (token.getPayload() instanceof File)
      m_ImagePanel.load((File) token.getPayload());
    else if (token.getPayload() instanceof BufferedImage)
      m_ImagePanel.setCurrentImage((BufferedImage) token.getPayload());
    else if (token.getPayload() instanceof AbstractImage)
      m_ImagePanel.setCurrentImage(((AbstractImage) token.getPayload()).toBufferedImage());
    m_ImagePanel.setScale(m_Zoom / 100.0);
    m_ImagePanel.setShowProperties(m_ShowProperties);
    m_ImagePanel.setBackgroundColor(m_BackgroundColor);
  }

  /**
   * Removes all graphical components.
   */
  protected void cleanUpGUI() {
    super.cleanUpGUI();

    if (m_ImagePanel != null)
      m_ImagePanel.clear();
  }

  /**
   * Creates a new panel for the token.
   *
   * @param token	the token to display in a new panel
   * @return		the generated panel
   */
  public AbstractDisplayPanel createDisplayPanel(Token token) {
    AbstractDisplayPanel	result;

    result = new AbstractDisplayPanel("Image") {
      private static final long serialVersionUID = 7384093089760722339L;
      protected ImagePanel m_ImagePanel;
      protected void initGUI() {
	super.initGUI();
	setLayout(new BorderLayout());
	m_ImagePanel = new ImagePanel();
	add(m_ImagePanel, BorderLayout.CENTER);
      }
      public void display(Token token) {
	if (token.getPayload() instanceof String)
	  m_ImagePanel.load(new PlaceholderFile((String) token.getPayload()));
	else if (token.getPayload() instanceof File)
	  m_ImagePanel.load((File) token.getPayload());
	else if (token.getPayload() instanceof BufferedImage)
	  m_ImagePanel.setCurrentImage((BufferedImage) token.getPayload());
	else if (token.getPayload() instanceof AbstractImage)
	  m_ImagePanel.setCurrentImage(((AbstractImage) token.getPayload()).toBufferedImage());
	m_ImagePanel.setScale(m_Zoom);
	m_ImagePanel.setShowProperties(m_ShowProperties);
	m_ImagePanel.setBackgroundColor(m_BackgroundColor);
      }
      public void cleanUp() {
      }
    };
    result.display(token);

    return result;
  }

  /**
   * Returns whether the created display panel requires a scroll pane or not.
   *
   * @return		true if the display panel requires a scroll pane
   */
  public boolean displayPanelRequiresScrollPane() {
    return true;
  }
}
