/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * AbstractApprovalDialog.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */

package adams.gui.dialog;

import java.awt.BorderLayout;
import java.awt.Dialog;
import java.awt.FlowLayout;
import java.awt.Frame;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;

import javax.swing.JButton;
import javax.swing.JPanel;

import adams.gui.core.BaseDialog;

/**
 * Ancestor for dialogs that offer approval/disapproval buttons, e.g.,
 * OK and Cancel (default). The option selected by the user can be retrieved
 * via <code>getOption()</code>.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public abstract class AbstractApprovalDialog
  extends BaseDialog {

  /** for serialization. */
  private static final long serialVersionUID = -7382983170735594052L;

  /** the approve option. */
  public final static int APPROVE_OPTION = 0;

  /** the cancel option. */
  public final static int CANCEL_OPTION = 1;

  /** the Approve button. */
  protected JButton m_ButtonApprove;

  /** the Cancel button. */
  protected JButton m_ButtonCancel;

  /** the option selected by the user (CANCEL_OPTION, APPROVE_OPTION). */
  protected int m_Option;

  /**
   * Creates a modeless dialog without a title with the specified Dialog as
   * its owner.
   *
   * @param owner	the owning dialog
   */
  public AbstractApprovalDialog(Dialog owner) {
    super(owner);
  }

  /**
   * Creates a dialog with the specified owner Dialog and modality.
   *
   * @param owner	the owning dialog
   * @param modality	the type of modality
   */
  public AbstractApprovalDialog(Dialog owner, ModalityType modality) {
    super(owner, modality);
  }

  /**
   * Creates a modeless dialog with the specified title and with the specified
   * owner dialog.
   *
   * @param owner	the owning dialog
   * @param title	the title of the dialog
   */
  public AbstractApprovalDialog(Dialog owner, String title) {
    super(owner, title);
  }

  /**
   * Creates a dialog with the specified title, modality and the specified
   * owner Dialog.
   *
   * @param owner	the owning dialog
   * @param title	the title of the dialog
   * @param modality	the type of modality
   */
  public AbstractApprovalDialog(Dialog owner, String title, ModalityType modality) {
    super(owner, title, modality);
  }

  /**
   * Creates a modeless dialog without a title with the specified Frame as
   * its owner.
   *
   * @param owner	the owning frame
   */
  public AbstractApprovalDialog(Frame owner) {
    super(owner);
  }

  /**
   * Creates a dialog with the specified owner Frame, modality and an empty
   * title.
   *
   * @param owner	the owning frame
   * @param modal	whether the dialog is modal or not
   */
  public AbstractApprovalDialog(Frame owner, boolean modal) {
    super(owner, modal);
  }

  /**
   * Creates a modeless dialog with the specified title and with the specified
   * owner frame.
   *
   * @param owner	the owning frame
   * @param title	the title of the dialog
   */
  public AbstractApprovalDialog(Frame owner, String title) {
    super(owner, title);
  }

  /**
   * Creates a dialog with the specified owner Frame, modality and title.
   *
   * @param owner	the owning frame
   * @param title	the title of the dialog
   * @param modal	whether the dialog is modal or not
   */
  public AbstractApprovalDialog(Frame owner, String title, boolean modal) {
    super(owner, title, modal);
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Option = CANCEL_OPTION;
  }

  /**
   * Initializes the widgets.
   */
  protected void initGUI() {
    JPanel	panel;

    super.initGUI();

    getContentPane().setLayout(new BorderLayout());

    // buttons
    panel = new JPanel(new FlowLayout(FlowLayout.RIGHT));
    getContentPane().add(panel, BorderLayout.SOUTH);

    m_ButtonApprove = new JButton();
    setApproveCaption("OK");
    setApproveMnemonic(KeyEvent.VK_O);
    m_ButtonApprove.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	m_Option = APPROVE_OPTION;
	setVisible(false);
      }
    });
    panel.add(m_ButtonApprove);

    m_ButtonCancel = new JButton();
    setCancelCaption("Cancel");
    setCancelMnemonic(KeyEvent.VK_C);
    m_ButtonCancel.addActionListener(new ActionListener() {
      public void actionPerformed(ActionEvent e) {
	m_Option = CANCEL_OPTION;
	setVisible(false);
      }
    });
    panel.add(m_ButtonCancel);
  }

  /**
   * Returns whether the user approved or canceled the dialog.
   *
   * @return		the result
   * @see		#APPROVE_OPTION
   * @see		#CANCEL_OPTION
   */
  public int getOption() {
    return m_Option;
  }

  /**
   * Hook method just before the dialog is made visible.
   */
  protected void beforeShow() {
    super.beforeShow();

    m_Option = CANCEL_OPTION;
  }

  /**
   * Sets the caption/text of the approve button.
   *
   * @param value	the new text
   */
  public void setApproveCaption(String value) {
    m_ButtonApprove.setText(value);
  }

  /**
   * Returns the caption/text of the approve button.
   *
   * @return		the current text
   */
  public String getApproveCaption() {
    return m_ButtonApprove.getText();
  }

  /**
   * Sets the mnemonic of the approve button.
   *
   * @param value	the new mnemonic, e.g., KeyEvent.VK_O
   */
  public void setApproveMnemonic(int value) {
    m_ButtonApprove.setMnemonic(value);
  }

  /**
   * Returns the mnemonic of the approve button.
   *
   * @return		the current mnemonic, e.g., KeyEvent.VK_O
   */
  public int getApproveMnemonic() {
    return m_ButtonApprove.getMnemonic();
  }

  /**
   * Sets the caption/text of the Cancel button.
   *
   * @param value	the new text
   */
  public void setCancelCaption(String value) {
    m_ButtonCancel.setText(value);
  }

  /**
   * Returns the caption/text of the Cancel button.
   *
   * @return		the current text
   */
  public String getCancelCaption() {
    return m_ButtonCancel.getText();
  }

  /**
   * Sets the mnemonic of the Cancel button.
   *
   * @param value	the new mnemonic, e.g., KeyEvent.VK_C
   */
  public void setCancelMnemonic(int value) {
    m_ButtonCancel.setMnemonic(value);
  }

  /**
   * Returns the mnemonic of the Cancel button.
   *
   * @return		the current mnemonic, e.g., KeyEvent.VK_C
   */
  public int getCancelMnemonic() {
    return m_ButtonCancel.getMnemonic();
  }

  /**
   * Returns a basic approval dialog.
   *
   * @param owner	the owner of the dialog
   */
  public static AbstractApprovalDialog getDialog(Dialog owner) {
    return new AbstractApprovalDialog(owner) {
      private static final long serialVersionUID = -8835324786582910841L;
    };
  }

  /**
   * Returns a basic approval dialog.
   *
   * @param owner	the owner of the dialog
   */
  public static AbstractApprovalDialog getDialog(Frame owner) {
    return new AbstractApprovalDialog(owner) {
      private static final long serialVersionUID = -8835324786582910841L;
    };
  }
}
