/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 *    EnumEditor.java
 *    Copyright (C) 2008-2012 University of Waikato, Hamilton, New Zealand
 *
 */

package adams.gui.goe;

import java.beans.PropertyEditorSupport;
import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.Arrays;
import java.util.Vector;

import adams.core.ClassLocator;
import adams.core.EnumWithCustomDisplay;
import adams.core.Utils;
import adams.core.option.EnumOption;
import adams.gui.dialog.AbstractApprovalDialog;

/**
 * A PropertyEditor that displays Enums. Based on Weka's SelectedTagEditor.
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @author Len Trigg (trigg@cs.waikato.ac.nz)
 * @version $Revision: 4852 $
 * @see weka.gui.SelectedTagEditor
 */
public class EnumEditor
  extends PropertyEditorSupport
  implements MultiSelectionEditor {

  /**
   * Returns a description of the property value as java source.
   *
   * @return 		a value of type 'String'
   */
  public String getJavaInitializationString() {
    String	result;

    result = "Enum.valueOf("
      		+ getValue().getClass().getSimpleName() + ", "
      		+ getValue().toString() + ")";

    return result;
  }

  /**
   * Gets the current value as text.
   *
   * @return 		a value of type 'String'
   */
  public String getAsText() {
    return getValue().toString();
  }

  /**
   * Sets the current property value as text.
   *
   * @param text 	the text of the selected tag.
   */
  public void setAsText(String text) {
    if (getValue() instanceof EnumWithCustomDisplay)
      setValue(((EnumWithCustomDisplay) getValue()).parse(text));
    else
      setValue(Enum.valueOf(getValue().getClass().asSubclass(Enum.class), text));
  }

  /**
   * Gets the list of tags that can be selected from.
   *
   * @return 		an array of string tags.
   */
  public String[] getTags() {
    String[]	result;
    Method	method;
    Object[]	obj;
    int		i;

    try {
      method = getValue().getClass().getMethod("values", new Class[0]);
      obj    = (Object[]) method.invoke(null, new Object[0]);
      result = new String[obj.length];
      for (i = 0; i < obj.length; i++)
	result[i] = obj[i].toString();
    }
    catch (Exception e) {
      e.printStackTrace();
      result = new String[0];
    }

    return result;
  }

  /**
   * Determines the base class.
   * 
   * @param obj		the object to inspect
   * @return		the determined class
   */
  protected static Class determineClass(Object obj) {
    Class	result;
    
    result = obj.getClass();
    
    if (result.isArray())
      result = result.getComponentType();
    
    return result;
  }

  /**
   * Parses the given string.
   * 
   * @param cls		the enum class
   * @param s		the string to parse
   * @return		the parsed enum, null in case of an error
   */
  protected Object parse(Class cls, String s) {
    Object			result;
    EnumWithCustomDisplay	enm;
    
    result = null;
    
    if (ClassLocator.hasInterface(EnumWithCustomDisplay.class, cls)) {
      enm    = EnumOption.getEnumInstance(cls);
      result = enm.parse(s);
    }
    else {
      result = Enum.valueOf(cls, s);
    }
    
    return result;
  }

  /**
   * Returns the selected objects.
   *
   * @return		the objects
   */
  public Object[] getSelectedObjects() {
    Object[]			result;
    MultiLineValueDialog	dialog;
    Vector<String>		lines;
    Class			cls;
    int				i;

    dialog = new MultiLineValueDialog();
    dialog.setInfoText("Enter the string representations, one per line:");
    dialog.setLocationRelativeTo(null);
    dialog.setVisible(true);

    cls = determineClass(getValue());
    if (dialog.getOption() == AbstractApprovalDialog.APPROVE_OPTION) {
      lines = new Vector<String>(Arrays.asList(dialog.getContent().split("\n")));
      Utils.removeEmptyLines(lines);
      result = (Object[]) Array.newInstance(cls, lines.size());
      for (i = 0; i < lines.size(); i++)
	Array.set(result, i, parse(cls, lines.get(i)));
    }
    else {
      result = (Object[]) Array.newInstance(cls, 0);
    }

    return result;
  }
}
