/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Tree.java
 * Copyright (C) 2011 University of Waikato, Hamilton, New Zealand
 */
package adams.gui.visualization.debug.objecttree;

import java.lang.reflect.Array;
import java.util.HashSet;
import java.util.Hashtable;

import javax.swing.tree.DefaultTreeModel;

import adams.gui.core.BaseTree;
import adams.gui.visualization.debug.inspectionhandler.AbstractInspectionHandler;
import adams.gui.visualization.debug.objecttree.Node.NodeType;
import adams.gui.visualization.debug.propertyextractor.AbstractPropertyExtractor;

/**
 * Specialized tree that displays the properties of an object.
 * <p/>
 * In order to avoid loops, a HashSet is used for keeping track of the processed
 * objects. Of course, custom equals(Object)/compareTo(Object) methods will
 * interfere with this mechanism.
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4745 $
 */
public class Tree
  extends BaseTree {

  /** for serialization. */
  private static final long serialVersionUID = -127345486742553561L;

  /** the tree itself. */
  protected Tree m_Self;

  /** the current object. */
  protected transient Object m_Object;

  /** the objects that have been inspected already. */
  protected HashSet<Object> m_Inspected;

  /**
   * Initializes the tree.
   */
  public Tree() {
    super();

    m_Self = this;
    setShowsRootHandles(true);
    setRootVisible(true);
    setCellRenderer(new Renderer());
    buildTree(null);
  }

  /**
   * Builds the tree from the given object.
   *
   * @param root	the object to build the tree from, null for empty tree
   */
  protected void buildTree(Object root) {
    DefaultTreeModel		model;
    Node		rootNode;

    m_Inspected = new HashSet<Object>();

    if (root == null) {
      model = new DefaultTreeModel(null);
    }
    else {
      rootNode = buildTree(null, null, root, NodeType.NORMAL);
      model    = new DefaultTreeModel(rootNode);
    }

    m_Inspected.clear();

    setModel(model);
  }

  /**
   * Adds the array below the parent.
   *
   * @param parent	the parent to add the array to
   * @param obj		the array to add
   */
  protected void addArray(Node parent, Object obj) {
    int		len;
    int		i;
    Object	value;

    len = Array.getLength(obj);
    for (i = 0; i < len; i++) {
      value = Array.get(obj, i);
      buildTree(parent, "[" + (i+1) + "]", value, NodeType.ARRAY_ELEMENT);
    }
  }

  /**
   * Checks whether object represents a primitive class.
   * 
   * @param obj		the object to check
   */
  protected boolean isPrimitive(Object obj)  {
    if (obj instanceof Boolean)
      return true;
    else if (obj instanceof Byte)
      return true;
    else if (obj instanceof Short)
      return true;
    else if (obj instanceof Integer)
      return true;
    else if (obj instanceof Long)
      return true;
    else if (obj instanceof Float)
      return true;
    else if (obj instanceof Double)
      return true;
    else if (obj instanceof Character)
      return true;
    else if (obj instanceof String)
      return true;
    else
      return false;
  }
  
  /**
   * Builds the tree recursively.
   *
   * @param parent	the parent to add the object to (null == root)
   * @param property	the name of the property the object belongs to (null == root)
   * @param obj		the object to add
   * @param type	the type of node
   * @return		the generated node
   */
  protected Node buildTree(Node parent, String property, Object obj, NodeType type) {
    Node			result;
    AbstractPropertyExtractor	extractor;
    AbstractInspectionHandler	inspection;
    Hashtable<String,Object>	additional;
    Object			current;
    String			label;
    int				i;

    result = new Node(property, obj, type);
    if (parent != null)
      parent.add(result);

    m_Inspected.add(obj);

    // Object's hashcode
    if (!isPrimitive(obj))
      result.add(new Node("hashCode", obj.hashCode(), NodeType.HASHCODE));

    // array?
    if (obj.getClass().isArray())
      addArray(result, obj);

    // child properties
    try {
      extractor = AbstractPropertyExtractor.getExtractor(obj);
      extractor.setCurrent(obj);
      for (i = 0; i < extractor.size(); i++) {
	current = extractor.getValue(i);
	if (current != null) {
	  label = extractor.getLabel(i);
	  buildTree(result, label, current, NodeType.NORMAL);
	}
      }
    }
    catch (Exception e) {
      System.err.println("Failed to obtain property descriptors for: " + obj);
      e.printStackTrace();
    }

    // additional values obtained through inspection handlers
    inspection = AbstractInspectionHandler.getHandler(obj);
    additional = inspection.inspect(obj);
    for (String key: additional.keySet())
      buildTree(result, key, additional.get(key), NodeType.NORMAL);

    return result;
  }

  /**
   * Sets the object to display.
   *
   * @param value	the object
   */
  public void setObject(Object value) {
    m_Object = value;
    buildTree(m_Object);
  }

  /**
   * Returns the object currently displayed.
   *
   * @return		the object
   */
  public Object getObject() {
    return m_Object;
  }
}