/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * ReportMathExpression.java
 * Copyright (C) 2012 University of Waikato, Hamilton, New Zealand
 */

package adams.parser;

import adams.data.report.Report;

/**
 <!-- globalinfo-start -->
 * Evaluates mathematical expressions on report values.<br/>
 * <br/>
 * The following grammar is used:<br/>
 * <br/>
 * expr_list ::= expr_list expr_part | expr_part ;<br/>
 * expr_part ::= expr ;<br/>
 * <br/>
 * numexpr   ::=   ( numexpr )<br/>
 *               |  NUMBER<br/>
 *               | -numexpr<br/>
 *               | constexpr<br/>
 *               | opexpr<br/>
 *               | funcexpr<br/>
 *               ;<br/>
 * <br/>
 *  strexpr  ::=   substr ( strexpr , start )<br/>
 *               | substr ( strexpr , start , end )<br/>
 *               | lowercase ( strexpr )<br/>
 *               | uppercase ( strexpr )<br/>
 *               | string<br/>
 *               | str ( VARIABLE (must represent a string) )<br/>
 *               ;<br/>
 * <br/>
 * constexpr ::=   PI<br/>
 *               | E<br/>
 *               ;<br/>
 * <br/>
 * opexpr    ::=   numexpr + numexpr<br/>
 *               | numexpr - numexpr<br/>
 *               | numexpr * numexpr<br/>
 *               | numexpr &#47; numexpr<br/>
 *               | numexpr ^ numexpr<br/>
 *               ;<br/>
 * <br/>
 * funcexpr ::=    abs ( numexpr )<br/>
 *               | sqrt ( numexpr )<br/>
 *               | log ( numexpr )<br/>
 *               | exp ( numexpr )<br/>
 *               | sin ( numexpr )<br/>
 *               | cos ( numexpr )<br/>
 *               | tan ( numexpr )<br/>
 *               | rint ( numexpr )<br/>
 *               | floor ( numexpr )<br/>
 *               | pow ( numexpr , numexpr )<br/>
 *               | ceil ( numexpr )<br/>
 *               | ifelse ( boolexpr , numexpr (if true) , numexpr (if false) )<br/>
 *               | ifmissing ( variable , numexpr (default value if variable is missing) )<br/>
 *               | num ( VARIABLE (must represent a number) )<br/>
 *               | length ( str )<br/>
 *               ;<br/>
 * <br/>
 * boolexpr ::=    ( boolexpr )<br/>
 *               | true<br/>
 *               | false<br/>
 *               | bool ( VARIABLE (must represent a boolean string: true|false) )<br/>
 *               | numexpr &amp;lt; numexpr<br/>
 *               | numexpr &amp;lt;= numexpr<br/>
 *               | numexpr &amp;gt; numexpr<br/>
 *               | numexpr &amp;gt;= numexpr<br/>
 *               | numexpr = numexpr<br/>
 *               | numexpr != numexpr<br/>
 *               | strexpr &lt; strexpr<br/>
 *               | strexpr &lt;= strexpr<br/>
 *               | strexpr &gt; strexpr<br/>
 *               | strexpr &gt;= strexpr<br/>
 *               | strexpr = strexpr<br/>
 *               | strexpr != strexpr<br/>
 *               | matches ( strexpr , regexp )<br/>
 *               | ! boolexpr<br/>
 *               | boolexpr &amp; boolexpr<br/>
 *               | boolexpr | boolexpr<br/>
 *               ;<br/>
 * <br/>
 * Notes:<br/>
 * - Variables are either all upper case letters (e.g., "ABC") or any character   apart from "]" enclosed by "[" and "]" (e.g., "[Hello World]").<br/>
 * - 'start' and 'end' for function 'substr' are indices that start at 1.<br/>
 * - Index 'end' for function 'substr' is excluded (like Java's 'String.substring(int,int)' method)<br/>
 * - Line comments start with '#'.<br/>
 * <p/>
 <!-- globalinfo-end -->
 *
 * Variables are either all upper case letters (e.g., "ABC") or any character 
 * apart from "]" enclosed by "[" and "]" (e.g., "[Hello World]").
 * <p/>
 * Code example 1:
 * <pre>
 * String expr = "pow(BASE,EXPONENT)*MULT";
 * HashMap symbols = new HashMap();
 * symbols.put("BASE", new Double(2));
 * symbols.put("EXPONENT", new Double(9));
 * symbols.put("MULT", new Double(0.1));
 * double result = ReportMathExpression.evaluate(expr, symbols);
 * System.out.println(expr + " and " + symbols + " = " + result);
 * </pre>
 *
 * Code Example 2 (uses the "ifelse" construct):
 * <pre>
 * String expr = "ifelse(I<0,pow(BASE,I*0.5),pow(BASE,I))";
 * ReportMathExpression.TreeNode tree = ReportMathExpression.parse(expr);
 * HashMap symbols = new HashMap();
 * symbols.put("BASE", new Double(2));
 * for (int i = -10; i <= 10; i++) {
 *   symbols.put("I", new Double(i));
 *   double result = ReportMathExpression.evaluate(expr, symbols);
 *   System.out.println(expr + " and " + symbols + " = " + result);
 * }
 * </pre>
 *
 <!-- options-start -->
 * Valid options are: <p/>
 * 
 * <pre>-D &lt;int&gt; (property: debugLevel)
 * &nbsp;&nbsp;&nbsp;The greater the number the more additional info the scheme may output to 
 * &nbsp;&nbsp;&nbsp;the console (0 = off).
 * &nbsp;&nbsp;&nbsp;default: 0
 * &nbsp;&nbsp;&nbsp;minimum: 0
 * </pre>
 * 
 * <pre>-env &lt;java.lang.String&gt; (property: environment)
 * &nbsp;&nbsp;&nbsp;The class to use for determining the environment.
 * &nbsp;&nbsp;&nbsp;default: adams.env.Environment
 * </pre>
 * 
 * <pre>-expression &lt;java.lang.String&gt; (property: expression)
 * &nbsp;&nbsp;&nbsp;The mathematical expression, including report fields, to evaluate (must 
 * &nbsp;&nbsp;&nbsp;evaluate to a double).
 * &nbsp;&nbsp;&nbsp;default: 42
 * </pre>
 * 
 <!-- options-end -->
 *
 * @author FracPete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4813 $
 */
public class ReportMathExpression
  extends AbstractExpressionEvaluator<Double> {

  /** for serialization. */
  private static final long serialVersionUID = 8014316012335802585L;

  /** the report to use as basis. */
  protected Report m_Report;
  
  /**
   * Returns a string describing the object.
   *
   * @return 			a description suitable for displaying in the gui
   */
  public String globalInfo() {
    return
        "Evaluates mathematical expressions on report values.\n\n"
    + "The following grammar is used:\n\n"
    + getGrammar();
  }

  /**
   * Initializes the members.
   */
  protected void initialize() {
    super.initialize();

    m_Report = new Report();
  }

  /**
   * Returns a string representation of the grammar.
   *
   * @return		the grammar, null if not available
   */
  public String getGrammar() {
    return
	new MathematicalExpression().getGrammar();
  }

  /**
   * Returns the default expression to use.
   *
   * @return		the default expression
   */
  protected String getDefaultExpression() {
    return "42";
  }

  /**
   * Returns the tip text for this property.
   *
   * @return 		tip text for this property suitable for
   * 			displaying in the GUI or for listing the options.
   */
  public String expressionTipText() {
    return "The mathematical expression, including report fields, to evaluate (must evaluate to a double).";
  }

  /**
   * Sets the report to use for evaluation.
   *
   * @param value	the report
   */
  public void setReport(Report value) {
    m_Report = value;
  }

  /**
   * Returns the report to use for evaluation.
   *
   * @return		the report
   */
  public Report getReport() {
    return m_Report;
  }

  /**
   * Performs the evaluation.
   *
   * @return		the evaluation, or null in case of error
   * @throws Exception	if evaluation fails
   */
  public Double evaluate() throws Exception {
    return evaluate(m_Expression, m_Report);
  }

  /**
   * Parses and evaluates the given expression.
   * Returns the result of the mathematical expression, based on the given
   * values of the symbols.
   *
   * @param expr	the expression to evaluate
   * @param report	the report
   * @return		the evaluated result
   * @throws Exception	if something goes wrong
   */
  public static double evaluate(String expr, Report report) throws Exception {
    return MathematicalExpression.evaluate(expr, report);
  }

  /**
   * Runs the evaluator from command-line.
   *
   * @param args	the command-line options, use "-help" to list them
   */
  public static void main(String[] args) {
    runEvaluator(ReportMathExpression.class, args);
  }
}
