/*
 *   This program is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * BaseDateTimeExpressionTest.java
 * Copyright (C) 2010 University of Waikato, Hamilton, New Zealand
 */
package adams.parser;

import java.text.SimpleDateFormat;
import java.util.Date;

import junit.framework.Test;
import junit.framework.TestSuite;
import adams.core.base.BaseDateTime;
import adams.env.Environment;

/**
 * Tests the adams.parser.BaseDateTimeExpression class. Run from commandline with: <p/>
 * java adams.parser.BaseDateTimeExpressionTest
 *
 * @author  fracpete (fracpete at waikato dot ac dot nz)
 * @version $Revision: 4584 $
 */
public class BaseDateTimeExpressionTest
  extends AbstractExpressionEvaluatorTestCase<Date, BaseDateTimeExpression> {

  /**
   * Initializes the test.
   *
   * @param name	the name of the test
   */
  public BaseDateTimeExpressionTest(String name) {
    super(name);
  }

  /**
   * Returns the expressions used in the regression test.
   *
   * @return		the data
   */
  protected String[][] getRegressionExpressions() {
    return new String[][]{
	{
	  "-INF",
	  "+INF",
	  "1999-12-31 01:02:03 +1 SECOND",
	  "1999-12-31 01:02:03 +1 MINUTE",
	  "1999-12-31 01:02:03 +1 HOUR",
	  "1999-12-31 01:02:03 +1 DAY",
	  "1999-12-31 01:02:03 +1 WEEK",
	  "1999-12-31 01:02:03 +1 MONTH",
	  "1999-12-31 01:02:03 +1 YEAR",
	  "1999-12-31 01:02:03 -1 SECOND",
	  "1999-12-31 01:02:03 -1 MINUTE",
	  "1999-12-31 01:02:03 -1 HOUR",
	  "1999-12-31 01:02:03 -1 DAY",
	  "1999-12-31 01:02:03 -1 WEEK",
	  "1999-12-31 01:02:03 -1 MONTH",
	  "1999-12-31 01:02:03 -1 YEAR",
	  "1999-12-31 01:02:03",
	  "START +1 HOUR +1 DAY",
	  "END -1 HOUR -1 DAY"
	}
    };
  }

  /**
   * Returns the setups to use in the regression test.
   *
   * @return		the setups
   */
  protected BaseDateTimeExpression[] getRegressionSetups() {
    return new BaseDateTimeExpression[]{new BaseDateTimeExpression()};
  }

  /**
   * Tests setitng the START variable programmatically.
   */
  public static void testStart() {
    String expr = "START +2 DAY +1 MONTH +1 HOUR";
    SimpleDateFormat format = new SimpleDateFormat(BaseDateTime.FORMAT);
    String startStr = "1901-01-01 12:34:00";
    Date start = null;
    Date parsed = null;
    try {
      start = format.parse(startStr);
    }
    catch (Exception e) {
      fail("Failed to generate Date object from '" + startStr + "': " + e);
    }
    try {
      parsed = BaseDateTimeExpression.evaluate(expr, start, null);
    }
    catch (Exception e) {
      fail("Failed to parse expression '" + expr + "': " + e);
    }
    assertNotNull("Generated null date", parsed);
    assertEquals("Generated different date", "1901-02-03 13:34:00", format.format(parsed));
  }

  /**
   * Tests setitng the END variable programmatically.
   */
  public static void testEnd() {
    String expr = "END -2 DAY -1 MONTH -1 HOUR";
    SimpleDateFormat format = new SimpleDateFormat(BaseDateTime.FORMAT);
    String endStr = "1901-10-27 12:34:00";
    Date end = null;
    Date parsed = null;
    try {
      end = format.parse(endStr);
    }
    catch (Exception e) {
      fail("Failed to generate Date object from '" + endStr + "': " + e);
    }
    try {
      parsed = BaseDateTimeExpression.evaluate(expr, null, end);
    }
    catch (Exception e) {
      fail("Failed to parse expression '" + expr + "': " + e);
    }
    assertNotNull("Generated null date", parsed);
    assertEquals("Generated different date", "1901-09-25 11:34:00", format.format(parsed));
  }

  /**
   * Returns a test suite.
   *
   * @return		the test suite
   */
  public static Test suite() {
    return new TestSuite(BaseDateTimeExpressionTest.class);
  }

  /**
   * Runs the test from commandline.
   *
   * @param args	ignored
   */
  public static void main(String[] args) {
    Environment.setEnvironmentClass(Environment.class);
    runTest(suite());
  }
}
